/* camdrv.c */
/* CAMAC device driver for CC/7000-ISA on Linux 2.2.x */
/* Created by Enomoto Sanshiro on 11 April 1999. */
/* Last updated by Enomoto Sanshiro on 11 May 1999. */


#define __KERNEL__
#define MODULE


#include <linux/autoconf.h>
#if defined(USE_MODVERSIONS) && USE_MODVERSIONS
#  ifdef CONFIG_MODVERSIONS
#    define MODVERSIONS
#    include <linux/modversions.h>
#  endif
#endif


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include "camdrv.h"


#ifndef CAMDRV_MAJOR
#  define CAMDRV_MAJOR 0
#endif
static int camdrv_major = CAMDRV_MAJOR;
MODULE_PARM(camdrv_major, "i");

static unsigned long ioport = 0;
static int irq = 0;
MODULE_PARM(ioport, "l");
MODULE_PARM(irq, "i");

static const char* camdrv_name = "camdrv";
struct wait_queue* camdrv_wait_queue = NULL;
static int interrupt_count;

static int camdrv_open(struct inode* inode, struct file* filep);
static int camdrv_release(struct inode* inode, struct file* filep);
static int camdrv_ioctl(struct inode *inode, struct file *filep, unsigned int cmd, unsigned long arg);
static void camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs);

static int initialize(void);
static int clear(void);
static int inhibit(void);
static int release_inhibit(void);
static int camac_action(unsigned naf, unsigned* data);
static int read_lam(unsigned* data);
static int enable_interrupt(void);
static int disable_interrupt(void);
static int wait_lam(unsigned time_out, unsigned* data);


static struct file_operations camdrv_fops = {
    NULL,              /* camdrv_llseek */
    NULL,              /* camdrv_read */
    NULL,              /* camdrv_write */
    NULL,              /* camdrv_readdir */
    NULL,              /* camdrv_poll */
    camdrv_ioctl,      /* camdrv_ioctl */
    NULL,              /* camdrv_mmap */
    camdrv_open,       /* camdrv_open */
    NULL,              /* camdrv_flush */
    camdrv_release,    /* camdrv_close */
};

int init_module(void)
{
    int result;

    if (ioport == 0) {
        printk(KERN_WARNING "%s: I/O port address (ioport) was not specified.\n", camdrv_name);
        return -EINVAL;
    }
    if (irq == 0) {
        printk(KERN_WARNING "%s: IRQ number (irq) was not specified.\n", camdrv_name);
        return -EINVAL;
    }

    result = register_chrdev(camdrv_major, camdrv_name, &camdrv_fops);
    if (result < 0) {
        printk(KERN_WARNING "%s: can't get major %d\n", camdrv_name, camdrv_major);
        return result;
    }
    if (camdrv_major == 0) {
        camdrv_major = result;
    }

    printk(KERN_INFO "%s: at 0x%04lx on irq %d (major = %d).\n", camdrv_name, ioport, irq, camdrv_major);

    return 0;
}

void cleanup_module(void)
{
    unregister_chrdev(camdrv_major, camdrv_name);
    printk(KERN_INFO "%s: removed.\n", camdrv_name);
}

static int camdrv_open(struct inode* inode, struct file* filep)
{
    int result;

    if (MOD_IN_USE) {
        return -EBUSY;
    }

    result = request_irq(irq, camdrv_interrupt, SA_INTERRUPT, camdrv_name, NULL);
    if (result < 0) {
        return result;
    }

    MOD_INC_USE_COUNT;

    return 0;
}

static int camdrv_release(struct inode* inode, struct file* filep)
{
    free_irq(irq, NULL);

    MOD_DEC_USE_COUNT;

    return 0;
}

static int camdrv_ioctl(struct inode* inode, struct file *filep, unsigned int cmd, unsigned long arg)
{
    unsigned naf = 0, data = 0, timeout = 0;
    unsigned *user_naf_ptr, *user_data_ptr;
    int result = -EINVAL;

    user_naf_ptr = (unsigned *) arg;
    user_data_ptr = (unsigned *) arg + 1;

    if (_IOC_TYPE(cmd) != CAMDRV_IOC_MAGIC) {
        return -EINVAL;
    }

    if (_IOC_DIR(cmd) & (_IOC_READ | _IOC_WRITE)) {
	get_user_ret(naf, user_naf_ptr, -EFAULT);
	get_user_ret(data, user_data_ptr, -EFAULT);
	timeout = naf;
    }

    switch (cmd) {
      case CAMDRV_IOC_INITIALIZE:
        result = initialize();
        break;
      case CAMDRV_IOC_CLEAR:
        result = clear();
        break;
      case CAMDRV_IOC_INHIBIT:
        result = inhibit();
        break;
      case CAMDRV_IOC_RELEASE_INHIBIT:
        result = release_inhibit();
        break;
      case CAMDRV_IOC_ENABLE_INTERRUPT:
        result = enable_interrupt();
        break;
      case CAMDRV_IOC_DISABLE_INTERRUPT:
        result = disable_interrupt();
        break;
      case CAMDRV_IOC_CAMAC_ACTION:
        result = camac_action(naf, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_READ_LAM:
        result = read_lam(&data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_WAIT_LAM:
        result = wait_lam(timeout, &data);
        put_user(data, user_data_ptr);
        break;
      default:
	result = -EINVAL;
    }

    return result;
}

static void camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs)
{
    disable_interrupt();
    interrupt_count++;
    wake_up_interruptible(&camdrv_wait_queue);
}



/* CAMAC service functions */

enum cc7000_regs {
    regCONTROL = 0x06,
    regSTATUS = 0x06,
    regCRATE_ADDRESS = 0x01,
    regCOMMAND_NA = 0x04,
    regLAM_LOW = 0x04,
    regLAM_HIGH = 0x06,
    regDATA_LOW = 0x00,
    regDATA_HIGH = 0x02,
    regGO = 0x07
};

enum cc7000_ctrlbits {
    ctrlCLEAR = 0x0100,
    ctrlINITIALIZE = 0x0200,
    ctrlINHIBIT = 0x0400,
    ctrlENABLEINTERRUPT = 0x0800,
    ctrlLAMINTERNAL = 0x8000
};

enum cc7000_statbits {
    statQ = 0x0100,
    statX = 0x0200,
    statINHIBIT = 0x0400,
    statENABLEINTERRUPT = 0x0800,
    statDONE = 0x1000,
    statONLINE = 0x2000,
    statLAMSUM = 0x4000,
    statLAMINTERNAL = 0x8000
};

#define CONTROL_KEPT_BIT_MASK (0x0400 | 0x0800 | 0x8000)
#define DATA_HIGH_MASK 0x00ff
#define LAM_HIGH_MASK 0x00ff

static int initialize(void)
{
    outb(0, ioport + regCRATE_ADDRESS);
    outw(ctrlINITIALIZE, ioport + regCONTROL);

    return 0;
}

static int clear(void)
{
    unsigned short reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlCLEAR;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int inhibit(void)
{
    unsigned short reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlINHIBIT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int release_inhibit(void)
{
    unsigned short reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg &= ~ctrlINHIBIT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int enable_interrupt(void)
{
    unsigned short reg;

    interrupt_count = 0;
    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlENABLEINTERRUPT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int disable_interrupt(void)
{
    unsigned short reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg &= ~ctrlENABLEINTERRUPT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int camac_action(unsigned naf, unsigned* data)
{
    unsigned long start_jiffies;
    int status;
    unsigned short control_reg;

    unsigned short n = (naf >> 9) & 0x1f;
    unsigned short a = (naf >> 5) & 0x0f;
    unsigned short f = (naf >> 0) & 0x1f;
    unsigned short na = (a << 8) | n;

    unsigned short data_l = (*data >> 0) & 0xffff;
    unsigned short data_h = (*data >> 16) & 0x00ff;

    outw(na, ioport + regCOMMAND_NA);
    control_reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    control_reg |= f;
    outw(control_reg, ioport + regCONTROL);

    outw(data_l, ioport + regDATA_LOW);
    outw(data_h, ioport + regDATA_HIGH);

    outb(0, ioport + regGO);

    start_jiffies = jiffies;
    while (! ((status = inw(ioport + regSTATUS)) & statDONE)) {
        schedule();
	if (jiffies - start_jiffies > HZ) {
	    *data = 0;
	    return -ETIMEDOUT;
	}
    }

    data_l = inw(ioport + regDATA_LOW);
    data_h = inw(ioport + regDATA_HIGH) & DATA_HIGH_MASK;

    *data = (data_h << 16) | data_l;

    return (status >> 8);
}

static int read_lam(unsigned* data)
{
    unsigned short lam_l, lam_h;

    lam_l = inw(ioport + regLAM_LOW);
    lam_h = inw(ioport + regLAM_HIGH) & LAM_HIGH_MASK;

    *data = (lam_h << 16) | lam_l;

    return *data;
}

static int wait_lam(unsigned timeout, unsigned* data)
{
    unsigned long flags;
    long remaining_time;
    timeout *= HZ;

    save_flags(flags);
    cli();
    if (interrupt_count == 0) {
        remaining_time = interruptible_sleep_on_timeout(&camdrv_wait_queue, timeout);
	restore_flags(flags);

	if (remaining_time == 0) {
            return -ETIMEDOUT;
	}
    }
    else {
        disable_interrupt();
        restore_flags(flags);
    }

    interrupt_count = 0;

    return read_lam(data);
}
