/* camdrv.c */
/* CAMAC device driver for CC/7700-ISA on Linux 2.4.x */
/* Created by Enomoto Sanshiro on 11 April 1999. */
/* Last updated by Enomoto Sanshiro on 18 September 2002. */


#define __KERNEL__
#define MODULE


#include <linux/autoconf.h>
#if defined(USE_MODVERSIONS) && USE_MODVERSIONS
#  ifdef CONFIG_MODVERSIONS
#    define MODVERSIONS
#    include <linux/modversions.h>
#  endif
#endif


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include "camdrv.h"

MODULE_LICENSE("GPL");


#ifndef CAMDRV_MAJOR
#  define CAMDRV_MAJOR 0
#endif
static int camdrv_major = CAMDRV_MAJOR;
MODULE_PARM(camdrv_major, "i");

static const char* camdrv_name = "camdrv";
static unsigned long ioport = 0;
static int irq = 0;
MODULE_PARM(ioport, "l");
MODULE_PARM(irq, "i");

static unsigned crate_address = 0;
static int interrupt_count;
wait_queue_head_t camdrv_wait_queue;

static int camdrv_open(struct inode* inode, struct file* filep);
static int camdrv_release(struct inode* inode, struct file* filep);
static int camdrv_ioctl(struct inode *inode, struct file *filep, unsigned int cmd, unsigned long arg);
static void camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs);

static int set_crate(void);
static int initialize(void);
static int clear(void);
static int inhibit(void);
static int release_inhibit(void);
static int camac_action(unsigned naf, unsigned* data);
static int read_lam(unsigned* data);
static int enable_interrupt(void);
static int disable_interrupt(void);
static int wait_lam(unsigned time_out, unsigned* data);


static struct file_operations camdrv_fops = {
    ioctl: camdrv_ioctl,
    open: camdrv_open,
    release: camdrv_release,
};


int init_module(void)
{
    int result;

    if (ioport == 0) {
        printk(KERN_WARNING "%s: I/O port address (ioport) was not specified.\n", camdrv_name);
        return -EINVAL;
    }
    if (irq == 0) {
        printk(KERN_WARNING "%s: IRQ number (irq) was not specified.\n", camdrv_name);
        return -EINVAL;
    }

    result = register_chrdev(camdrv_major, camdrv_name, &camdrv_fops);
    if (result < 0) {
        printk(KERN_WARNING "%s: can't get major %d\n", camdrv_name, camdrv_major);
        return result;
    }
    if (camdrv_major == 0) {
        camdrv_major = result;
    }
    init_waitqueue_head(&camdrv_wait_queue);

    printk(KERN_INFO "%s: at 0x%04lx on irq %d (major = %d).\n", camdrv_name, ioport, irq, camdrv_major);

    return 0;
}

void cleanup_module(void)
{
    unregister_chrdev(camdrv_major, camdrv_name);
    printk(KERN_INFO "%s: removed.\n", camdrv_name);
}

static int camdrv_open(struct inode* inode, struct file* filep)
{
    int result;

    if (MOD_IN_USE) {
        return -EBUSY;
    }

    result = request_irq(irq, camdrv_interrupt, SA_INTERRUPT, camdrv_name, NULL);
    if (result < 0) {
        return result;
    }

    MOD_INC_USE_COUNT;

    return 0;
}

static int camdrv_release(struct inode* inode, struct file* filep)
{
    free_irq(irq, NULL);

    MOD_DEC_USE_COUNT;

    return 0;
}

static int camdrv_ioctl(struct inode* inode, struct file *filep, unsigned int cmd, unsigned long arg)
{
    unsigned parameter = 0, data = 0;
    unsigned *user_parameter_ptr, *user_data_ptr;
    int result = -EINVAL;

    user_parameter_ptr = (unsigned *) arg;
    user_data_ptr = (unsigned *) arg + 1;

    if (_IOC_TYPE(cmd) != CAMDRV_IOC_MAGIC) {
        return -EINVAL;
    }

    if (_IOC_DIR(cmd) & (_IOC_READ | _IOC_WRITE)) {
	if (get_user(parameter, user_parameter_ptr) < 0) {
	    return -EFAULT;
	}
	if (get_user(data, user_data_ptr) < 0) {
	    return -EFAULT;
	}
    }

    switch (cmd) {
      case CAMDRV_IOC_INITIALIZE:
        result = initialize();
        break;
      case CAMDRV_IOC_CLEAR:
        result = clear();
        break;
      case CAMDRV_IOC_INHIBIT:
        result = inhibit();
        break;
      case CAMDRV_IOC_RELEASE_INHIBIT:
        result = release_inhibit();
        break;
      case CAMDRV_IOC_ENABLE_INTERRUPT:
        result = enable_interrupt();
        break;
      case CAMDRV_IOC_DISABLE_INTERRUPT:
        result = disable_interrupt();
        break;
      case CAMDRV_IOC_CAMAC_ACTION:
        result = camac_action(parameter, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_READ_LAM:
        result = read_lam(&data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_WAIT_LAM:
        result = wait_lam(parameter, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_SET_CRATE:
        crate_address = parameter;
        result = set_crate();
        break;
      default:
	result = -EINVAL;
    }

    return result;
}

static void camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs)
{
    disable_interrupt();
    interrupt_count++;
    wake_up_interruptible(&camdrv_wait_queue);
}



/* CAMAC service functions */

enum cc7700_regs {
    regCONTROL = 0x00,
    regSTATUS = 0x00,
    regCRATE_ADDRESS = 0x02,
    regCOMMAND = 0x04,
    regDATA_LOW = 0x06,
    regDATA_HIGH = 0x08,
    regLAM_LOW = 0x02,
    regLAM_HIGH = 0x04,
    regGO = 0x0A
};

enum cc7700_ctrlbits {
    ctrlCLEAR = 0x01,
    ctrlINITIALIZE = 0x02,
    ctrlINHIBIT = 0x04,
    ctrlENABLEINTERRUPT = 0x08,
    ctrlRESET = 0x40,
    ctrlLAMINTERNAL = 0x80
};

enum cc7700_statbits {
    statNOQ = 0x01,
    statNOX = 0x02,
    statINHIBIT = 0x04,
    statENABLEINTERRUPT = 0x08,
    statDONE = 0x10,
    statONLINE = 0x20,
    statLAMSUM = 0x40,
    statLAMINTERNAL = 0x80
};


#define CONTROL_KEPT_BIT_MASK (0x04 | 0x08 | 0x80)
#define DATA_HIGH_MASK 0x00ff
#define LAM_HIGH_MASK 0x00ff
#define STATUS_BIT_MASK 0x00ff

static int set_crate(void)
{
    outw(crate_address, ioport + regCRATE_ADDRESS);
    return 0;
}

static int initialize(void)
{
    outw(crate_address, ioport + regCRATE_ADDRESS);
    outw(ctrlINITIALIZE, ioport + regCONTROL);

    return 0;
}

static int clear(void)
{
    unsigned reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlCLEAR;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int inhibit(void)
{
    unsigned reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlINHIBIT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int release_inhibit(void)
{
    unsigned reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg &= ~ctrlINHIBIT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int enable_interrupt(void)
{
    unsigned reg;

    interrupt_count = 0;
    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlENABLEINTERRUPT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int disable_interrupt(void)
{
    unsigned reg;

    reg = inw(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg &= ~ctrlENABLEINTERRUPT;
    
    outw(reg, ioport + regCONTROL);

    return 0;
}

static int camac_action(unsigned naf, unsigned* data)
{
    unsigned long start_jiffies;
    int status;

    unsigned short data_l = (*data >> 0) & 0xffff;
    unsigned short data_h = (*data >> 16) & 0x00ff;
    
    /* Hardware Problem of CC/7700 ??? */
    /* subtract 1 from station number */
    naf -= (1 << 9);
    
    outw(naf, ioport + regCOMMAND);
    outw(data_l, ioport + regDATA_LOW);
    outw(data_h, ioport + regDATA_HIGH);

    outw(0, ioport + regGO);

    start_jiffies = jiffies;
    while (! ((status = inw(ioport + regSTATUS)) & statDONE)) {
        schedule();
	if (jiffies - start_jiffies > HZ) {
	    *data = 0;
	    return -ETIMEDOUT;
	}
    }

    data_l = inw(ioport + regDATA_LOW);
    data_h = inw(ioport + regDATA_HIGH) & DATA_HIGH_MASK;

    *data = (data_h << 16) | data_l;

    return status & STATUS_BIT_MASK;
}

static int read_lam(unsigned* data)
{
    unsigned short lam_l, lam_h;

    lam_l = inw(ioport + regLAM_LOW);
    lam_h = inw(ioport + regLAM_HIGH) & LAM_HIGH_MASK;

    *data = (lam_h << 16) | lam_l;

    return *data;
}

static int wait_lam(unsigned timeout, unsigned* data)
{
    unsigned long flags;
    long remaining_time;
    timeout *= HZ;

    save_flags(flags);
    cli();
    if (interrupt_count == 0) {
        remaining_time = interruptible_sleep_on_timeout(&camdrv_wait_queue, timeout);
	restore_flags(flags);

	if (remaining_time == 0) {
            return -ETIMEDOUT;
	}
    }
    else {
        disable_interrupt();
        restore_flags(flags);
    }

    interrupt_count = 0;

    return read_lam(data);
}
