/* camdrv.c */
/* CAMAC device driver for Hoshin CCP-PCI on Linux 2.4.x */
/* Created by Enomoto Sanshiro on 11 April 1999. */
/* Last updated by Enomoto Sanshiro on 18 September 2002. */


#define __KERNEL__
#define MODULE


#include <linux/autoconf.h>
#if defined(USE_MODVERSIONS) && USE_MODVERSIONS
#  ifdef CONFIG_MODVERSIONS
#    define MODVERSIONS
#    include <linux/modversions.h>
#  endif
#endif


#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include "camdrv.h"

MODULE_LICENSE("GPL");


#ifndef CAMDRV_MAJOR
#  define CAMDRV_MAJOR 0
#endif
static int camdrv_major = CAMDRV_MAJOR;
MODULE_PARM(camdrv_major, "i");

static unsigned long ioport = 0;
MODULE_PARM(ioport, "l");

static const char* camdrv_name = "camdrv";
static int crate_number = 0;

static int camdrv_open(struct inode* inode, struct file* filep);
static int camdrv_release(struct inode* inode, struct file* filep);
static int camdrv_ioctl(struct inode *inode, struct file *filep, unsigned int cmd, unsigned long arg);

static int initialize(void);
static int clear(void);
static int inhibit(void);
static int release_inhibit(void);
static int camac_action(unsigned naf, unsigned* data);
static int read_lam(unsigned* data);
static int enable_interrupt(void);
static int disable_interrupt(void);
static int wait_lam(unsigned time_out, unsigned* data);


static struct file_operations camdrv_fops = {
    ioctl: camdrv_ioctl,
    open: camdrv_open,
    release: camdrv_release,
};


int init_module(void)
{
    int result;

    if (ioport == 0) {
        printk(KERN_WARNING "%s: I/O port address (ioport) was not specified.\n", camdrv_name);
        return -EINVAL;
    }

    result = register_chrdev(camdrv_major, camdrv_name, &camdrv_fops);
    if (result < 0) {
        printk(KERN_WARNING "%s: can't get major %d\n", camdrv_name, camdrv_major);
        return result;
    }
    if (camdrv_major == 0) {
        camdrv_major = result;
    }

    printk(KERN_INFO "%s: at 0x%04lx (major = %d).\n", camdrv_name, ioport, camdrv_major);

    return 0;
}

void cleanup_module(void)
{
    unregister_chrdev(camdrv_major, camdrv_name);
    printk(KERN_INFO "%s: removed.\n", camdrv_name);
}

static int camdrv_open(struct inode* inode, struct file* filep)
{
    if (MOD_IN_USE) {
        return -EBUSY;
    }
    MOD_INC_USE_COUNT;

    return 0;
}

static int camdrv_release(struct inode* inode, struct file* filep)
{
    MOD_DEC_USE_COUNT;

    return 0;
}

static int camdrv_ioctl(struct inode* inode, struct file *filep, unsigned int cmd, unsigned long arg)
{
    unsigned parameter = 0, data = 0;
    unsigned *user_parameter_ptr, *user_data_ptr;
    int result = -EINVAL;

    user_parameter_ptr = (unsigned *) arg;
    user_data_ptr = (unsigned *) arg + 1;

    if (_IOC_TYPE(cmd) != CAMDRV_IOC_MAGIC) {
        return -EINVAL;
    }

    if (_IOC_DIR(cmd) & (_IOC_READ | _IOC_WRITE)) {
	if (get_user(parameter, user_parameter_ptr) < 0) {
	    return -EFAULT;
	}
	if (get_user(data, user_data_ptr) < 0) {
	    return -EFAULT;
	}
    }

    switch (cmd) {
      case CAMDRV_IOC_INITIALIZE:
        result = initialize();
        break;
      case CAMDRV_IOC_CLEAR:
        result = clear();
        break;
      case CAMDRV_IOC_INHIBIT:
        result = inhibit();
        break;
      case CAMDRV_IOC_RELEASE_INHIBIT:
        result = release_inhibit();
        break;
      case CAMDRV_IOC_ENABLE_INTERRUPT:
        result = enable_interrupt();
        break;
      case CAMDRV_IOC_DISABLE_INTERRUPT:
        result = disable_interrupt();
        break;
      case CAMDRV_IOC_CAMAC_ACTION:
        result = camac_action(parameter, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_READ_LAM:
        result = read_lam(&data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_WAIT_LAM:
        result = wait_lam(parameter, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_SET_CRATE:
        crate_number = parameter;
        result = 0;
        break;
      default:
	result = -EINVAL;
    }

    return result;
}



/* CAMAC service functions */

enum ccp_regs {
    regDATA_LOW = 0x00,
    regDATA_MIDDLE = 0x01,
    regDATA_HIGH = 0x02,
    regLAM = 0x03,
    regSTATION = 0x03,
    regADDRESS = 0x04,
    regSTATUS = 0x05,
    regFUNCTION = 0x05,
    regOTHER_STATUS = 0x06,
    regOTHER_FUNCTION = 0x06,
    regENABLE_DEMAND = 0x07
};

enum ccp_ctrlbits {
    ctrlCLEAR = 0x80,
    ctrlINITIALIZE = 0x40,
    ctrlINHIBIT = 0x01,
};

enum ccp_statbits {
    statQ = 0x01,
    statX = 0x02,
    statDEMMAND = 0x40,
    statREQUEST = 0x80
};


static void ccp_write(int address, int data)
{
    address |= crate_number << 4;
    outb(address, ioport);
    outb(data, ioport + 1);
}

static int ccp_read(int address)
{
    address |= crate_number << 4;
    outb(address, ioport);

    return inb(ioport + 1);
}


static int initialize(void)
{
    ccp_write(regFUNCTION, ctrlINITIALIZE);
    release_inhibit();
    clear();

    return 0;
}

static int clear(void)
{
    ccp_write(regFUNCTION, ctrlCLEAR);
    return 0;
}

static int inhibit(void)
{
    ccp_write(regOTHER_FUNCTION, ctrlINHIBIT | 0x02);
    return 0;
}

static int release_inhibit(void)
{
    ccp_write(regOTHER_FUNCTION, 0x02);
    return 0;
}

static int enable_interrupt(void)
{
    /* this function is not supported by the hardware */
    return 0;
}

static int disable_interrupt(void)
{
    /* this function is not supported by the hardware */
    return 0;
}

static int camac_action(unsigned naf, unsigned* data)
{
    unsigned long start_jiffies;
    int status, nq, nx;

    unsigned short data_l = (*data >> 0) & 0x00ff;
    unsigned short data_m = (*data >> 8) & 0x00ff;
    unsigned short data_h = (*data >> 16) & 0x00ff;

    unsigned short n = (naf >> 9) & 0x1f;
    unsigned short a = (naf >> 5) & 0x0f;
    unsigned short f = (naf >> 0) & 0x1f;

    ccp_write(regDATA_LOW, data_l);
    ccp_write(regDATA_MIDDLE, data_m);
    ccp_write(regDATA_HIGH, data_h);

    ccp_write(regSTATION, n);
    ccp_write(regADDRESS, a);
    ccp_write(regFUNCTION, f);

    start_jiffies = jiffies;
    while ((status = ccp_read(regSTATUS)) & statREQUEST) {
        schedule();
	if (jiffies - start_jiffies > HZ) {
	    *data = 0;
	    return -ETIMEDOUT;
	}
    }

    data_l = ccp_read(regDATA_LOW);
    data_m = ccp_read(regDATA_MIDDLE);
    data_h = ccp_read(regDATA_HIGH);

    *data = (data_h << 16) | (data_m << 8) | data_l;
    nq = (status & statQ) ? 0x00 : 0x01;
    nx = (status & statX) ? 0x00 : 0x01;

    return ((nx << 1) | nq);
}

static int read_lam(unsigned* data)
{
    unsigned short encoded_lam;
    encoded_lam = ccp_read(regLAM);

    if (encoded_lam > 0) {
        *data = 0x0001 << (encoded_lam - 1);
    }
    else {
        *data = 0;
    }

    return *data;
}

struct wait_queue* camdrv_wait_queue = NULL;

static int wait_lam(unsigned timeout, unsigned* data)
{
    unsigned long flags;
    long remaining_time;
    unsigned long timeout_jiffies;
    unsigned short encoded_lam;

    /* The hardware does not support "interrupt on LAM". */
    /* The following code is a "polling loop" to wait for any LAM bits. */
    timeout_jiffies = jiffies + timeout * HZ;
    while ((encoded_lam = ccp_read(regLAM)) == 0) {
#if 0
	/* sleep for one system scheduling cycle */
	save_flags(flags);
	cli();
	remaining_time = interruptible_sleep_on_timeout(&camdrv_wait_queue, 1);
	restore_flags(flags);

	/* Since the hardware does not make interrupts, */
	/* this condition will always be caused by software interrupts, */
	/* such as user signals. */
	if (remaining_time > 0) {
	    *data = 0;
	    return -EINTR;
	}
#else
	schedule();
#endif
	if (jiffies > timeout_jiffies) {
	    *data = 0;
	    return -ETIMEDOUT;
	}
    }
    *data = 0x0001 << (encoded_lam - 1);

    return *data;
}
