/* vmeintwait.c */
/* Created by Enomoto Sanshiro on 19 June 2001. */
/* Last updated by Enomoto Sanshiro on 19 June 2001. */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include "vmedrv.h"


#define DEV_FILE "/dev/vmedrv16d16"
#define TIMEOUT_SEC 10

#define RINEI_RPV130
#define RPV130_BASE_ADDRESS 0x8000


void enable_module_interrupt(int fd) ;
void on_catch_signal(int signal_id);


int main(int argc, char** argv)
{
    int fd;
    int irq, vector;
    int result;
    struct vmedrv_interrupt_property_t interrupt_property;
    static char excess[32];

    if (
	(argc < 3) ||
	(sscanf(argv[1], "%d%s", &irq, excess) != 1) ||
	(sscanf(argv[2], "%x%s", &vector, excess) != 1)
    ){
        fprintf(stderr, "Usage: %s IRQ VECTOR\n", argv[0]);
        fprintf(stderr, "  ex) %s 4 0xaaaa\n", argv[0]);
	exit(EXIT_FAILURE);
    }

    if ((fd = open(DEV_FILE, O_RDWR)) == -1) {
        fprintf(stderr, "ERROR: %s: open(): %s\n", argv[0], strerror(errno));
	exit(EXIT_FAILURE);
    }

    signal(SIGINT, on_catch_signal);
    enable_module_interrupt(fd);

    interrupt_property.irq = irq;
    interrupt_property.vector = vector;
    interrupt_property.signal_id = 0;
    if (ioctl(fd, VMEDRV_IOC_REGISTER_INTERRUPT, &interrupt_property) == -1) {
        fprintf(stderr, "ERROR: %s: ioctl(REGISTER_INTERRUPT): %s\n", argv[0], strerror(errno));
	exit(EXIT_FAILURE);
    }

    if (ioctl(fd, VMEDRV_IOC_ENABLE_INTERRUPT) == -1) {
        fprintf(stderr, "ERROR: %s: ioctl(ENABLE_INTERRUPT): %s\n", argv[0], strerror(errno));
	exit(EXIT_FAILURE);
    }

    interrupt_property.timeout = TIMEOUT_SEC;
    result = ioctl(fd, VMEDRV_IOC_WAIT_FOR_INTERRUPT, &interrupt_property);
    if (result > 0) {
	printf("VME interrupt handled.\n");
    }
    else if (errno == ETIMEDOUT) {
	printf("timed out.\n");
    }
    else if (errno == EINTR) {
	printf("interrupted.\n");
    }
    else {
        fprintf(stderr, "ERROR: %s: ioctl(WAIT_FOR_INTERRUPT): %s\n", argv[0], strerror(errno));
	exit(EXIT_FAILURE);
    }

    if (ioctl(fd, VMEDRV_IOC_DISABLE_INTERRUPT) == -1) {
        fprintf(stderr, "ERROR: %s: ioctl(DISABLE_INTERRUPT): %s\n", argv[0], strerror(errno));
	exit(EXIT_FAILURE);
    }

    if (ioctl(fd, VMEDRV_IOC_UNREGISTER_INTERRUPT, &interrupt_property) == -1) {
        fprintf(stderr, "ERROR: %s: ioctl(UNREGISTER_INTERRUPT): %s\n", argv[0], strerror(errno));
	exit(EXIT_FAILURE);
    }

    close(fd);

    return 0;
}

void on_catch_signal(int signal_id)
{
    printf("signal %d is handled.\n", signal_id);
}

/* set interrupt enable flag on device */
void enable_module_interrupt(int fd) 
{
#ifdef RINEI_RPV130
    void* mapped_address;
    static const int address = RPV130_BASE_ADDRESS;
    static const int map_size = 0x1000;
    if ((mapped_address = mmap(0, map_size, PROT_WRITE, MAP_SHARED, fd, address)) == MAP_FAILED) {
        fprintf(stderr, "ERROR: mmap(): %s\n", strerror(errno));
	exit(EXIT_FAILURE);
    }
    
    /* clear */
    *((short*) (mapped_address + (off_t) 0x000c)) |= 0x03;
    *((short*) (mapped_address + (off_t) 0x000e)) |= 0x03;
    
    /* enable interrupt */
    *((short*) (mapped_address + (off_t) 0x000c)) |= 0x50;
    *((short*) (mapped_address + (off_t) 0x000e)) |= 0x50;

    munmap(mapped_address, map_size);
#endif
}
