/* bufferbench.cc */
/* data buffer benchmark program */
/* Created by Enomoto Sanshiro on 24 July 1998. */
/* Last updated by Enomoto Sanshiro on 13 January 1999. */


#include <iostream>
#include <cstdio>
#include <unistd.h>
#include <string>
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "KinokoBufferServer.hh"
#include "KinokoBufferClient.hh"

using namespace std;


#define SharedMemoryProjectID 100
#define MessageQueueProjectID 101


static char SourceArea[256*1024];
static char DestinationArea[256*1024];

TMushRealTimeTimer TimeKeeper(0, 100000);
TKinokoBufferStreamLogger BufferLogger(cout);


void Write(void) throw(TKinokoException)
{
    TKinokoBufferWriter BufferWriter(
	SharedMemoryProjectID, MessageQueueProjectID, 
	&TimeKeeper, &BufferLogger
    );

    long TotalSize = 256 * 1024 * 1024;
    long PacketSize;

    for (PacketSize = 1024; PacketSize < 128 * 1024; PacketSize += 2 * 1024) {
	long NumberOfPackets = TotalSize / PacketSize;
	double TransferPacketSize = PacketSize * NumberOfPackets;

	void *Address;
	size_t Size = PacketSize;	
	TMushDateTime StartTime, StopTime;
	
	StartTime.Update();
	for (int i = 0; i < NumberOfPackets; i++) {
	    BufferWriter.GetWriteEntry(Address, Size);
	    if (Address != 0) {
		//BufferWriter.ReserveWriteEntry(Size);
		memcpy(Address, SourceArea, Size);
		BufferWriter.SendStrobe(Address, Size);
	    }
	}
	StopTime.Update();
	//BufferWriter.SendStrobe(Address, Size);
		
	double TotalTime = (
            (StopTime.AsLong() - StartTime.AsLong()) + 
            (StopTime.USec() - StartTime.USec()) * 1e-6
        );
	double Speed = TransferPacketSize / TotalTime;
	double PacketTime = (double) TotalTime / NumberOfPackets;
	
	cerr << "total size = " << (TransferPacketSize / 1024 / 1024) << " [MB]" << endl;
	cerr << "# of packets = " << NumberOfPackets << endl;
	cerr << "packet size = " << (PacketSize / 1024) << " [kB]" << endl;
	cerr << "packet time = " << (PacketTime * 1e+3) << " [msec]" << endl;
	cerr << "total time = " << TotalTime << " [sec]" << endl;
	cerr << "speed = " << (Speed / 1024.0 / 1024.0) << " [MB/sec]" << endl;
	cerr << endl;

	cout << (PacketSize / 1024) << "\t";
        cout << (PacketTime * 1e+3) << "\t";
	cout << TotalTime << "\t";
        cout << (Speed / 1024.0 / 1024.0)  << "\t";
	cout << endl; 

	/* flush the buffer */
	sleep(1);
    }
}


void Read(void) throw(TKinokoException)
{
    TKinokoBufferReader BufferReader(
	SharedMemoryProjectID, MessageQueueProjectID, 
	&TimeKeeper, &BufferLogger
    );

    void *Address;
    size_t Size;

    while (1) {
	if (BufferReader.GetReadEntry(Address, Size) > 0) {
	    memcpy(DestinationArea, Address, Size);
	    BufferReader.SendAcknowledge(Address);
	}
    }
}



int main(int argc, char **argv)
{
    if (argc < 2) {
	cerr << "Usage: bufferbench [reader/writer]" << endl;
	return -1;

    }
    string Type = argv[1];

    try {
	if (Type == "reader") {
	    Read();
	}
	else if (Type == "writer") {
	    Write();
	}
	else {
	    cerr << "Usage: bufferbench [reader/writer]" << endl;
	    return -1;
	}
    }
    catch (TKinokoException e) {
	cerr << e << endl;
    }

    return 0;
}
