/* KaspHistogram.hh */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 29 June 2002. */


#ifndef __KaspHistogram_hh__
#define __KaspHistogram_hh__


#include <string>
#include <cmath>
#include "KaspObject.hh"


class TKaspHistogramScale {
  public:
    TKaspHistogramScale(int NumberOfBins, double Min, double Max);
    ~TKaspHistogramScale();
    std::string Title(void) const;
    void SetTitle(const std::string& Title);
    inline int BinNumberOf(double Data) const;
    inline double BinValueOf(int BinNumber) const;
    inline double BinWidthOf(int BinNumber) const;
    inline double BinCenterOf(int BinNumber) const;
    inline double Min(void) const;
    inline double Max(void) const;
    inline int NumberOfBins(void) const;
  protected:
    std::string _Title;
    long _NumberOfBins;
    double _Min, _Max;
    double _BinWidth;
};


class TKaspHistogram: public TKaspObject {
  public:
    TKaspHistogram(const std::string& Title, const TKaspHistogramScale& Scale);
    TKaspHistogram(const std::string& Title, int NumberOfBins, double Min, double Max);
    ~TKaspHistogram();
    void Clear(void);
    bool HasData(void);
  public:
    inline void Fill(double Data, double Weight = 1.0);
    inline TKaspHistogramScale& Scale(void);
    inline const TKaspHistogramScale& Scale(void) const;
    inline double CountsOf(int BinNumber) const;
    inline double ErrorOf(int BinNumber) const;
    inline double UnderflowCounts(void) const;
    inline double OverflowCounts(void) const;
    inline double NumberOfEntries(void) const;
    inline double Mean(void) const;
    inline double Deviation(void) const;
    inline double PeakCounts(void) const;
    inline double PeakValue(void) const;
  protected:
    TKaspHistogramScale _Scale;
    double* _Storage;
    double _UnderflowCounts;
    double _OverflowCounts;
    double _NumberOfEntries;
    double _Sum, _SumOfWeight, _SumOfSquared;
    double _PeakCounts;
    int _PeakBinIndex;
};


inline void TKaspHistogram::Fill(double Data, double Weight)
{
    if (Data < _Scale.Min()) {
	_UnderflowCounts += Weight;
    }
    else if (Data >= _Scale.Max()) {
	_OverflowCounts += Weight;
    }
    else {
	double Counts = (_Storage[_Scale.BinNumberOf(Data)] += Weight);
	if (Counts > _PeakCounts) {
	     _PeakCounts = Counts;
	     _PeakBinIndex = _Scale.BinNumberOf(Data);
	}
    }

    _NumberOfEntries += Weight;

    _Sum += Data * Weight;
    _SumOfWeight += Weight;
    _SumOfSquared += Data * Data * Weight;
}

inline TKaspHistogramScale& TKaspHistogram::Scale(void)
{
    return _Scale;
}

inline const TKaspHistogramScale& TKaspHistogram::Scale(void) const
{
    return _Scale;
}

inline double TKaspHistogram::CountsOf(int BinNumber) const
{
    return _Storage[BinNumber];
}

inline double TKaspHistogram::ErrorOf(int BinNumber) const
{
    return (_Storage[BinNumber] > 0) ? sqrt(_Storage[BinNumber]) : 0;
}

inline double TKaspHistogram::OverflowCounts(void) const
{
    return _OverflowCounts;
}

inline double TKaspHistogram::UnderflowCounts(void) const
{
    return _UnderflowCounts;
}

inline double TKaspHistogram::NumberOfEntries(void) const
{
    return _NumberOfEntries;
}

inline double TKaspHistogram::Mean(void) const
{
    return _Sum / _SumOfWeight;
}

inline double TKaspHistogram::Deviation(void) const
{
    return sqrt(_SumOfSquared / _SumOfWeight - Mean() * Mean());
}

inline double TKaspHistogram::PeakCounts(void) const
{
    return _PeakCounts;
}

inline double TKaspHistogram::PeakValue(void) const
{
    return _Scale.BinCenterOf(_PeakBinIndex);
}



inline int TKaspHistogramScale::BinNumberOf(double Data) const
{
    if (Data < _Min) {
	// Underflow
	return _NumberOfBins + 0;
    }
    else if (Data >= _Max) {
	// Overflow
	return _NumberOfBins + 1;
    }
    else {
	return (int) ((Data - _Min) / (_Max - _Min) * _NumberOfBins);
    }
}

inline double TKaspHistogramScale::BinValueOf(int BinNumber) const
{
    if ((BinNumber < 0) || (BinNumber > _NumberOfBins)) {
	// out of range
	return 2.0 * _Max;
    }

    return _Min + (double) BinNumber / _NumberOfBins * (_Max - _Min);
}

inline double TKaspHistogramScale::BinWidthOf(int BinNumber) const
{
    return _BinWidth;
}

inline double TKaspHistogramScale::BinCenterOf(int BinNumber) const
{
    return BinValueOf(BinNumber) + _BinWidth / 2;
}

inline double TKaspHistogramScale::Min(void) const
{
    return _Min;
}

inline double TKaspHistogramScale::Max(void) const
{
    return _Max;
}

inline int TKaspHistogramScale::NumberOfBins(void) const
{
    return _NumberOfBins;
}


#endif
