/* KaspHistory.hh */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 29 June 2002. */


#ifndef __KaspHistory_hh__
#define __KaspHistory_hh__


#include <string>
#include "KaspObject.hh"


class TKaspHistory: public TKaspObject {
  public:
    TKaspHistory(const std::string& Title, long MaxNumberOfSamples);
    ~TKaspHistory();
    void Clear(void);
    bool HasData(void);
  public:
    void Start(long CurrentTime);
    void HoldSample(long CurrentTime);
    void InsertSample(long Time, long Counts, double Sum, double Mean, double Deviation);
  public:
    inline void Fill(double Data);
    inline long MaxNumberOfSamples(void);
    inline long NumberOfSamples(void);
    inline long StartTime(void);
    inline long TimeOf(long SampleIndex);
    inline long PassedTimeOf(long SampleIndex);
    inline long CountsOf(long SampleIndex);
    inline double SumOf(long SampleIndex);
    inline double MeanOf(long SampleIndex);
    inline double DeviationOf(long SampleIndex);
    inline long LastTime(void);
    inline long LastPassedTime(void);
    inline long LastCounts(void);
    inline double LastSum(void);
    inline double LastMean(void);
    inline double LastDeviation(void);
  protected:
    long _MaxNumberOfSamples;
    long _NumberOfSamples;
    long _BaseIndex;
    long _StartTime;
    long* _TimeStorage;
    long* _CountStorage;
    double* _SumStorage;
    double* _MeanStorage;
    double* _DeviationStorage;
  private:
    long _CurrentCounts;
    double _CurrentSum;
    double _CurrentSumOfSquared;
};


inline void TKaspHistory::Fill(double Data)
{
    _CurrentSum += Data;
    _CurrentSumOfSquared += Data * Data;
    _CurrentCounts++;
}

inline long TKaspHistory::MaxNumberOfSamples(void)
{
    return _MaxNumberOfSamples;
}

inline long TKaspHistory::NumberOfSamples(void)
{
    return _NumberOfSamples;
}

inline long TKaspHistory::StartTime(void)
{
    return _StartTime;
}

inline long TKaspHistory::TimeOf(long SampleIndex)
{
    return _TimeStorage[(_BaseIndex + SampleIndex) % _MaxNumberOfSamples];
}

inline long TKaspHistory::PassedTimeOf(long SampleIndex)
{
    return _TimeStorage[(_BaseIndex + SampleIndex) % _MaxNumberOfSamples] - _StartTime;
}

inline double TKaspHistory::SumOf(long SampleIndex)
{
    return _SumStorage[(_BaseIndex + SampleIndex) % _MaxNumberOfSamples];
}

inline double TKaspHistory::MeanOf(long SampleIndex)
{
    return _MeanStorage[(_BaseIndex + SampleIndex) % _MaxNumberOfSamples];
}

inline double TKaspHistory::DeviationOf(long SampleIndex)
{
    return _DeviationStorage[(_BaseIndex + SampleIndex) % _MaxNumberOfSamples];
}

inline long TKaspHistory::CountsOf(long SampleIndex)
{
    return _CountStorage[(_BaseIndex + SampleIndex) % _MaxNumberOfSamples];
}

inline long TKaspHistory::LastTime(void)
{
    return _TimeStorage[(_BaseIndex + _NumberOfSamples - 1) % _MaxNumberOfSamples];
}

inline long TKaspHistory::LastPassedTime(void)
{
    return _TimeStorage[(_BaseIndex + _NumberOfSamples - 1) % _MaxNumberOfSamples] - _StartTime;
}

inline long TKaspHistory::LastCounts(void)
{
    return _CountStorage[(_BaseIndex + _NumberOfSamples - 1) % _MaxNumberOfSamples];
}

inline double TKaspHistory::LastSum(void)
{
    return _SumStorage[(_BaseIndex + _NumberOfSamples - 1) % _MaxNumberOfSamples];
}

inline double TKaspHistory::LastMean(void)
{
    return _MeanStorage[(_BaseIndex + _NumberOfSamples - 1) % _MaxNumberOfSamples];
}

inline double TKaspHistory::LastDeviation(void)
{
    return _DeviationStorage[(_BaseIndex + _NumberOfSamples - 1) % _MaxNumberOfSamples];
}

#endif
