/* KaspNtuple.hh */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 29 June 2002. */


#ifndef __KaspNtuple_hh__
#define __KaspNtuple_hh__


#include <string>
#include <vector>
#include <map>
#include "KaspObject.hh"


class TKaspNtuple: public TKaspObject {
  public:
    TKaspNtuple(const std::string& Title, unsigned NumberOfColumns);
    ~TKaspNtuple();
    void Clear(void);
    bool HasData(void);
  public:
    void SetColumnName(unsigned ColumnIndex, const std::string& ColumnName);
    const std::string& ColumnNameOf(unsigned ColumnIndex) throw(TKaspException);
    unsigned ColumnIndexOf(const std::string& ColumnName) throw(TKaspException);
  public:
    inline unsigned Fill(const double RowData[]);
    inline unsigned NumberOfColumns(void) const;
    inline unsigned NumberOfRows(void) const;
    inline double* operator[] (unsigned RowIndex) throw(TKaspException);
  protected:
    std::map<unsigned, std::string> _ColumnIndexNameTable;
    std::map<std::string, unsigned> _ColumnNameIndexTable;
    unsigned _NumberOfColumns, _NumberOfRows;
    unsigned _NumberOfBlocks, _BlockRows;
    std::vector<double*> _StorageBlockList;
};


inline unsigned TKaspNtuple::NumberOfColumns(void) const
{
    return _NumberOfColumns;
}

inline unsigned TKaspNtuple::NumberOfRows(void) const
{
    return _NumberOfRows;
}

inline unsigned TKaspNtuple::Fill(const double RowData[])
{
    _NumberOfRows++;
    if (_NumberOfRows > _NumberOfBlocks * _BlockRows) {
	_StorageBlockList.push_back(new double[_BlockRows * _NumberOfColumns]);
	_NumberOfBlocks++;
    }

    double* Row = this->operator[](_NumberOfRows - 1);
    memcpy(Row, RowData, sizeof(double) * _NumberOfColumns);

    return _NumberOfRows;
}

inline double* TKaspNtuple::operator[] (unsigned RowIndex) throw(TKaspException)
{
    if (RowIndex >= _NumberOfRows) {
	throw TKaspException("TKaspNtuple::opeartor[]", "index out of range");
    }

    double*& Block = _StorageBlockList[RowIndex / _BlockRows];
    return Block + ((RowIndex % _BlockRows) * _NumberOfColumns);
}


#endif
