/* KaspPlainTextRepository.cc */
/* Created by Enomoto Sanshiro on 26 June 2002. */
/* Last updated by Enomoto Sanshiro on 23 December 2002. */


#include <string>
#include <fstream>
#include <strstream>
#include <cstdio>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "KaspDefs.hh"
#include "KaspNtuple.hh"
#include "KaspGraph.hh"
#include "KaspHistogram.hh"
#include "Kasp2dHistogram.hh"
#include "KaspHistory.hh"
#include "KaspWave.hh"
#include "KaspMap.hh"
#include "KaspTabular.hh"
#include "KaspRepository.hh"
#include "KaspPlainTextRepository.hh"

using namespace std;


TKaspPlainTextRepository::TKaspPlainTextRepository(const string& RepositoryName)
{
    // repository is a directory //
    _RepositoryName = RepositoryName;
    _IsRepositoryCreated = false;
}

TKaspPlainTextRepository::TKaspPlainTextRepository(void)
{
    // repository is a file //
    _RepositoryName = "";
    _IsRepositoryCreated = false;
}

TKaspPlainTextRepository::~TKaspPlainTextRepository()
{
}

string TKaspPlainTextRepository::FileNameOf(const string& ObjectName, int Revision)
{
    char FileName[128];
    ostrstream os(FileName, sizeof(FileName));

    if (! _RepositoryName.empty()) {
	os << _RepositoryName << "/";
    }

    os << ObjectName;
    if (Revision >= 0) {
	os << ";" << Revision;
    }

    if (! _RepositoryName.empty()) {
	os << ".knt";
    }

    os << ends;

    return FileName;
}

ofstream* TKaspPlainTextRepository::OpenToWrite(const string& ObjectName, int Revision) throw(TKaspException)
{
    if ((! _IsRepositoryCreated) && (! _RepositoryName.empty())) {
	if (access(_RepositoryName.c_str(), F_OK) != 0) {
	    mkdir(_RepositoryName.c_str(), 0755);
	}
	else {
	    struct stat FileStat;
	    if (stat(_RepositoryName.c_str(), &FileStat) < 0) {
		throw TKaspException(
		    "TKaspPlainTextRepository::OpenToWrite()",
		    "systemcall stat(2) failed: " + _RepositoryName
		);
	    }
	    if (! S_ISDIR(FileStat.st_mode)) {
		throw TKaspException(
		    "TKaspPlainTextRepository::OpenToWrite()",
		    "file already exists: " + _RepositoryName
		);
	    }
	}
    }

    string FileName = FileNameOf(ObjectName);
    if (Revision > 0) {
	string PreviousFileName = FileNameOf(ObjectName, Revision - 1);
	rename(FileName.c_str(), PreviousFileName.c_str());
    }

    ofstream* File = new ofstream(FileName.c_str());

    if (! *File) {
	delete File;
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenToWrite()",
	    "unable to create file: " + FileName
	);
    }

    return File;
}

ifstream* TKaspPlainTextRepository::OpenToRead(const string& ObjectName, int Revision) throw(TKaspException)
{
    string FileName = FileNameOf(ObjectName, Revision);
    ifstream* File = new ifstream(FileName.c_str());

    if (! *File) {
	delete File;
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenToRead()",
	    "unable to open file: " + FileName
	);
    }

    return File;
}

void TKaspPlainTextRepository::SaveNtuple(const string& Name, TKaspNtuple* Ntuple) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    *File << "# Type: Ntuple" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << Ntuple->Title() << endl;

    *File << "# Fields: ";
    unsigned NumberOfColumns = Ntuple->NumberOfColumns();
    for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	*File << Ntuple->ColumnNameOf(Column) << " ";
    }
    *File << endl << endl;

    for (unsigned Row = 0; Row < Ntuple->NumberOfRows(); Row++) {
	for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	    *File << (*Ntuple)[Row][Column] << " ";
	}
	*File << endl;
    }

    delete File;
}

void TKaspPlainTextRepository::SaveGraph(const string& Name, TKaspGraph* Graph) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    *File << "# Type: Graph" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << Graph->Title() << endl;

    if (Graph->HasErrors()) {
	*File << "# Fields: x y xerror yerror" << endl;
    }
    else {
	*File << "# Fields: x y" << endl;
    }
    *File << endl;

    const double* XValueList = Graph->XValueList();
    const double* YValueList = Graph->YValueList();
    const double* XErrorList = Graph->XErrorList();
    const double* YErrorList = Graph->YErrorList();
    unsigned NumberOfPoints = Graph->NumberOfPoints();

    for (unsigned i = 0; i < NumberOfPoints; i++) {
	*File << XValueList[i] << " " << YValueList[i];
	if (Graph->HasErrors()) {
	    *File << " " << XErrorList[i] << " " << YErrorList[i];
	}
	*File << endl;
    }

    delete File;
}

void TKaspPlainTextRepository::SaveHistogram(const string& Name, TKaspHistogram* Histogram) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    const TKaspHistogramScale& Scale = Histogram->Scale();

    *File << "# Type: Histogram" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << Histogram->Title() << endl;
    *File << "# AxisTitle: " << Scale.Title() << endl;
    *File << "# NumberOfBins: " << Scale.NumberOfBins() << endl;
    *File << "# Min: " << Scale.Min() << endl;
    *File << "# Max: " << Scale.Max() << endl;
    *File << "# Underflow: " <<  Histogram->UnderflowCounts() << endl;
    *File << "# Overflow: " <<  Histogram->OverflowCounts() << endl;
    *File << "# Fields: bin_center counts" << endl;
    *File << endl;

    for (int BinIndex = 0; BinIndex < Scale.NumberOfBins(); BinIndex++) {
	*File << Scale.BinCenterOf(BinIndex) << " ";
	*File << Histogram->CountsOf(BinIndex) << endl;
    }

    delete File;
}

void TKaspPlainTextRepository::Save2dHistogram(const string& Name, TKasp2dHistogram* Histogram) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    const TKaspHistogramScale& XScale = Histogram->XScale();
    const TKaspHistogramScale& YScale = Histogram->YScale();

    *File << "# Type: Histogram2d" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << Histogram->Title() << endl;
    *File << "# XAxisTitle: " << XScale.Title() << endl;
    *File << "# YAxisTitle: " << YScale.Title() << endl;
    *File << "# NumberOfXBins: " << XScale.NumberOfBins() << endl;
    *File << "# XMin: " << XScale.Min() << endl;
    *File << "# XMax: " << XScale.Max() << endl;
    *File << "# NumberOfYBins: " << YScale.NumberOfBins() << endl;
    *File << "# YMin: " << YScale.Min() << endl;
    *File << "# YMax: " << YScale.Max() << endl;
    *File << "# Fields: xbin_center ybin_center counts" << endl;
    *File << endl;

    for (int YBinIndex = 0; YBinIndex < YScale.NumberOfBins(); YBinIndex++) {
	for (int XBinIndex = 0; XBinIndex < XScale.NumberOfBins(); XBinIndex++) {
	    *File << XScale.BinCenterOf(XBinIndex) << " ";
	    *File << YScale.BinCenterOf(YBinIndex) << " ";
	    *File << Histogram->CountsOf(XBinIndex, YBinIndex) << endl;
	}
    }

    delete File;
}

void TKaspPlainTextRepository::SaveHistory(const string& Name, TKaspHistory* History) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    *File << "# Type: History" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << History->Title() << endl;
    *File << "# Depth: " << History->MaxNumberOfSamples() << endl;
    *File << "# Fields: time counts sum mean deviation" << endl;
    *File << endl;

    for (int Index = 0; Index < History->NumberOfSamples(); Index++) {
	*File << History->PassedTimeOf(Index) << " ";
	*File << History->CountsOf(Index) << " ";
	*File << History->SumOf(Index) << " ";
	*File << History->MeanOf(Index) << " ";
	*File << History->DeviationOf(Index) << endl;
    }

    delete File;
}

void TKaspPlainTextRepository::SaveWave(const string& Name, TKaspWave* Wave) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    *File << "# Type: Wave" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << Wave->Title() << endl;
    *File << "# Fields: index value" << endl;
    *File << endl;

    for (int Index = 0; Index < Wave->NumberOfPoints(); Index++) {
	*File << Index << " ";
	*File << Wave->ValueOf(Index) << endl;
    }

    delete File;
}

void TKaspPlainTextRepository::SaveMap(const string& Name, TKaspMap* Map) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    *File << "# Type: Map" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << Map->Title() << endl;
    *File << "# Fields: address value" << endl;
    *File << endl;

    for (int Index = 0; Index < Map->NumberOfPoints(); Index++) {
	*File << Map->AddressOf(Index) << " ";
	*File << Map->ValueOf(Index) << endl;
    }

    delete File;
}

void TKaspPlainTextRepository::SaveTabular(const string& Name, TKaspTabular* Tabular) throw(TKaspException)
{
    int Revision = _RevisionTable[Name];
    _RevisionTable[Name] += 1;

    ofstream* File = OpenToWrite(Name, Revision);

    *File << "# Type: Tabular" << endl;
    *File << "# Name: " << Name << endl;
    *File << "# Revision: " << Revision << endl;
    *File << "# Title: " << Tabular->Title() << endl;
    *File << "# Fields: field_name value" << endl;
    *File << endl;

    for (int Index = 0; Index < Tabular->NumberOfFields(); Index++) {
	*File << Tabular->FieldNameOf(Index) << " ";
	*File << Tabular->ValueOf(Index) << endl;
    }

    delete File;
}

TKaspNtuple* TKaspPlainTextRepository::LoadNtuple(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKaspNtuple* Ntuple = 0;
    static const int MaxNumberOfColumns = 1024;

    int NumberOfColumns = 0;
    double* Row = new double[MaxNumberOfColumns];

    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istrstream LineStream(Line.c_str());
	int ColumnIndex = 0;
	double Value;
	while (LineStream >> ws >> Value) {
	    if (ColumnIndex >= MaxNumberOfColumns) {
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadNtuple()",
		    "too many columns"
		);
            }

            if (Ntuple == 0) {
       	        NumberOfColumns++;
            }
            Row[ColumnIndex] = Value;
            ColumnIndex++;
        }
	
	if (Ntuple == 0) {
	    Ntuple = new TKaspNtuple(Name, NumberOfColumns);
	}
	else if (ColumnIndex != NumberOfColumns) {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadNtuple()",
		"inconsistent number of columns"
	    );
	}

	Ntuple->Fill(Row);
    }

    if (Ntuple == 0) {
	NumberOfColumns = 0;
	Ntuple = new TKaspNtuple(Name, NumberOfColumns);
    }

    Ntuple->SetTitle(Header.ItemValueOf("Title"));
    vector<string> FieldNameList = Header.ItemValueListOf("Fields");
    for (unsigned Index = 0; Index < FieldNameList.size(); Index++) {
	Ntuple->SetColumnName(Index, FieldNameList[Index]);
    }
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Ntuple->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    delete[] Row;

    return Ntuple;
}

TKaspGraph* TKaspPlainTextRepository::LoadGraph(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKaspGraph* Graph = new TKaspGraph(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istrstream LineStream(Line.c_str());
	double X, Y, XError, YError;
	if (LineStream >> X >> Y) {
	    if (LineStream >> XError >> YError) {
		Graph->Fill(X, Y, XError, YError);
	    }
	    else {
		Graph->Fill(X, Y);
	    }
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadGraph()",
		"invalid data line: " + Line
	    );
	}
    }

    Graph->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Graph->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    return Graph;
}

TKaspHistogram* TKaspPlainTextRepository::LoadHistogram(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKaspHistogram* Histogram = 0;
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	if (Histogram == 0) {
	    string NumberOfBinsString = Header.ItemValueOf("NumberOfBins");
	    string MinString = Header.ItemValueOf("Min");
	    string MaxString = Header.ItemValueOf("Max");

	    istrstream NumberOfBinsStream(NumberOfBinsString.c_str());
	    istrstream MinStream(MinString.c_str());
	    istrstream MaxStream(MaxString.c_str());

	    int NumberOfBins;
	    float Min, Max;
	    if (
		! (NumberOfBinsStream >> NumberOfBins) ||
		! (MinStream >> Min) ||
		! (MaxStream >> Max)
	    ){
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadHistogram()",
		    "unable to get histogram parameters"
		);
	    }

	    Histogram = new TKaspHistogram(Name, NumberOfBins, Min, Max);
	}

	istrstream LineStream(Line.c_str());
	double BinCenter, Counts;
	if (LineStream >> BinCenter >> Counts) {
	    Histogram->Fill(BinCenter, Counts);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadHistogram()",
		"invalid data line: " + Line
	    );
	}
    }

    Histogram->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Histogram->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    return Histogram;
}

TKasp2dHistogram* TKaspPlainTextRepository::Load2dHistogram(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKasp2dHistogram* Histogram = 0;
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	if (Histogram == 0) {
	    string NumberOfXBinsString = Header.ItemValueOf("NumberOfXBins");
	    string XMinString = Header.ItemValueOf("XMin");
	    string XMaxString = Header.ItemValueOf("XMax");
	    string NumberOfYBinsString = Header.ItemValueOf("NumberOfYBins");
	    string YMinString = Header.ItemValueOf("YMin");
	    string YMaxString = Header.ItemValueOf("YMax");

	    istrstream NumberOfXBinsStream(NumberOfXBinsString.c_str());
	    istrstream XMinStream(XMinString.c_str());
	    istrstream XMaxStream(XMaxString.c_str());
	    istrstream NumberOfYBinsStream(NumberOfYBinsString.c_str());
	    istrstream YMinStream(YMinString.c_str());
	    istrstream YMaxStream(YMaxString.c_str());

	    int NumberOfXBins, NumberOfYBins;
	    float XMin, XMax, YMin, YMax;
	    if (
		! (NumberOfXBinsStream >> NumberOfXBins) ||
		! (XMinStream >> XMin) ||
		! (XMaxStream >> XMax) ||
		! (NumberOfYBinsStream >> NumberOfYBins) ||
		! (YMinStream >> YMin) ||
		! (YMaxStream >> YMax)
	    ){
		throw TKaspException(
		    "TKaspPlainTextRepository::Load2dHistogram()",
		    "unable to get histogram parameters"
		);
	    }

	    Histogram = new TKasp2dHistogram(
		Name, 
		TKaspHistogramScale(NumberOfXBins, XMin, XMax),
		TKaspHistogramScale(NumberOfYBins, YMin, YMax)
	    );
	}

	istrstream LineStream(Line.c_str());
	double XBinCenter, YBinCenter, Counts;
	if (LineStream >> XBinCenter >> YBinCenter >> Counts) {
	    Histogram->Fill(XBinCenter, YBinCenter, Counts);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadHistogram()",
		"invalid data line: " + Line
	    );
	}
    }

    Histogram->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Histogram->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    return Histogram;
}

TKaspHistory* TKaspPlainTextRepository::LoadHistory(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKaspHistory* History = 0;
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	if (History == 0) {
	    string DepthString = Header.ItemValueOf("Depth");
	    istrstream DepthStream(DepthString.c_str());
	    int MaxNumberOfSamples;
	    if (! (DepthStream >> MaxNumberOfSamples)) {
		throw TKaspException(
		    "TKaspPlainTextRepository::LoadHistory()",
		    "unable to get history parameters"
		);
	    }

	    History = new TKaspHistory(Name, MaxNumberOfSamples);
	}

	istrstream LineStream(Line.c_str());
	long Time, Counts; double Sum, Mean, Deviation;
	if (LineStream >> Time >> Counts >> Sum >> Mean >> Deviation) {
	    History->InsertSample(Time, Counts, Sum, Mean, Deviation);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadHistory()",
		"invalid data line: " + Line
	    );
	}
    }

    History->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	History->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    return History;
}

TKaspWave* TKaspPlainTextRepository::LoadWave(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKaspWave* Wave = new TKaspWave(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istrstream LineStream(Line.c_str());
	long Index; double Value;
	if (LineStream >> Index >> Value) {
	    Wave->Fill(Index, Value);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadWave()",
		"invalid data line: " + Line
	    );
	}
    }

    Wave->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Wave->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    return Wave;
}

TKaspMap* TKaspPlainTextRepository::LoadMap(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKaspMap* Map = new TKaspMap(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istrstream LineStream(Line.c_str());
	long Address; double Value;
	if (LineStream >> Address >> Value) {
	    Map->Fill(Address, Value);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadMap()",
		"invalid data line: " + Line
	    );
	}
    }

    Map->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Map->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    return Map;
}

TKaspTabular* TKaspPlainTextRepository::LoadTabular(const string& Name, int Revision) throw(TKaspException)
{
    ifstream* File = OpenToRead(Name, Revision);

    TKaspTabular* Tabular = new TKaspTabular(Name);
    TKaspPlainTextRepositoryHeader Header;

    string Line;
    while (getline(*File, Line, '\n')) {
	if (Line.empty()) {
	    continue;
	}
	if (Line[0] == '#') {
	    Header.ReadLine(Line);
	    continue;
	}

	istrstream LineStream(Line.c_str());
	string FieldName, Value;
	if (
	    (LineStream >> ws >> FieldName) && 
	    getline(LineStream >> ws, Value)
	){
	    Tabular->Fill(FieldName, Value);
	}
	else {
	    throw TKaspException(
		"TKaspPlainTextRepository::LoadTabular()",
		"invalid data line: " + Line
	    );
	}
    }

    Tabular->SetTitle(Header.ItemValueOf("Title"));
    for (unsigned i = 0; i < Header.NumberOfItems(); i++) {
	Tabular->AddProperty(Header.ItemNameOf(i), Header.ItemValueOf(i));
    }

    return Tabular;
}



TKaspPlainTextRepositoryHeader::TKaspPlainTextRepositoryHeader(void)
{
}

TKaspPlainTextRepositoryHeader::~TKaspPlainTextRepositoryHeader(void)
{
}

unsigned TKaspPlainTextRepositoryHeader::NumberOfItems()
{
    return _ItemList.size();
}

bool TKaspPlainTextRepositoryHeader::ReadLine(const string& HeaderItemLine)
{
    // syntax: ^[#\s]*([\w]*)[\s:]*(.*)$, Name = \1, Value = \2

    string Name;
    string Value;
    
    enum TState { 
	State_Head, State_Name, State_Separator, State_Value
    };

    TState State = State_Head;

    for (unsigned i = 0; i < HeaderItemLine.size(); i++) {
	char Char = HeaderItemLine[i];

	if (State == State_Head) {
	    if ((Char == '#') || isspace(Char)) {
		;
	    }
	    else {
		State = State_Name;
	    }
	}
	if (State == State_Name) {
	    if (! isspace(Char) && (Char != ':')) {
		Name += Char;
	    }
	    else {
		State = State_Separator;
	    }
	}
	if (State == State_Separator) {
	    if (isspace(Char) || (Char == ':')) {
		;
	    }
	    else {
		State = State_Value;
	    }
	}
	if (State == State_Value) {
	    Value += Char;
	}
    }

    if (State != State_Value) {
	return false;
    }

    _ItemList.push_back(make_pair(Name, Value));
    _ItemTable[Name] = Value;

    return true;
}

string TKaspPlainTextRepositoryHeader::ItemNameOf(unsigned ItemIndex)
{
    return _ItemList[ItemIndex].first;
}

string TKaspPlainTextRepositoryHeader::ItemValueOf(unsigned ItemIndex)
{
    return _ItemList[ItemIndex].second;
}

string TKaspPlainTextRepositoryHeader::ItemValueOf(const string& ItemName)
{
    if (_ItemTable.count(ItemName) == 0) {
	return string();
    }
    else {
	return _ItemTable[ItemName];
    }
}

vector<string> TKaspPlainTextRepositoryHeader::ItemValueListOf(unsigned ItemIndex)
{
    return SplitValue(_ItemList[ItemIndex].second);
}

vector<string> TKaspPlainTextRepositoryHeader::ItemValueListOf(const string& ItemName)
{
    if (_ItemTable.count(ItemName) == 0) {
	return vector<string>();
    }
    else {
	return SplitValue(_ItemTable[ItemName]);
    }
}

vector<string> TKaspPlainTextRepositoryHeader::SplitValue(const string& Value, const string& SeparatorSet)
{
    vector<string> Result;
    string CurrentText;

    for (unsigned i = 0; i < Value.size(); i++) {
	if (SeparatorSet.find_first_of(Value[i]) == string::npos) {
	    CurrentText += Value[i];
	}
	else {
	    if (! CurrentText.empty()) {
		Result.push_back(CurrentText);
		CurrentText.erase(CurrentText.begin(), CurrentText.end());
	    }
	}
    }

    if (! CurrentText.empty()) {
	Result.push_back(CurrentText);
	CurrentText.erase(CurrentText.begin(), CurrentText.end());
    }

    return Result;
}
