/* KaspRootRepository.cc */
/* Created by Enomoto Sanshiro on 26 June 2002. */
/* Last updated by Enomoto Sanshiro on 26 June 2002. */


#include <string>
#include <fstream>
#include <cstring>
#include "KaspDefs.hh"
#include "KaspNtuple.hh"
#include "KaspGraph.hh"
#include "KaspHistogram.hh"
#include "Kasp2dHistogram.hh"
#include "KaspHistory.hh"
#include "KaspWave.hh"
#include "KaspMap.hh"
#include "KaspTabular.hh"
#include "KaspRepository.hh"
#include "KaspRootRepository.hh"

#include <TFile.h>
#include <TNtuple.h>
#include <TTree.h>
#include <TH1.h>
#include <TH2.h>
#include <TGraph.h>
#include <TGraphErrors.h>

using namespace std;


TKaspRootRepository::TKaspRootRepository(const string& FileName)
{
    _FileName = FileName;

    _OutputFile = 0;
    _InputFile = 0;
}

TKaspRootRepository::~TKaspRootRepository()
{
    Close();

    delete _InputFile;
    delete _OutputFile;
}

void TKaspRootRepository::OpenToWrite(void) throw(TKaspException)
{
    if (_OutputFile) {
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenToWrite()", 
	    "file already opened"
	);
    }
    if (_InputFile) {
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenToWrite()", 
	    "file already opened in read-only mode"
	);
    }

    _OutputFile = new TFile(_FileName.c_str(), "RECREATE");
}

void TKaspRootRepository::OpenToRead(void) throw(TKaspException)
{
    if (_InputFile) {
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenToRead()", 
	    "file already opened"
	);
    }
    if (_OutputFile) {
	throw TKaspException(
	    "TKaspPlainTextRepository::OpenToRead()", 
	    "file already opened in write-only mode"
	);
    }

    _InputFile = new TFile(_FileName.c_str());
}

void TKaspRootRepository::Close(void)
{
    if (_OutputFile) {
	_OutputFile->Write();
	delete _OutputFile;
	_OutputFile = 0;
    }

    if (_InputFile) {
	_InputFile->Write();
	delete _InputFile;
	_InputFile = 0;
    }
}

void TKaspRootRepository::SaveNtuple(const std::string& Name, TKaspNtuple* Ntuple) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    unsigned NumberOfColumns = Ntuple->NumberOfColumns();

    string ColumnNameList;
    for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	string ColumnName = Ntuple->ColumnNameOf(Column);
	if (Column != 0) {
	    ColumnNameList += ':';
	}
	ColumnNameList += ColumnName;
    }
    
    TNtuple* RootNtuple = new TNtuple(
	Name.c_str(), Ntuple->Title().c_str(), ColumnNameList.c_str()
    );

    float* Data = new float[NumberOfColumns];
    for (unsigned Row = 0; Row < Ntuple->NumberOfRows(); Row++) {
	for (unsigned Column = 0; Column < NumberOfColumns; Column++) {
	    Data[Column] = (*Ntuple)[Row][Column];
	}
	RootNtuple->Fill(Data);
    }
    delete[] Data;

    RootNtuple->Write();
    delete RootNtuple;
}

void TKaspRootRepository::SaveGraph(const std::string& Name, TKaspGraph* Graph) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    int NumberOfPoints = Graph->NumberOfPoints();
    if (NumberOfPoints == 0) {
	// ROOT cannot make a Graph object when NumberOfPoints is 0.
	return;
    }

    TGraph* RootGraph = 0;
    if (Graph->HasErrors()) {
	RootGraph = new TGraphErrors(
	    Graph->NumberOfPoints(), 
	    Graph->XValueList(), Graph->YValueList(),
	    Graph->XErrorList(), Graph->YErrorList()
	);
    }
    else {
	RootGraph = new TGraph(
	    Graph->NumberOfPoints(), Graph->XValueList(), Graph->YValueList()
	);
    }

    RootGraph->SetNameTitle(Name.c_str(), Graph->Title().c_str());
    RootGraph->Write();

    delete RootGraph;
}

void TKaspRootRepository::SaveHistogram(const std::string& Name, TKaspHistogram* Histogram) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    const TKaspHistogramScale& Scale = Histogram->Scale();
    int NumberOfBins = Scale.NumberOfBins();

    TH1* RootHistogram = new TH1D(
	Name.c_str(), Histogram->Title().c_str(),
	NumberOfBins, Scale.Min(), Scale.Max()
    );

    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	double BinCenter = Scale.BinCenterOf(BinIndex);
	double Count = Histogram->CountsOf(BinIndex);
	RootHistogram->Fill(BinCenter, Count);
    }

    RootHistogram->SetBinContent(0, Histogram->UnderflowCounts());
    RootHistogram->SetBinContent(NumberOfBins + 1, Histogram->OverflowCounts());
    RootHistogram->SetEntries(Histogram->NumberOfEntries());

    RootHistogram->Write();

    delete RootHistogram;
}

void TKaspRootRepository::Save2dHistogram(const std::string& Name, TKasp2dHistogram* Histogram) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    const TKaspHistogramScale& XScale = Histogram->XScale();
    const TKaspHistogramScale& YScale = Histogram->YScale();

    int NumberOfXBins = XScale.NumberOfBins();
    int NumberOfYBins = XScale.NumberOfBins();

    TH2* RootHistogram = new TH2D(
	Name.c_str(), Histogram->Title().c_str(),
	NumberOfXBins, XScale.Min(), XScale.Max(),
	NumberOfYBins, YScale.Min(), YScale.Max()
    );

    double NumberOfEntries = 0;
    for (int YBinIndex = 0; YBinIndex < NumberOfXBins; YBinIndex++) {
	for (int XBinIndex = 0; XBinIndex < NumberOfXBins; XBinIndex++) {
	    double XBinCenter = XScale.BinCenterOf(XBinIndex);
	    double YBinCenter = YScale.BinCenterOf(YBinIndex);
	    double Count = Histogram->CountsOf(XBinIndex, YBinIndex);

	    RootHistogram->Fill(XBinCenter, YBinCenter, Count);
	    NumberOfEntries += Count;
	}
    }

    RootHistogram->SetEntries(NumberOfEntries);

    RootHistogram->Write();

    delete RootHistogram;
}

void TKaspRootRepository::SaveHistory(const std::string& Name, TKaspHistory* History) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    TNtuple* RootNtuple = new TNtuple(
	Name.c_str(), History->Title().c_str(),
	"time:counts:sum:mean:deviation"
    );

    float Data[5];
    for (int Index = 0; Index < History->NumberOfSamples(); Index++) {
	Data[0] = History->PassedTimeOf(Index);
	Data[1] = History->CountsOf(Index);
	Data[2] = History->SumOf(Index);
	Data[3] = History->MeanOf(Index);
	Data[4] = History->DeviationOf(Index);

	RootNtuple->Fill(Data);
    }

    RootNtuple->Write();
    delete RootNtuple;
}

void TKaspRootRepository::SaveWave(const std::string& Name, TKaspWave* Wave) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    int NumberOfPoints = Wave->NumberOfPoints();
    if (NumberOfPoints == 0) {
	// ROOT cannot make a Graph object when the NumberOfPoints is 0.
	return;
    }

    double* XDataList = new double[NumberOfPoints];
    double* YDataList = new double[NumberOfPoints];

    for (int Index = 0; Index < NumberOfPoints; Index++) {
	XDataList[Index] = Index;
	YDataList[Index] = Wave->ValueOf(Index);
    }

    TGraph* RootGraph = new TGraph(NumberOfPoints, XDataList, YDataList);
    RootGraph->SetNameTitle(Name.c_str(), Wave->Title().c_str());
    RootGraph->Write();

    delete RootGraph;
    delete[] XDataList;
    delete[] YDataList;
}

void TKaspRootRepository::SaveMap(const std::string& Name, TKaspMap* Map) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    TNtuple* RootNtuple = new TNtuple(
	Name.c_str(), Map->Title().c_str(), "address:value"
    );

    for (int Index = 0; Index < Map->NumberOfPoints(); Index++) {
	RootNtuple->Fill(Map->AddressOf(Index), Map->ValueOf(Index));
    }

    RootNtuple->Write();
    delete RootNtuple;
}

void TKaspRootRepository::SaveTabular(const std::string& Name, TKaspTabular* Tabular) throw(TKaspException)
{
    if (_OutputFile == 0) {
	OpenToWrite();
    }

    int NumberOfFields = Tabular->NumberOfFields();
    char** BufferList = new char* [NumberOfFields];

    TTree* Tree = new TTree(Name.c_str(), Tabular->Title().c_str());
    for (int Index = 0; Index < NumberOfFields; Index++) {
	const char* FieldName = Tabular->FieldNameOf(Index).c_str();
	const char* Value = Tabular->ValueOf(Index).c_str();

	BufferList[Index] = new char[64];
	strncpy(BufferList[Index], Value, sizeof(BufferList[Index]) - 1);
	BufferList[Index][sizeof(BufferList[Index]) - 1] = '\0';
	Tree->Branch(FieldName, (void*) BufferList[Index], "value[64]/C");
    }
    Tree->Fill();

    Tree->Write();
    delete Tree;

    for (int i = 0; i < NumberOfFields; i++) {
	delete[] BufferList[i];
    }
    delete[] BufferList;
}

TKaspNtuple* TKaspRootRepository::LoadNtuple(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::LoadNtuple()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspGraph* TKaspRootRepository::LoadGraph(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::LoadGraph()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspHistogram* TKaspRootRepository::LoadHistogram(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::LoadHistogram()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKasp2dHistogram* TKaspRootRepository::Load2dHistogram(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::Load2dHistogram()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspHistory* TKaspRootRepository::LoadHistory(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::LoadHistory()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspWave* TKaspRootRepository::LoadWave(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::LoadWave()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspMap* TKaspRootRepository::LoadMap(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::LoadMap()",
	"sorry, function not implemented yet"
    );

    return 0;
}

TKaspTabular* TKaspRootRepository::LoadTabular(const string& Name, int Revision) throw(TKaspException)
{
    //... TODO:
    throw TKaspException(
	"TKaspRootRepository::LoadTabular()",
	"sorry, function not implemented yet"
    );

    return 0;
}
