/* KcomComponent.cc */
/* Created by Enomoto Sanshiro on 25 March 2000. */
/* Last updated by Enomoto Sanshiro on 19 May 2003. */


#include <iostream>
#include <fstream>
#include <string>
#include "MushSignal.hh"
#include "MushTimer.hh"
#include "KorbOrb.hh"
#include "KorbObjectReference.hh"
#include "KorbNamingContext.hh"
#include "KcomExchangeCenter.hh"
#include "KcomRegistryOrb.hh"
#include "KcomComponentDescriptor.hh"
#include "KcomTerminal.hh"
#include "KcomConsoleTerminal.hh"
#include "KcomComponent.hh"

using namespace std;

//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KcomComponent", ios::app);
#endif

static const char* PropertyBasePath = "/__properties/";


TKcomComponent::TKcomComponent(const string& TypeName)
{
    _TypeName = TypeName;

    _Orb = 0;

    _InputStream = 0;
    _OutputStream = 0;
    _Terminal = 0;

    _ExchangeCenter = 0;
    _Registry = 0;

    _Descriptor = 0;
}

TKcomComponent::~TKcomComponent()
{
    delete _Descriptor;
    delete _Registry;
    delete _ExchangeCenter;
}

string TKcomComponent::TypeName(void) const
{
    return _TypeName;
}

string TKcomComponent::ComponentName(void) const
{
    return _ComponentName;
}

void TKcomComponent::AttachSystem(const string& ComponentName, TKorbOrb* Orb)
{
    _ComponentName = ComponentName;
    _Orb = Orb;
	
    TKorbObjectReference ObjRef;
    try {
	_ExchangeCenter = new TKcomExchangeCenterProxy(_Orb);
	if (! _ExchangeCenter->Bind("KcomExchangeCenter")) {
	    throw TKcomException(
		"TKcomComponent::Attach()",
		"can't resolve reference to KcomExchangeCenter"
	    );
	}

	_Registry = new TKcomRegistryProxy(_Orb);
	if (! _Registry->Bind("KcomRegistry")) {
	    throw TKcomException(
		"TKcomComponent::Attach()",
		"can't resolve reference to Registry"
	    );
	}
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomComponent::AttachSystem()",
	    "korb exception: " + e.Message()
	);
    }

    _Descriptor = new TKcomComponentDescriptor(_TypeName);
    BuildDescriptor(*_Descriptor);

    ExportObjects(_Descriptor, _Orb->ObjectAdapter());
    ImportObjects(_Descriptor, _ExchangeCenter, _Orb);
    _ExchangeCenter->AttachComponent(_ComponentName);
}

void TKcomComponent::ExportObjects(TKcomComponentDescriptor* Descriptor, TKorbObjectAdapter* ObjectAdapter) throw(TKcomException)
{
    string ObjectName, ExportName;
    TKorbObjectMessenger* Messenger;

    for (unsigned i = 0; i < Descriptor->ExportObjectList().size(); i++) {
	Messenger = Descriptor->ExportObjectList()[i].first;
	ObjectName = Descriptor->ExportObjectList()[i].second.ObjectName();

	ExportName = _ComponentName + "." + ObjectName;
	ObjectAdapter->AddMessenger(Messenger, ExportName);    
    }
}

void TKcomComponent::ImportObjects(TKcomComponentDescriptor* Descriptor, TKcomExchangeCenter* ExchangeCenter, TKorbOrb* Orb) throw(TKcomException)
{
    string ClassName, ObjectName, ImportName;
    TKorbObjectProxy* Proxy;
    TKorbObjectReference ObjectReference;

    for (unsigned i = 0; i < Descriptor->ImportObjectList().size(); i++) {
	Proxy = Descriptor->ImportObjectList()[i].first;
	ClassName = Descriptor->ImportObjectList()[i].second.ClassName();
	ObjectName = Descriptor->ImportObjectList()[i].second.ObjectName();
	
	ImportName = _ComponentName + "." + ObjectName;

	int Result = ExchangeCenter->ImportObject(
	    ClassName, ImportName, ObjectReference
	);
	if (! Result) {
	    throw TKcomException(
		"TKcomComponent::ConnectObjects()",
		"unable to assign remote object: " + ImportName
	    );
	}
	
	Proxy->AttachOrb(_Orb);
	Proxy->Bind(ObjectReference);
    }
}

void TKcomComponent::AttachIO(istream* InputStream, ostream* OutputStream, TKcomTerminal* Terminal)
{
    _InputStream = InputStream;
    _OutputStream = OutputStream;
    _Terminal = Terminal;
}

void TKcomComponent::DetachSystem(void)
{
    _ExchangeCenter->DetachComponent(_ComponentName);

    _Orb = 0;
    _ExchangeCenter = 0;
}

void TKcomComponent::DetachIO(void)
{
    _InputStream = 0;
    _OutputStream = 0;
}

TKcomComponentDescriptor* TKcomComponent::Descriptor(void)
{
    if (_Descriptor == 0) {
	_Descriptor = new TKcomComponentDescriptor(_TypeName);
	BuildDescriptor(*_Descriptor);
    }

    return _Descriptor;
}

int TKcomComponent::DoTransaction(void) throw(TKcomException)
{
    TMushRealTimeTimer(0, 10000).Suspend();
    return 0;
}

void TKcomComponent::OnTransaction(void) throw(TKcomException)
{
}

int TKcomComponent::EmitEvent(int EventId, TKcomEvent& Event) throw(TKcomException)
{
    if (_ExchangeCenter == 0) {
	return 0;
    }

    if (Event.Name().empty()) {
	string EventName = Descriptor()->EventNameOf(EventId);
	if (! EventName.empty()) {
	    Event.Name() = EventName;
	}
    }
    Event.SourceName() = _ComponentName;

    _ExchangeCenter->ProcessEvent(Event);

    return 1;
}

int TKcomComponent::EmitEventOneWay(int EventId, TKcomEvent& Event) throw(TKcomException)
{
    if (_ExchangeCenter == 0) {
	return 0;
    }

    if (Event.Name().empty()) {
	string EventName = Descriptor()->EventNameOf(EventId);
	if (! EventName.empty()) {
	    Event.Name() = EventName;
	}
    }
    Event.SourceName() = _ComponentName;

    _ExchangeCenter->ProcessEventOneWay(Event);

    return 1;
}

int TKcomComponent::EmitEventTo(const string& TargetName, TKcomEvent& Event, TKcomEventResponse& EventResponse) throw(TKcomException)
{
    if (_ExchangeCenter == 0) {
	return 0;
    }

    Event.SourceName() = _ComponentName;
    _ExchangeCenter->EmitEventTo(TargetName, Event, EventResponse);

    return 1;
}

int TKcomComponent::DispatchEvent(TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int EventId = Descriptor()->EventIdOf(Event.Name());

#ifdef DEBUG
    dumpfile << _ComponentName << ": dispatch event: ";
    dumpfile << Event.Name() << " (id = " << EventId << ")" << endl;
#endif

    if (EventId < 0) {
	return 0;
    }
    
    int Result = ProcessEvent(EventId, Event, EventResponse);
    if (Result == 0) {
	Result = ProcessEvent(EventId, Event);

	if (Result) {
	    cerr << "WARNING: KcomComponent: old interface is used: ";
	    cerr << ComponentName() + "." + Event.Name() << "()" << endl;
	}
    }

    return Result;
}

int TKcomComponent::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    return 0;
}

//... backward compatibility ...//
int TKcomComponent::ProcessEvent(int EventId, TKcomEvent& Event)
{
    return 0;
}

bool TKcomComponent::SetProperty(int PropertyId, const string& Value)
{
    string PropertyName = Descriptor()->PropertyNameOf(PropertyId);

    if (! PropertyName.empty()) {
	_Registry->SetValue(
	    PropertyRegistryPathOf(_ComponentName, PropertyName), Value
	);
	return true;
    }
    else {
	return false;
    }
}

bool TKcomComponent::SetLongProperty(int PropertyId, long LongValue)
{
    string PropertyName = Descriptor()->PropertyNameOf(PropertyId);

    if (! PropertyName.empty()) {
	_Registry->SetLongValue(
	    PropertyRegistryPathOf(_ComponentName, PropertyName), LongValue
	);
	return true;
    }
    else {
	return false;
    }
}

bool TKcomComponent::GetPropertyOf(const string& ComponentName, const string& PropertyName, string& Value)
{
    return _Registry->GetValue(
	PropertyRegistryPathOf(ComponentName, PropertyName), Value
    );
}

bool TKcomComponent::GetLongPropertyOf(const string& ComponentName, const string& PropertyName, long& LongValue)
{
    return _Registry->GetLongValue(
	PropertyRegistryPathOf(ComponentName, PropertyName), LongValue
    );
}

const std::string TKcomComponent::PropertyRegistryPathOf(const string& ComponentName, const string& PropertyName)
{
    return string(PropertyBasePath + ComponentName + "/" + PropertyName);
}

TKcomRegistry* TKcomComponent::Registry(void)
{
    return _Registry;
}

istream& TKcomComponent::InputStream(void)
{
    return *_InputStream;
}

ostream& TKcomComponent::OutputStream(void)
{
    return *_OutputStream;
}

TKcomTerminal* TKcomComponent::Terminal(void)
{
    return _Terminal;
}

void TKcomComponent::Terminate(void)
{
    TMushSignalSender::Raise(SIGTERM);
}

