/* KcomExchangeCenter.cc */
/* Created by Enomoto Sanshiro on 27 March 2000. */
/* Last updated by Enomoto Sanshiro on 31 March 2000. */


#include <string>
#include <fstream>
#include "MushSignal.hh"
#include "KorbOrb.hh"
#include "KorbObjectAdapter.hh"
#include "KorbObjectReference.hh"
#include "KorbObjectProxy.hh"
#include "KorbObjectMessenger.hh"
#include "KorbNamingContext.hh"
#include "KcomDefs.hh"
#include "KcomObjectAssembler.hh"
#include "KcomEventProcessor.hh"
#include "KcomExchangeCenter.hh"


using namespace std;


#define VERBOSE
//#define DEBUG
#ifdef DEBUG
//static ofstream dumpfile("Dump-KcomExchangeCenter", ios::app);
#define dumpfile cerr
#endif


static const char* ExchangeCenterClassName = "KcomExchangeCenter";


TKcomExchangeCenter::TKcomExchangeCenter(TKcomObjectAssembler* ObjectAssembler, TKcomEventProcessor* EventProcessor)
{
    _ObjectAssembler = ObjectAssembler;
    _EventProcessor = EventProcessor;
    _NumberOfAttachedComponents = 0;
}

TKcomExchangeCenter::~TKcomExchangeCenter()
{
}

void TKcomExchangeCenter::AttachComponent(const string& ComponentName) throw(TKcomException)
{
#ifdef VERBOSE
    cout << "component \"" << ComponentName << "\" is attached" << endl;
#endif

    _NumberOfAttachedComponents++;
}

void TKcomExchangeCenter::DetachComponent(const string& ComponentName) throw(TKcomException)
{
#ifdef DEBUG
    cout << "component \"" << ComponentName << "\" is detached" << endl;
#endif

    _NumberOfAttachedComponents--;
}

int TKcomExchangeCenter::NumberOfAttachedComponents(void)
{
    return _NumberOfAttachedComponents;
}

int TKcomExchangeCenter::ProcessEvent(TKcomEvent& Event) throw(TKcomException)
{
#ifdef DEBUG
    dumpfile << "process event: ";
    dumpfile << Event.SourceName() << "." << Event.Name() << "(";
    for (unsigned i = 0; i < Event.ArgumentList().size(); i++) {
	dumpfile << "\"" << Event.ArgumentList()[i] << "\", ";
    }
    dumpfile << ")" << endl;
#endif

    return _EventProcessor->ProcessEvent(Event);
}

void TKcomExchangeCenter::EmitEventTo(const string& TargetName, TKcomEvent& Event, TKcomEventResponse& EventResponse) throw(TKcomException)
{
    _EventProcessor->EmitEvent(TargetName, Event, EventResponse);
}

int TKcomExchangeCenter::ImportObject(const string& ClassName, const string& ImportName, TKorbObjectReference& ObjectReference) throw(TKcomException)
{
#ifdef DEBUG
    dumpfile << "import object: " << ImportName << endl;
#endif

    int Result;
    if (! _ObjectAssembler->AssignedObjectOf(ImportName).empty()) {
	Result = _ObjectAssembler->LookupObject(ImportName, ObjectReference);
    }
    else {
	Result = _ObjectAssembler->LookupService(ClassName, ObjectReference);
    }
    
#ifdef VERBOSE
    if (Result > 0) {
	cerr << "assign " << ImportName;
	cerr << " (" << ClassName << ")";
	cerr << " => " << ObjectReference.ObjectName();
	cerr << " (" << ObjectReference.ClassName() << ")" << endl;
    }
#endif
    
    return Result;
}



TKcomExchangeCenterProxy::TKcomExchangeCenterProxy(TKorbOrb* Orb)
: TKorbObjectProxy(ExchangeCenterClassName, Orb), TKcomExchangeCenter(0, 0)
{
}

TKcomExchangeCenterProxy::~TKcomExchangeCenterProxy()
{
}

int TKcomExchangeCenterProxy::ProcessEvent(TKcomEvent& Event) throw(TKcomException)
{
    int MethodId = TKcomExchangeCenterMessenger::MethodId_ProcessEvent;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

#ifdef DEBUG
    dumpfile << "invoke event: ";
    dumpfile << Event.SourceName() << "." << Event.Name() << "(";
    for (unsigned i = 0; i < Event.ArgumentList().size(); i++) {
	dumpfile << "\"" << Event.ArgumentList()[i] << "\", ";
    }
    dumpfile << ")" << endl;
#endif

    try {
        OrbPacket.ArgumentSerializer().PutObject(Event);
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomExchangeCenterProxy::ProcessEvent()",
	    "korb exception: " + e.Message()
        );
    }

    return 1;
}

int TKcomExchangeCenterProxy::ProcessEventOneWay(TKcomEvent& Event) throw(TKcomException)
{
    int MethodId = TKcomExchangeCenterMessenger::MethodId_ProcessEvent;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

#ifdef DEBUG
    dumpfile << "invoke event (one way): ";
    dumpfile << Event.SourceName() << "." << Event.Name() << "(";
    for (unsigned i = 0; i < Event.ArgumentList().size(); i++) {
	dumpfile << "\"" << Event.ArgumentList()[i] << "\", ";
    }
    dumpfile << ")" << endl;
#endif

    try {
        OrbPacket.ArgumentSerializer().PutObject(Event);
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomExchangeCenterProxy::ProcessEventOneWay()",
	    "korb exception: " + e.Message()
        );
    }

    return 1;
}

void TKcomExchangeCenterProxy::EmitEventTo(const string& TargetName, TKcomEvent& Event, TKcomEventResponse& EventResponse) throw(TKcomException)
{
    int MethodId = TKcomExchangeCenterMessenger::MethodId_EmitEventTo;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

#ifdef DEBUG
    dumpfile << "emit event to: ";
    dumpfile << TargetName << "." << Event.Name() << "(";
    for (unsigned i = 0; i < Event.ArgumentList().size(); i++) {
	dumpfile << "\"" << Event.ArgumentList()[i] << "\", ";
    }
    dumpfile << ")" << endl;
#endif

    try {
        OrbPacket.ArgumentSerializer().PutString(TargetName);
        OrbPacket.ArgumentSerializer().PutObject(Event);
	InvokeMethod(OrbPacket);

        OrbPacket.ArgumentSerializer().GetObject(EventResponse);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomExchangeCenterProxy::EmitEventTo()",
	    "korb exception: " + e.Message()
        );
    }
}

int TKcomExchangeCenterProxy::ImportObject(const string& ClassName, const string& ImportName, TKorbObjectReference& ObjectReference) throw(TKcomException)
{
    int Result;

    int MethodId = TKcomExchangeCenterMessenger::MethodId_ImportObject;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    try {
        OrbPacket.ArgumentSerializer().PutString(ClassName);
        OrbPacket.ArgumentSerializer().PutString(ImportName);
	InvokeMethod(OrbPacket);
	
        OrbPacket.ArgumentSerializer().GetInt(Result);
	if (Result > 0) {
	    OrbPacket.ArgumentSerializer().GetObject(ObjectReference);
	}
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomExchangeCenterProxy::ImportObject()",
	    "korb exception: " + e.Message()
        );
    }

    return Result;
}

void TKcomExchangeCenterProxy::AttachComponent(const string& ComponentName) throw(TKcomException)
{
    int MethodId = TKcomExchangeCenterMessenger::MethodId_AttachComponent;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

    try {
        OrbPacket.ArgumentSerializer().PutString(ComponentName);
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomExchangeCenterProxy::AttachComponent()",
	    "korb exception: " + e.Message()
        );
    }
}

void TKcomExchangeCenterProxy::DetachComponent(const string& ComponentName) throw(TKcomException)
{
    int MethodId = TKcomExchangeCenterMessenger::MethodId_DetachComponent;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

    try {
        OrbPacket.ArgumentSerializer().PutString(ComponentName);
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomExchangeCenterProxy::DetachComponent()",
	    "korb exception: " + e.Message()
        );
    }
}



TKcomExchangeCenterMessenger::TKcomExchangeCenterMessenger(TKcomExchangeCenter* ExchangeCenter)
: TKorbObjectMessenger(ExchangeCenterClassName)
{
    _ExchangeCenter = ExchangeCenter;
}

TKcomExchangeCenterMessenger::~TKcomExchangeCenterMessenger()
{
}

int TKcomExchangeCenterMessenger::DispatchMethodInvocation(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (! OrbPacket.IsRequest()) {
	// could be a reply from a 'ghost' component
	return 0;
    }

    if (OrbPacket.MethodId() == MethodId_ProcessEvent) {
	return ProcessEvent(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_EmitEventTo) {
	return EmitEventTo(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_ImportObject) {
	return ImportObject(OrbPacket);
    }
    else if (OrbPacket.MethodId()== MethodId_AttachComponent) {
	return AttachComponent(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_DetachComponent) {
	return DetachComponent(OrbPacket);
    }
    else {
	return 0;
    }
}

int TKcomExchangeCenterMessenger::ProcessEvent(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    TKcomEvent Event;
    OrbPacket.ArgumentSerializer().GetObject(Event);
    
    try {
	_ExchangeCenter->ProcessEvent(Event);
    }
    catch (TKcomException &e)  {
	return 0;
    }

    return 1;
}

int TKcomExchangeCenterMessenger::EmitEventTo(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string TargetName;
    TKcomEvent Event;
    TKcomEventResponse EventResponse;
    OrbPacket.ArgumentSerializer().GetString(TargetName);
    OrbPacket.ArgumentSerializer().GetObject(Event);
    
    try {
	_ExchangeCenter->EmitEventTo(TargetName, Event, EventResponse);
    }
    catch (TKcomException &e)  {
	return 0;
    }

    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutObject(EventResponse);

    return 1;
}

int TKcomExchangeCenterMessenger::ImportObject(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string ClassName, ImportName;
    OrbPacket.ArgumentSerializer().GetString(ClassName);
    OrbPacket.ArgumentSerializer().GetString(ImportName);

    int Result;
    TKorbObjectReference ObjRef;
    try {
	Result = _ExchangeCenter->ImportObject(ClassName, ImportName, ObjRef);
    }
    catch (TKcomException &e)  {
	Result = 0;
    }

    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(Result);
    OrbPacket.ArgumentSerializer().PutObject(ObjRef);	

    return 1;
}

int TKcomExchangeCenterMessenger::AttachComponent(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string ComponentName;
    OrbPacket.ArgumentSerializer().GetString(ComponentName);

    try {
	_ExchangeCenter->AttachComponent(ComponentName);
    }
    catch (TKcomException &e)  {
	return 0;
    }

    return 1;
}

int TKcomExchangeCenterMessenger::DetachComponent(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string ComponentName;
    OrbPacket.ArgumentSerializer().GetString(ComponentName);

    try {
	_ExchangeCenter->DetachComponent(ComponentName);
    }
    catch (TKcomException &e)  {
	return 0;
    }

    return 1;
}
