/* KcomInterfaceDeclaration.cc */
/* Created by Enomoto Sanshiro on 25 March 2000. */
/* Last updated by Enomoto Sanshiro on 31 March 2000. */


#include <string>
#include <vector>
#include "KcomInterfaceDeclaration.hh"

using namespace std;


static const string TypeNameTable[TKcomPropertyDeclaration::NumberOfPropertyTypes] = {
    "int", "long", "float", "double", "string"
};



TKcomComponentDeclaration::TKcomComponentDeclaration(const string& TypeName)
{
    _TypeName = TypeName;

    _MajorVersion = 0;
    _MinorVersion = 0;
}

TKcomComponentDeclaration::~TKcomComponentDeclaration()
{
}

void TKcomComponentDeclaration::Print(ostream& os)
{
    os << "//" << endl;
    os << "// " << _TypeName << ".kidl" << endl;
    os << "//" << endl;

    if (_CommentList.size() > 0) {
        for (unsigned i = 0; i < _CommentList.size(); i++) {
	    os << "// " << _CommentList[i] << endl;
	}
	os << "//" << endl;
    }

    if (_Author.size() > 0) {
        os << "// Author: " << _Author << endl;
    }
    if (_Date.size() > 0) {
        os << "// Date: " << _Date << endl;
    }
    if ((_MajorVersion != 0) || (_MinorVersion != 0)) {
        os << "// Version: " << _MajorVersion << "." << _MinorVersion << endl;
    }
    os << endl;	

    os << "component " << _TypeName << " {" << endl;

    if (_PropertyTable.size() > 0) {
	for (unsigned i = 0; i < _PropertyTable.size(); i++) {
            os << "    property ";
            _PropertyTable[i].Print(os);
	}
    }

    if (_ExportObjectTable.size() > 0) {
	for (unsigned i = 0; i < _ExportObjectTable.size(); i++) {
            os << "    provides ";
            _ExportObjectTable[i].Print(os);
	}
    }

    if (_ImportObjectTable.size() > 0) {
	for (unsigned i = 0; i < _ImportObjectTable.size(); i++) {
            os << "    uses ";
            _ImportObjectTable[i].Print(os);
	}
    }

    if (_EventSourceTable.size() > 0) {
	for (unsigned i = 0; i < _EventSourceTable.size(); i++) {
            os << "    emits ";
            _EventSourceTable[i].Print(os);
	}
    }

    if (_EventSlotTable.size() > 0) {
	for (unsigned i = 0; i < _EventSlotTable.size(); i++) {
            os << "    accepts ";
            _EventSlotTable[i].Print(os);
	}
    }

    os << "}" << endl;

    os << endl;	
}

void TKcomComponentDeclaration::AddComment(const string& Comment)
{
    _CommentList.push_back(Comment);
}

void TKcomComponentDeclaration::SetAuthor(const string& Author)
{
    _Author = Author;
}

void TKcomComponentDeclaration::SetDate(const string& Date)
{
    _Date = Date;
}

void TKcomComponentDeclaration::SetVersion(int MajorVersion, int MinorVersion)
{
    _MajorVersion = MajorVersion;
    _MinorVersion = MinorVersion;
}

void TKcomComponentDeclaration::AddProperty(const TKcomPropertyDeclaration& Property)
{
    _PropertyTable.push_back(Property);
}

void TKcomComponentDeclaration::AddExportObject(const TKcomObjectDeclaration& Object)
{
    _ExportObjectTable.push_back(Object);
}

void TKcomComponentDeclaration::AddImportObject(const TKcomObjectDeclaration& Object)
{
    _ImportObjectTable.push_back(Object);
}

void TKcomComponentDeclaration::AddEventSource(const TKcomEventDeclaration& Event)
{
    _EventSourceTable.push_back(Event);
}

void TKcomComponentDeclaration::AddEventSlot(const TKcomEventDeclaration& Event)
{
    _EventSlotTable.push_back(Event);
}



TKcomPropertyDeclaration::TKcomPropertyDeclaration(void)
{
}

TKcomPropertyDeclaration::TKcomPropertyDeclaration(const string& Name, TKcomPropertyDeclaration::TPropertyType Type)
{
    _Name = Name;
    _Type = Type;
}

TKcomPropertyDeclaration::TKcomPropertyDeclaration(const TKcomPropertyDeclaration& PropertyDeclaration)
{
    _Name = PropertyDeclaration._Name;
    _Type = PropertyDeclaration._Type;
    _Comment = PropertyDeclaration._Comment;
}

TKcomPropertyDeclaration& TKcomPropertyDeclaration::operator=(const TKcomPropertyDeclaration& PropertyDeclaration)
{
    _Name = PropertyDeclaration._Name;
    _Type = PropertyDeclaration._Type;
    _Comment = PropertyDeclaration._Comment;

    return *this;
}

TKcomPropertyDeclaration::~TKcomPropertyDeclaration()
{
}

void TKcomPropertyDeclaration::SetComment(const string& Comment)
{
    _Comment = Comment;
}

string TKcomPropertyDeclaration::Name(void) const
{
    return _Name;
}

TKcomPropertyDeclaration::TPropertyType TKcomPropertyDeclaration::Type(void) const
{
    return _Type;
}

string TKcomPropertyDeclaration::TypeAsString(void) const 
{
    return TypeNameTable[_Type];
}

void TKcomPropertyDeclaration::Print(ostream& os) const
{
    os << TypeNameTable[_Type] << " " << _Name << ";";
    if (_Comment.size() > 0) {
        os << "  //" << _Comment;
    }
    os << endl;
}



TKcomEventDeclaration::TKcomEventDeclaration(void)
{
}

TKcomEventDeclaration::TKcomEventDeclaration(const string Name)
{
    _Name = Name;
}

TKcomEventDeclaration::TKcomEventDeclaration(const TKcomEventDeclaration& EventDeclaration)
{
    _Name = EventDeclaration._Name;
    _ArgumentList = EventDeclaration._ArgumentList;
    _Comment = EventDeclaration._Comment;
}

TKcomEventDeclaration& TKcomEventDeclaration::operator=(const TKcomEventDeclaration& EventDeclaration)
{
    _Name = EventDeclaration._Name;
    _ArgumentList = EventDeclaration._ArgumentList;
    _Comment = EventDeclaration._Comment;

    return *this;
}

TKcomEventDeclaration::~TKcomEventDeclaration()
{
}

void TKcomEventDeclaration::AddArgument(const TKcomPropertyDeclaration& Argument)
{
    _ArgumentList.push_back(Argument);
}

void TKcomEventDeclaration::SetComment(const string& Comment)
{
    _Comment = Comment;
}

string TKcomEventDeclaration::Name(void) const
{
    return _Name;
}

vector<TKcomPropertyDeclaration>& TKcomEventDeclaration::ArgumentList(void)
{
    return _ArgumentList;
}

void TKcomEventDeclaration::Print(ostream& os) const
{
    os << _Name << "(";
    for (unsigned i = 0; i < _ArgumentList.size(); i++) {
        os << _ArgumentList[i].TypeAsString() << " ";
	os << _ArgumentList[i].Name();
	if ((i + 1) < _ArgumentList.size()) {
	  os << ", ";
	}
    }
    os << ");";

    if (_Comment.size() > 0) {
        os << "  // " << _Comment;
    }

    os << endl;
}



TKcomObjectDeclaration::TKcomObjectDeclaration(void)
{
}

TKcomObjectDeclaration::TKcomObjectDeclaration(const string& ClassName, const string& ObjectName)
{
    _ClassName = ClassName;
    _ObjectName = ObjectName;
}

TKcomObjectDeclaration::TKcomObjectDeclaration(const TKcomObjectDeclaration& ObjectDeclaration)
{
    _ClassName = ObjectDeclaration._ClassName;
    _ObjectName = ObjectDeclaration._ObjectName;    
    _Comment = ObjectDeclaration._Comment;
}

TKcomObjectDeclaration& TKcomObjectDeclaration::operator=(const TKcomObjectDeclaration& ObjectDeclaration)
{
    _ClassName = ObjectDeclaration._ClassName;
    _ObjectName = ObjectDeclaration._ObjectName;    
    _Comment = ObjectDeclaration._Comment;

    return *this;
}

TKcomObjectDeclaration::~TKcomObjectDeclaration()
{
}

void TKcomObjectDeclaration::SetComment(const string& Comment)
{
    _Comment = Comment;
}

string TKcomObjectDeclaration::ClassName(void) const
{
    return _ClassName;
}

string TKcomObjectDeclaration::ObjectName(void) const
{
    return _ObjectName;
}

void TKcomObjectDeclaration::Print(ostream& os) const
{
    os << _ClassName << " " << _ObjectName << ";";
    if (_Comment.size() > 0) {
        os << "  //" << _Comment;
    }
    os << endl;
}
