/* KcomManager.cc */
/* Created by Enomoto Sanshiro on 28 March 2000. */
/* Last updated by Enomoto Sanshiro on 1 August 2001. */


#include "MushTimer.hh"
#include "MushProcess.hh"
#include "MushSignal.hh"
#include "MushNetworkSocket.hh"
#include "MushMisc.hh"
#include "KorbOrb.hh"
#include "KorbBroker.hh"
#include "KorbObjectAdapter.hh"
#include "KorbObjectMessenger.hh"
#include "KorbObjectReference.hh"
#include "KorbNamingContext.hh"
#include "KcomRegistry.hh"
#include "KcomRegistryOrb.hh"
#include "KcomComponentAssembler.hh"
#include "KcomObjectAssembler.hh"
#include "KcomEventProcessor.hh"
#include "KcomManager.hh"

using namespace std;


TKcomManager::TKcomManager(void)
{
    _Orb = new TKorbOrb();
    _KorbCommander = new TKorbCommander(_Orb);
    _ObjectAdapter = _Orb->ObjectAdapter();

    _ExchangeCenter = 0;
    _ExchangeCenterMessenger = 0;
    _RegistryProcess = 0;

    _Registry = new TKcomRegistryProxy(_Orb);
}

TKcomManager::~TKcomManager()
{
    delete _Registry;
    delete _ExchangeCenterMessenger;
    delete _ExchangeCenter;

    delete _KorbCommander;
    delete _Orb;
}

void TKcomManager::Initialize(void) throw(TKcomException)
{
    Construct(_KorbCommander, _Orb);

    TKcomComponentAssembler* ComponentAssembler = GetComponentAssembler();
    TKcomObjectAssembler* ObjectAssembler = GetObjectAssembler();
    TKcomEventProcessor* EventProcessor = GetEventProcessor();

    _ExchangeCenterMessenger = new TKcomExchangeCenterMessenger(
        _ExchangeCenter = new TKcomExchangeCenter(
	    ObjectAssembler, EventProcessor
	)
    );

    try {
        LaunchBroker();
	LaunchRegistryServer();

	_ObjectAdapter->AddMessenger(
	    _ExchangeCenterMessenger, "KcomExchangeCenter"
	);

	ComponentAssembler->DeployComponents();
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomManager::Initialize()",
	    "korb exception: " + e.Message()
	);
    }

    int NumberOfComponents = ComponentAssembler->NumberOfComponents();
    int NumberOfTries = 0;
    while (_ExchangeCenter->NumberOfAttachedComponents() < NumberOfComponents) {
	TMushRealTimeTimer(0, 10000).Suspend();
	_ObjectAdapter->DoNowaitTransaction();
	if (++NumberOfTries % 1000 == 0) {
	    cerr << NumberOfComponents - _ExchangeCenter->NumberOfAttachedComponents();
	    cerr << " components are not attached. still waiting..." << endl;
	}
    }

    for (int i = 0; i < NumberOfComponents; i++) {
	EventProcessor->RegisterComponent((*ComponentAssembler)[i].Name());
    }

    OnStartup();
}

void TKcomManager::Finalize(void) throw(TKcomException)
{
    OnShutdown();

    const int TimeOut = 5;
    long StartTime = TMushDateTime::SecSinceEpoch();
    while (_ExchangeCenter->NumberOfAttachedComponents() > 0) {
	TMushRealTimeTimer(0, 10000).Suspend();
	_ObjectAdapter->DoNowaitTransaction();

	if (TMushDateTime::SecSince(StartTime) > TimeOut) {
	    cerr << _ExchangeCenter->NumberOfAttachedComponents();
	    cerr << " components are still running. terminating..." << endl;
	    break;
	}
    }

    TerminateRegistryServer();
    TMushRealTimeTimer(1, 0).Suspend();

    map<unsigned long, int>::iterator DomainIdEntry;
    for (
	DomainIdEntry = _DomainIdTable.begin(); 
	DomainIdEntry != _DomainIdTable.end(); 
	DomainIdEntry++
    ){
	int DomainId = DomainIdEntry->second;
	if (DomainId != TKorbBroker::PrimaryDomainId()) {
	    _KorbCommander->TerminateBroker(DomainId);
	}
    }

    TMushRealTimeTimer(1, 0).Suspend();
}

int TKcomManager::Start(void) throw(TKcomException)
{
    Initialize();

    TMushSignalHandler SignalHandler;
    TMushSignalCounter ExitSignalCounter;

    SignalHandler.RegisterClient(SIGINT, &ExitSignalCounter);
    SignalHandler.RegisterClient(SIGTERM, &ExitSignalCounter);
    SignalHandler.RegisterClient(SIGQUIT, &ExitSignalCounter);
    SignalHandler.StartHandling();

    try {
	while (ExitSignalCounter.SignalCount() == 0) {
	    _ObjectAdapter->DoNowaitTransaction();
	    DoTransaction();
	}
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomManager::Start()", "korb exception: " + e.Message()
	);
    }   

    Finalize();

    return 0;
}

int TKcomManager::DoTransaction(void) throw(TKcomException)
{
    TMushRealTimeTimer(0, 10000).Suspend();
    return 0;
}

int TKcomManager::LaunchBroker(const string& HostName) throw(TKcomException)
{
    string KcomPath;
    try {
	KcomPath = TMushEnvironmentVariable("KCOM_PATH").AsString();
	cout << "KCOM_PATH: " << KcomPath << endl;
    }
    catch (TSystemCallException &e) {
	cout << "KCOM_PATH is not set." << endl;
    }
    cout << "KORB_PROJECT_PATH: " << TKorbOrb::ProjectPath() << endl;
    
    int DomainId;
    if (HostName.empty() || (HostName == "localhost")) {
        _KorbCommander->LaunchBroker(KcomPath);
	DomainId = TKorbBroker::PrimaryDomainId();
    }
    else {
	int ParentDomainId = TKorbBroker::PrimaryDomainId();
	DomainId = _KorbCommander->LaunchRemoteBroker(
	    ParentDomainId, HostName, KcomPath
	);
    }

    return DomainId;
}

int TKcomManager::DomainIdOf(const string& HostName) throw(TKcomException)
{
    if (HostName.empty() || HostName == "localhost") {
	return TKorbBroker::PrimaryDomainId();
    }

    if (_DomainIdTable.empty()) {
	string LocalHostName = TMushLocalHost::HostName();
	unsigned long LocalHostAddress;
	try {
	    LocalHostAddress = TMushNetworkSocket::HostAddressByName(
		LocalHostName
	    );
	}
	catch (TSystemCallException &e) {
	    throw TKcomException(
		"TKcomLaunchar::DomainIdOf()",
		"system call exception: " + e.Message()
	    );
	}

	_DomainIdTable[LocalHostAddress] = TKorbBroker::PrimaryDomainId();
    }

    unsigned long HostAddress;
    try {
	HostAddress = TMushNetworkSocket::HostAddressByName(HostName);
    }
    catch (TSystemCallException &e) {
	throw TKcomException(
	    "TKcomLaunchar::DomainIdOf()",
	    "system call exception: " + e.Message()
	);
    }

    if (_DomainIdTable.count(HostAddress) == 0) {
	_DomainIdTable[HostAddress] = LaunchBroker(HostName);
    }

    return _DomainIdTable[HostAddress];
}

void TKcomManager::LaunchRegistryServer(void) throw(TKcomException)
{
    string Path = "kcom-registry-server";
    vector<string> ArgumentList;
    ArgumentList.push_back(Path);

    try {
	_RegistryProcess = new TMushChildProcess(Path, ArgumentList);
	_RegistryProcess->Run();
    }
    catch (TSystemCallException &e) {
	throw TKcomException(
	    "TKcomManager::LaunchRegistryServer()",
	    "system call exception: " + e.Message()
	);
    }

    string Name = "KcomRegistry";
    TKorbNamingContext* NamingContext = _Orb->NamingContext();    
    TMushRealTimeTimer Timer(0, 100000);
    do {
	Timer.Suspend();
    } while (! NamingContext->LookupObject(Name, _RegistryReference));

    _Registry->Bind(_RegistryReference);
}

void TKcomManager::TerminateRegistryServer(void) throw(TKcomException)
{
    try {
	TMushSignalSender::SendSignal(_RegistryProcess->ProcessId(), SIGTERM);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomManager::TerminateRegistryServer()",
	    "korb exception: " + e.Message()
	);
    }

    delete _RegistryProcess;
    _RegistryProcess = 0;
}

TKcomRegistry* TKcomManager::GetRegistry(void)
{
    return _Registry;
}
