/* KcomProcess.cc */
/* Created by Enomoto Sanshiro on 25 March 2000. */
/* Last updated by Enomoto Sanshiro on 19 May 2003. */


#include <iostream>
#include <fstream>
#include <strstream>
#include <string>
#include "MushSignal.hh"
#include "MushNetworkSocket.hh"
#include "MushSocketStream.hh"
#include "MushArgumentList.hh"
#include "MushMisc.hh"
#include "KorbOrb.hh"
#include "KorbObjectAdapter.hh"
#include "KorbObjectMessenger.hh"
#include "KorbNamingContext.hh"
#include "KcomComponent.hh"
#include "KcomComponentOrb.hh"
#include "KcomEvent.hh"
#include "KcomTerminal.hh"
#include "KcomConsoleTerminal.hh"
#include "KcomTerminalStream.hh"
#include "KcomProcess.hh"

using namespace std;


TKcomProcess::TKcomProcess(TKcomComponent *Component)
{
    _Component = Component;
    _ComponentName = "";
    _ExportsObject = false;

    _Orb = new TKorbOrb();
    _ObjectAdapter = _Orb->ObjectAdapter();

    _ComponentMessenger = 0;

    _InputStream = 0;
    _OutputStream = 0;
    _Terminal = 0;

    _TerminalClient = 0;
    _Socket = 0;
    _LocalTerminal = 0;
}

TKcomProcess::~TKcomProcess()
{
    delete _LocalTerminal;
    delete _Socket;
    delete _TerminalClient;

    delete _InputStream;
    delete _OutputStream;

    delete _ComponentMessenger;
    delete _Orb;
}

void TKcomProcess::Start(TMushArgumentList& ArgumentList) throw(TKcomException)
{
    _ComponentName = ArgumentList[0];
    string Control = ArgumentList[1]; // optional

    if (_ComponentName.empty()) {
	_Component->Descriptor()->Print(cout);
    }
    else {
	try {
	    AttachIO(Control);
	}
	catch (TKcomException &e) {
	    cerr << "ERROR: " << _ComponentName << ": " << e << endl;
	    cerr << "attach null-stream..." << endl;
	    AttachNullIO();
	}

	AttachSystem();

	StartTransaction();

	DetachIO();
	DetachSystem();
    }
}

void TKcomProcess::AttachIO(const string& Control) throw(TKcomException)
{
    if (Control.empty()) {
	AttachConsoleIO();
    }
    else {
	string ControlType;
	istrstream ControlStream(Control.c_str());
	getline(ControlStream, ControlType, ':');

	if (ControlType == "display") {
	    string Display;
	    ControlStream >> Display;
	    AttachTerminalIO(Display);
	}
	else if (ControlType == "port") {
	    int PortNumber;
	    if (ControlStream >> PortNumber) {
		AttachSocketIO(PortNumber);
	    }
	    else {
		throw TKcomException(
		    "TKcomProcess::Start()", "Invalid port number"
		);
	    }
	}
	else if (ControlType == "file") {
	    string FileName;
	    ControlStream >> FileName;
	    AttachFileIO(FileName);
	}
	else if (ControlType == "null") {
	    AttachNullIO();
	}
	else {
	    throw TKcomException(
		"TKcomProcess::Start()", "Invalid control argument"
	    );
	}
    }
}

void TKcomProcess::AttachSocketIO(int PortNumber) throw(TKcomException)
{
    _Socket = new TMushServerNetworkSocket(PortNumber);
    try {
	_Socket->EnableLocalAddressReuse();
	_Socket->Open();
    }
    catch (TSystemCallException &e) {
	throw TKcomException(
	    "TKcomProcess::AttachSocketIO()",
	    "system call exception: " + e.Message()
	);
    }

    _InputStream = new TMushInputSocketStream(_Socket);
    _OutputStream = new TMushOutputSocketStream(_Socket);    
    
    _Component->AttachIO(_InputStream, _OutputStream);
}

void TKcomProcess::AttachTerminalIO(const string& Display) throw(TKcomException)
{
    _TerminalClient = new TKcomConsoleTerminalClient(
	_ComponentName, _Orb, Display
    );
    _Terminal = _TerminalClient->Attach();

    _InputStream = new TKcomInputTerminalStream(_Terminal);
    _OutputStream = new TKcomOutputTerminalStream(_Terminal);    
    
    _Component->AttachIO(_InputStream, _OutputStream, _Terminal);
}

void TKcomProcess::AttachFileIO(const string& FileName) throw(TKcomException)
{
    _InputStream = new ifstream("/dev/null");
    _OutputStream = new ofstream(FileName.c_str(), ios::app);
    
    _Component->AttachIO(_InputStream, _OutputStream);
}

void TKcomProcess::AttachConsoleIO(void) throw(TKcomException)
{
    _LocalTerminal = new TKcomConsoleTerminal();
    _Terminal = _LocalTerminal;

    _InputStream = new TKcomInputTerminalStream(_Terminal);
    _OutputStream = new TKcomOutputTerminalStream(_Terminal);    

    _Component->AttachIO(_InputStream, _OutputStream, _Terminal);
}

void TKcomProcess::AttachNullIO(void) throw(TKcomException)
{
    _InputStream = new ifstream("/dev/null");
    _OutputStream = new ofstream("/dev/null");
    
    _Component->AttachIO(_InputStream, _OutputStream);
}

void TKcomProcess::AttachSystem(void) throw(TKcomException)
{
    try {
	_ComponentMessenger = new TKcomComponentMessenger(_Component);
	_ObjectAdapter->AddMessenger(_ComponentMessenger, _ComponentName);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomProcess::Attach()",
	    "korb exception: " + e.Message()
	);
    }

    _Component->AttachSystem(_ComponentName, _Orb);    
    if (_ObjectAdapter->NumberOfMessengers() > 1) {
	_ExportsObject = true;
    }
}

void TKcomProcess::DetachSystem(void) throw(TKcomException)
{
    try {
	_ObjectAdapter->RemoveMessenger(_ComponentName);
	_Component->DetachSystem();
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomProcess::DetachSystem()",
	    "korb exception: " + e.Message()
	);
    }
}

void TKcomProcess::DetachIO(void) throw(TKcomException)
{
    try {
	if (_Socket != 0) {
	    _Socket->Close();
	}
	if (_TerminalClient != 0) {
	    _TerminalClient->Detach();
	}
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomProcess::DetachIO()",
	    "korb exception: " + e.Message()
	);
    }
}

void TKcomProcess::StartTransaction(void) throw(TKcomException)
{
    TMushSignalHandler SignalHandler;
    TMushSignalCounter DispatchSignalCounter;
    TMushSignalCounter ExitSignalCounter;

    SignalHandler.RegisterClient(SIGURG, &DispatchSignalCounter);
    SignalHandler.RegisterClient(SIGTERM, &ExitSignalCounter);
    SignalHandler.RegisterClient(SIGINT, &ExitSignalCounter);
    SignalHandler.RegisterClient(SIGQUIT, &ExitSignalCounter);
    SignalHandler.StartHandling();

    try {
	_Component->Initialize();

	long LastEventCheckTime = TMushDateTime::SecSinceEpoch();
	static const long EventCheckInterval = 3;

	while (ExitSignalCounter.SignalCount() == 0) {
	    if (DispatchSignalCounter.SignalCount() > 0) {
		DispatchSignalCounter.DecrimentCount();
		_ObjectAdapter->DoTransaction();
	    }
	    else if (
		_ExportsObject || 
		(TMushDateTime::SecSince(LastEventCheckTime) > EventCheckInterval) 
	    ){
		_ObjectAdapter->DoNowaitTransaction();
		LastEventCheckTime = TMushDateTime::SecSinceEpoch();
	    }

	    SignalHandler.StartBlocking();
	    _Component->OnTransaction();
	    _Component->DoTransaction();
	    SignalHandler.StopBlocking();
	}

	_Component->Finalize();
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomProcess::Start()",
	    "korb exception: " + e.Message()
        );
    }
    catch (TKcomException &e) {
	throw;
    }
}
