/* KcomRegistry.cc */
/* Created by Enomoto Sanshiro on 26 May 2000. */
/* Last updated by Enomoto Sanshiro on 19 May 2003. */


#include <iostream>
#include <fstream>
#include <strstream>
#include <cctype>
#include <string>
#include <vector>
#include "KcomRegistryRepository.hh"
#include "KcomRegistry.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KcomRegistry", ios::app);
#endif


static const char EntrySeparator = '/';


TKcomRegistryEntry::TKcomRegistryEntry(const string& Name)
{
    _Name = Name;
}

TKcomRegistryEntry::TKcomRegistryEntry(const string& Name, const string& Value)
{
    _Name = Name;
    _Value = Value;
}

TKcomRegistryEntry::~TKcomRegistryEntry()
{
    for (unsigned i = 0; i < _ChildEntryList.size(); i++) {
	delete _ChildEntryList[i];
    }
}

void TKcomRegistryEntry::SetValue(const string& Value)
{
    _Value = Value;
}

void TKcomRegistryEntry::AddChildEntry(TKcomRegistryEntry* ChildEntry)
{
    _ChildEntryList.push_back(ChildEntry);
}

const string& TKcomRegistryEntry::Name(void) const
{
    return _Name;
}

const string& TKcomRegistryEntry::Value(void) const
{
    return _Value;
}

const vector<TKcomRegistryEntry*>& TKcomRegistryEntry::ChildEntryList(void) const
{
    return _ChildEntryList;
}



TKcomRegistry::TKcomRegistry(void)
{
    _RootEntry = new TKcomRegistryEntry("");
}

TKcomRegistry::~TKcomRegistry()
{
    delete _RootEntry;
}

TKcomRegistryEntry* TKcomRegistry::GetEntry(string RegistryPath, TKcomRegistryEntry* BaseEntry)
{
    if (BaseEntry == 0) {
	BaseEntry = _RootEntry;
    }
    if (*RegistryPath.begin() == EntrySeparator) {
	BaseEntry = _RootEntry;
	RegistryPath.erase(RegistryPath.begin());
    }

    TKcomRegistryEntry* Entry = BaseEntry;

    while (! RegistryPath.empty()) {
	int Length = RegistryPath.find_first_of(EntrySeparator);
	string EntryName = RegistryPath.substr(0, Length);

	const vector<TKcomRegistryEntry*>& ChildEntryList = Entry->ChildEntryList();
	vector<TKcomRegistryEntry*>::const_iterator ChildEntryIterator;
	for (
	    ChildEntryIterator = ChildEntryList.begin(); 
	    ChildEntryIterator != ChildEntryList.end(); 
	    ChildEntryIterator++
	){
	    if ((*ChildEntryIterator)->Name() == EntryName) {
		Entry = *ChildEntryIterator;
		break;
	    }
	}
	if (ChildEntryIterator == ChildEntryList.end()) {
	    TKcomRegistryEntry* ChildEntry = new TKcomRegistryEntry(EntryName);
	    Entry->AddChildEntry(ChildEntry);
	    Entry = ChildEntry;
	}

	RegistryPath.erase(0, Length);
	if (
	    (! RegistryPath.empty()) && 
	    (*RegistryPath.begin() == EntrySeparator)
	){
	    RegistryPath.erase(RegistryPath.begin());
	}
    }

    return Entry;
}

void TKcomRegistry::SetValue(const string& RegistryPath, const string& Value)
{
#ifdef DEBUG
    dumpfile << "set [" << RegistryPath << "] <- " << Value << endl;
#endif

    GetEntry(RegistryPath)->SetValue(Value);
}

bool TKcomRegistry::GetValue(const string& RegistryPath, string& Value)
{
    Value = GetEntry(RegistryPath)->Value();

#ifdef DEBUG
    dumpfile << "get [" << RegistryPath << "] (" << Value << ")" << endl;
#endif

    return true;
}

void TKcomRegistry::SetLongValue(const string& RegistryPath, long Value)
{
    static char Buffer[64];
    ostrstream BufferStream(Buffer, sizeof(Buffer)); 
    BufferStream<< Value << ends;

    SetValue(RegistryPath, string(Buffer));
}

bool TKcomRegistry::GetLongValue(const string& RegistryPath, long &Value)
{
    string StringValue;
    if (GetValue(RegistryPath, StringValue)) {
	if ((StringValue[0] == '0') && (tolower(StringValue[1]) == 'x')) {
	    istrstream ValueStream(StringValue.c_str() + 2);
	    return (ValueStream >> hex >> Value);
	}
	else {
	    istrstream ValueStream(StringValue.c_str());
	    return (ValueStream >> Value);
	}
    }
    else {
	return false;
    }
}

vector<string> TKcomRegistry::ChildEntryNameListOf(const string& RegistryPath)
{
    vector<string> ChildEntryNameList;

    TKcomRegistryEntry* Entry = GetEntry(RegistryPath);
    const vector<TKcomRegistryEntry*>& ChildEntryList = Entry->ChildEntryList();
    vector<TKcomRegistryEntry*>::const_iterator ChildEntryIterator;

    for (
	ChildEntryIterator = ChildEntryList.begin(); 
	ChildEntryIterator != ChildEntryList.end(); 
	ChildEntryIterator++
    ){
	ChildEntryNameList.push_back((*ChildEntryIterator)->Name());
    }
    
    return ChildEntryNameList;
}

int TKcomRegistry::GetSequenceValue(const string& SequenceName)
{
    string EntryPath = "/__sequences/" + SequenceName;

    long CurrentValue;    
    if (! GetLongValue(EntryPath, CurrentValue)) {
	CurrentValue = 0;
    }

    SetLongValue(EntryPath, CurrentValue + 1);

    return CurrentValue;
}

// this method is used by ORB //
void TKcomRegistry::SaveTo(const std::string& FileName, const std::string& FileType, const std::string& BasePath) throw(TKcomException)
{
    if (FileType.empty() || (FileType == "plain")) {
	TKcomRegistryRepository RegistryRepository(FileName);
	RegistryRepository.Save(this, BasePath);
    }
    else if (FileType == "xml") {
	TKcomXmlRegistryRepository RegistryRepository(FileName);
	RegistryRepository.Save(this, BasePath);
    }
    else {
	TKcomRegistryRepository RegistryRepository(FileName);
	RegistryRepository.Save(this, BasePath);
    }
}
