/* KcomRegistryRepository.cc */
/* Created by Enomoto Sanshiro on 27 May 2000. */
/* Last updated by Enomoto Sanshiro on 19 May 2003. */


#include <iostream>
#include <fstream>
#include <string>
#include "KcomRegistry.hh"
#include "KcomRegistryRepository.hh"

using namespace std;


TKcomRegistryRepository::TKcomRegistryRepository(const string& FileName)
{
    _FileName = FileName;
}

TKcomRegistryRepository::~TKcomRegistryRepository()
{
}

void TKcomRegistryRepository::Save(TKcomRegistry* Registry, const string& BasePath) throw(TKcomException)
{
    ofstream File(_FileName.c_str());
    if (! File) {
	throw TKcomException(
	    "TKckomRegistryRepository::Save()", 
	    "unable to open file: " + _FileName
	);
    }

    TKcomRegistryEntry* BaseEntry = Registry->GetEntry(BasePath);
    WriteChildEntries(BaseEntry, File);
}

TKcomRegistry* TKcomRegistryRepository::Load(void) throw(TKcomException)
{
    ifstream File(_FileName.c_str());
    if (! File) {
	throw TKcomException(
	    "TKcomRegistryRepository::Save()", 
	    "unable to open file: " + _FileName
	);
    }

    TParaCxxTokenTable TokenTable;
    TParaTokenizer Tokenizer(File, &TokenTable);

    TKcomRegistry* Registry = new TKcomRegistry();
    TKcomRegistryEntry* RootEntry = Registry->GetEntry("/");

    try {
	ReadChildEntries(RootEntry, Tokenizer);
    }
    catch (TScriptException &e) {
	delete Registry;
	throw TKcomException(
	    "TKcomRegistryRepository::Load()",
	    "script exception: " + e.Message()
	);
    }

    return Registry;
}

void TKcomRegistryRepository::WriteChildEntries(TKcomRegistryEntry* Entry, ostream& os, string Indent)
{
    const vector<TKcomRegistryEntry*>& ChildEntryList = Entry->ChildEntryList();
    for (unsigned i = 0; i < ChildEntryList.size(); i++) {
	TKcomRegistryEntry* ChildEntry = ChildEntryList[i];

	os << Indent << "[" << ChildEntry->Name();
	if (! ChildEntry->Value().empty()) {
	    os << " = \"" << ChildEntry->Value() << "\"";
	}
	os << "]";
	
	if (! ChildEntry->ChildEntryList().empty()) {
	    os << " {" << endl;
	    WriteChildEntries(ChildEntry, os, Indent + "    ");
	    os << Indent << "}";
	    
	    if (Indent.empty()) {
		os << endl;
	    }
	}

	os << endl;
    }    
}

void TKcomRegistryRepository::ReadChildEntries(TKcomRegistryEntry* Entry, TParaTokenizer& Tokenizer) throw(TScriptException)
{
    TParaToken Token;
    Tokenizer.Next().MustBe("[");
    
    (Token = Tokenizer.Next()).MustBe(TParaToken::ttIdentifier);
    string Name = Token.AsString();
    
    string Value;
    if (Tokenizer.LookAhead().IsNot("]")) {
	Tokenizer.Next().MustBe("=");
	Value = Tokenizer.Next().RemoveQuotation('"').AsString();
    }
    
    Tokenizer.Next().MustBe("]");
    
    TKcomRegistryEntry* ChildEntry = new TKcomRegistryEntry(Name, Value);
    Entry->AddChildEntry(ChildEntry);

    if (Tokenizer.LookAhead().Is("{")) {
	Tokenizer.Next().MustBe("{");
	ReadChildEntries(ChildEntry, Tokenizer);
	Tokenizer.Next().MustBe("}");
    }

    if (Tokenizer.LookAhead().Is("[")) {
	ReadChildEntries(Entry, Tokenizer);
    }
}



TKcomXmlRegistryRepository::TKcomXmlRegistryRepository(const string& FileName)
: TKcomRegistryRepository(FileName)
{
}

TKcomXmlRegistryRepository::~TKcomXmlRegistryRepository()
{
}

void TKcomXmlRegistryRepository::Save(TKcomRegistry* Registry, const string& BasePath) throw(TKcomException)
{
    ofstream File(_FileName.c_str());
    if (! File) {
	throw TKcomException(
	    "TKcomXmlRegistryRepository::Save()", 
	    "unable to open file: " + _FileName
	);
    }

    File << "<?xml version=\"1.0\"?>" << endl;
    File << endl;
    File << "<kcom-registry>" << endl;

    TKcomRegistryEntry* BaseEntry = Registry->GetEntry(BasePath);
    WriteChildEntries(BaseEntry, File, "  ");

    File << "</kcom-registry>" << endl;
}

TKcomRegistry* TKcomXmlRegistryRepository::Load(void) throw(TKcomException)
{
    throw TKcomException(
	"TKcomRegistryRepository::Load()",
	"function not implemented yet"
    );

    return 0;
}

void TKcomXmlRegistryRepository::WriteChildEntries(TKcomRegistryEntry* Entry, ostream& os, string Indent)
{
    const vector<TKcomRegistryEntry*>& ChildEntryList = Entry->ChildEntryList();
    for (unsigned i = 0; i < ChildEntryList.size(); i++) {
	TKcomRegistryEntry* ChildEntry = ChildEntryList[i];

	os << Indent << "<" << ChildEntry->Name();
	if (! ChildEntry->Value().empty()) {
	    os << " value=\"" << ChildEntry->Value() << "\"";
	}
	
	if (! ChildEntry->ChildEntryList().empty()) {
	    os << ">" << endl;
	    WriteChildEntries(ChildEntry, os, Indent + "  ");	    
	    os << Indent << "</" << ChildEntry->Name() << ">" << endl;
	}
	else {
	    os << "/>" << endl;
	}
    }    
}

void TKcomXmlRegistryRepository::ReadChildEntries(TKcomRegistryEntry* Entry, TParaTokenizer& Tokenizer) throw(TScriptException)
{
    throw TKcomException(
	"TKcomXmlRegistryRepository::ReadChildEntries()", 
	"function not implemented yet"
    );
}
