/* KcomScript.cc */
/* Created by Enomoto Sanshiro on 8 April 2000. */
/* Last updated by Enomoto Sanshiro on 19 May 2003. */


#include <iostream>
#include <fstream>
#include <strstream>
#include <string>
#include "MushSignal.hh"
#include "KiscScript.hh"
#include "KcomManager.hh"
#include "KcomEvent.hh"
#include "KcomRegistry.hh"
#include "KcomRegistryRepository.hh"
#include "KcomEventProcessor.hh"
#include "KcomComponentAssembler.hh"
#include "KcomScript.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KcomScript", ios::app);
#endif


TKcomScript::TKcomScript(TKcomManager* Manager, int argc, char** argv)
: TKiscScript(argc, argv)
{
    _Manager = Manager;

    _ComponentAssembler = _Manager->GetComponentAssembler();
    _ObjectAssembler = _Manager->GetObjectAssembler();
    _EventProcessor = _Manager->GetEventProcessor();
    _Registry = _Manager->GetRegistry();
}

TKcomScript::~TKcomScript()
{
}

TParaTokenTable* TKcomScript::CreateTokenTable(void)
{
    TParaTokenTable* TokenTable = TKiscScript::CreateTokenTable();
    
    TokenTable->AddKeyword("import");
    TokenTable->AddKeyword("component");
    TokenTable->AddKeyword("assign");
    TokenTable->AddKeyword("asynchronously");
    TokenTable->AddKeyword("oneway");
    TokenTable->AddKeyword("exit");
    TokenTable->AddKeyword("on");
    TokenTable->AddOperator(":=");
    TokenTable->AddOperator("=>");

    return TokenTable;
}

TParaOperatorTable* TKcomScript::CreateOperatorTable(void)
{
    return TKiscScript::CreateOperatorTable();
}

TParaStatementTable* TKcomScript::CreateStatementTable(void)
{
    TParaStatementTable* StatementTable;
    StatementTable = TKiscScript::CreateStatementTable();

    StatementTable->AddStatement(
	new TKcomImportStatement(
	    _ComponentAssembler, _EventProcessor, _Registry
	)
    );
    StatementTable->AddStatement(
	new TKcomAsynchronousStatement(_EventProcessor)
    );
    StatementTable->AddStatement(
	new TKcomOnewayStatement(_EventProcessor)
    );
    StatementTable->AddStatement(new TKcomAssignStatement(_ObjectAssembler));
    StatementTable->AddStatement(new TKcomExitStatement());

    //... for backward compatibility ...//
    StatementTable->AddStatement(new TKcomComponentStatement());

    return StatementTable;
}

TParaPackage* TKcomScript::CreatePackage(void)
{
    TParaPackage* Package = TKiscScript::CreatePackage();

    Package->AddEntry(new TKcomEventHandlerEntry());
    Package->AddEntry(new TKcomTimeHookEntry(this));

    return Package;
}

TParaObjectPrototypeTable* TKcomScript::CreateObjectPrototypeTable(void)
{
    TParaObjectPrototypeTable* ObjectPrototypeTable;
    ObjectPrototypeTable = TKiscScript::CreateObjectPrototypeTable();

    return ObjectPrototypeTable;
}

TParaBuiltinFunctionTable* TKcomScript::CreateBuiltinFunctionTable(void)
{
    TParaBuiltinFunctionTable* BuiltinFunctionTable;
    BuiltinFunctionTable = TKiscScript::CreateBuiltinFunctionTable();

    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKcomSystemScriptMessenger(_Manager)
    );
    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKcomRegistryScriptMessenger(_Registry)
    );

    return BuiltinFunctionTable;
}

int TKcomScript::ExecuteHook(const string& Name) throw(TKcomException)
{
    string HandlerName = TKcomEventHandler::HandlerNameOf(
	TKcomEventHandler::ManagerObjectName(), Name
    );

    if (_Package->GetEntry(HandlerName) == 0) {
	return 0;
    }

    vector<TParaValue*> ArgumentList;
    try {
	_Package->Execute(HandlerName, ArgumentList, _SymbolTable);
    }
    catch (TScriptException &e) {
	throw TKcomException(
	    "TKcomScript::ExecuteHook()",
	    "script exception: " + e.Message()
	);
    }

    return 1;    
}

int TKcomScript::DispatchEvent(TKcomEvent& Event) throw(TKcomException)
{
    string HandlerName = TKcomEventHandler::HandlerNameOf(
	Event.SourceName(), Event.Name()
    );
    if (_Package->GetEntry(HandlerName) == 0) {
	HandlerName = TKcomEventHandler::HandlerNameOf(
	    TKcomEventHandler::AnonymousObjectName(), Event.Name()
	);

	if (_Package->GetEntry(HandlerName) == 0) {
	    return 0;
	}
    }

    int NumberOfArguments = Event.ArgumentList().size();
    TParaValue* ArgumentValueList = 0;
    vector<TParaValue*> ArgumentList;

    if (NumberOfArguments > 0) {
	ArgumentValueList = new TParaValue[NumberOfArguments];
	for (int i = 0; i < NumberOfArguments; i++) {
	    ArgumentValueList[i] = TParaValue(Event.ArgumentList()[i]);
	    ArgumentList.push_back(&ArgumentValueList[i]);
	}
    }

    try {
	_Package->Execute(HandlerName, ArgumentList, _SymbolTable);
    }
    catch (TScriptException &e) {
	throw TKcomException(
	    "TKcomScript::DispatchEvent()",
	    "script exception: " + e.Message()
	);
    }

    delete[] ArgumentValueList;

    return 1;
}

void TKcomScript::AddTimeHook(long Interval, TParaPackageEntry* Entry)
{
    _TimeHookTable.push_back(make_pair(Interval, Entry));
    _NextExecutionTimeTable.push_back(0);
}

int TKcomScript::ExecuteTimeHook(long PresentTime) throw(TKcomException)
{
    int Result = 0;

    for (unsigned i = 0; i < _TimeHookTable.size(); i++) {
	long NextTime = _NextExecutionTimeTable[i];
	if (NextTime <= PresentTime) {
	    TParaPackageEntry* Entry = _TimeHookTable[i].second;
	    vector<TParaValue*> ArgumentList;
	    try {
		Entry->Execute(ArgumentList, _SymbolTable);
	    }
	    catch (TScriptException &e) {
		throw TKcomException(
		    "TKcomScript::ExecuteTimeHook()",
		    "script exception: " + e.Message()
		);
	    }

	    long Interval = _TimeHookTable[i].first;
	    NextTime += Interval * ((PresentTime - NextTime) / Interval + 1);
	    _NextExecutionTimeTable[i] = NextTime;

	    Result++;
	}
    }

    return Result;
}



TKcomEventHandler::TKcomEventHandler(void)
{
}

TKcomEventHandler::~TKcomEventHandler()
{
}

void TKcomEventHandler::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("on");

    TParaToken Token = Tokenizer->Next();
    if (Token.IsIdentifier()) {
	if (Tokenizer->LookAhead().Is(".")) {
	    _ObjectName = Token.AsString();
	    Tokenizer->Next();
	}
	else {
	    _ObjectName = ManagerObjectName();
	    Tokenizer->Unget(Token);
	}
    }
    else if (Token.Is(".")) {
	_ObjectName = AnonymousObjectName();
    }
    else {
        throw TScriptException("event name is expected.");
    }

    Token = Tokenizer->Next();
    if (! Token.IsIdentifier()) {
        throw TScriptException("event name is expected.");
    }
    _EventName = Token.AsString();

    string FunctionName = HandlerNameOf(_ObjectName, _EventName);
    SetName(FunctionName);

    ParseArgumentDeclaration(Tokenizer, StatementParser, SymbolTable);
    
    TParaStatement* Statement = StatementParser->Parse(Tokenizer, SymbolTable);
    SetStatement(Statement);
}

string TKcomEventHandler::HandlerNameOf(const string& ObjectName, const string& EventName)
{
    return ObjectName + "." + EventName;
}

string TKcomEventHandler::AnonymousObjectName(void)
{
    return "[ANONYMOUS]";
}

string TKcomEventHandler::ManagerObjectName(void)
{
    return "[Manager]";
}



TKcomEventHandlerEntry::TKcomEventHandlerEntry(void)
: TParaPackageEntry("event handler")
{
    _EventHandler = 0;
}

TKcomEventHandlerEntry::~TKcomEventHandlerEntry()
{
    delete _EventHandler;
}

TParaPackageEntry* TKcomEventHandlerEntry::Clone(void)
{
    return new TKcomEventHandlerEntry();
}

bool TKcomEventHandlerEntry::HasEntryWordsOf(TParaTokenizer* Tokenizer)
{
    return (
	Tokenizer->LookAhead(1).Is("on") &&
	Tokenizer->LookAhead(2).IsNot("every")
    );
}

void TKcomEventHandlerEntry::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _EventHandler = new TKcomEventHandler();
    _EventHandler->Parse(Tokenizer, StatementParser, SymbolTable);

    string EntryName = _EventHandler->Name();

    SetEntryName(EntryName);
}

TParaValue TKcomEventHandlerEntry::Execute(const vector<TParaValue*>& ArgumentList, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _EventHandler->Execute(ArgumentList, SymbolTable);
}



TKcomTimeHookEntry::TKcomTimeHookEntry(TKcomScript* Script)
: TParaPackageEntry("time hook")
{
    _Script = Script;
    _Statement = 0;
}

TKcomTimeHookEntry::~TKcomTimeHookEntry()
{
    delete _Statement;
}

TParaPackageEntry* TKcomTimeHookEntry::Clone(void)
{
    return new TKcomTimeHookEntry(_Script);
}

bool TKcomTimeHookEntry::HasEntryWordsOf(TParaTokenizer* Tokenizer)
{
    return (
	Tokenizer->LookAhead(1).Is("on") &&
	Tokenizer->LookAhead(2).Is("every")
    );
}

void TKcomTimeHookEntry::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("on");
    Tokenizer->Next().MustBe("every");
    Tokenizer->Next().MustBe("(");

    long Interval = Tokenizer->Next().AsLong();
    if (Tokenizer->LookAhead().Is("sec")) {
	Tokenizer->Next();
    }

    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);

    _Script->AddTimeHook(Interval, this);
}

TParaValue TKcomTimeHookEntry::Execute(const vector<TParaValue*>& ArgumentList, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _Statement->Execute(SymbolTable).ReturnValue;
}



TKcomImportStatement::TKcomImportStatement(TKcomComponentAssembler* ComponentAssembler, TKcomEventProcessor* EventProcessor, TKcomRegistry* Registry)
{
    _ComponentAssembler = ComponentAssembler;
    _EventProcessor = EventProcessor;
    _Registry = Registry;
}

TKcomImportStatement::~TKcomImportStatement()
{
}

TParaStatement* TKcomImportStatement::Clone(void)
{
    return new TKcomImportStatement(
	_ComponentAssembler, _EventProcessor, _Registry
    );
}

string TKcomImportStatement::FirstToken(void) const
{
    return string("import");
}

void TKcomImportStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("import");
    TParaToken ComponentTypeToken = Tokenizer->Next();
    Tokenizer->Next().MustBe(";");

    ComponentTypeToken.MustBe(TParaToken::ttIdentifier);

    SymbolTable->ObjectPrototypeTable()->RegisterClass(
	ComponentTypeToken.AsString(), 
	new TKcomComponentScriptMessenger(
	    _ComponentAssembler, _EventProcessor, _Registry
	)
    );
}

TParaStatement::TExecResult TKcomImportStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return TParaStatement::TExecResult();
}



TKcomAssignStatement::TKcomAssignStatement(TKcomObjectAssembler* ObjectAssembler)
{
    _ObjectAssembler = ObjectAssembler;
}

TKcomAssignStatement::~TKcomAssignStatement()
{
}

TParaStatement* TKcomAssignStatement::Clone(void)
{
    return new TKcomAssignStatement(_ObjectAssembler);
}

string TKcomAssignStatement::FirstToken(void) const
{
    return string("assign");
}

void TKcomAssignStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("assign");

    _UserName = Tokenizer->Next().AsString();
    Tokenizer->Next().MustBe(".");
    _UserObjectName = Tokenizer->Next().AsString();
    
    TParaToken Token = Tokenizer->Next();
    if (Token.IsNot(":=") && Token.IsNot("=>")) {
	Token.MustBe("=");
    }
    
    _ProviderName = Tokenizer->Next().AsString();
    Tokenizer->Next().MustBe(".");
    _ProviderObjectName = Tokenizer->Next().AsString();
    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKcomAssignStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _ObjectAssembler->AddAssignment(
	TKcomAssignment(
	    _ProviderName, _ProviderObjectName, _UserName, _UserObjectName
	)
    );

    return TParaStatement::TExecResult();
}




TKcomAsynchronousStatement::TKcomAsynchronousStatement(TKcomEventProcessor* EventProcessor)
{
    _EventProcessor = EventProcessor;
}

TKcomAsynchronousStatement::~TKcomAsynchronousStatement()
{
}

TParaStatement* TKcomAsynchronousStatement::Clone(void)
{
    return new TKcomAsynchronousStatement(_EventProcessor);
}

string TKcomAsynchronousStatement::FirstToken(void) const
{
    return string("asynchronous");
}

void TKcomAsynchronousStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKcomAsynchronousStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _EventProcessor->EnterAsynchronousMode();
    TExecResult Result = _Statement->Execute(SymbolTable);
    _EventProcessor->ExitAsynchronousMode();

    TKcomEventResponse EventResponse;
    while (_EventProcessor->WaitForNextReply(EventResponse)) {
	if (EventResponse.IsError()) {
#if 0
	    throw TScriptException(EventResponse.ReturnValue());
#else
	    cerr << "ERROR: " << EventResponse.ReturnValue();
	    cerr << " (continue processing)" << endl;
#endif
	}
    }

    return Result;
}



TKcomOnewayStatement::TKcomOnewayStatement(TKcomEventProcessor* EventProcessor)
{
    _EventProcessor = EventProcessor;
}

TKcomOnewayStatement::~TKcomOnewayStatement()
{
}

TParaStatement* TKcomOnewayStatement::Clone(void)
{
    return new TKcomOnewayStatement(_EventProcessor);
}

string TKcomOnewayStatement::FirstToken(void) const
{
    return string("oneway");
}

void TKcomOnewayStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKcomOnewayStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _EventProcessor->EnterOnewayMode();
    TExecResult Result = _Statement->Execute(SymbolTable);
    _EventProcessor->ExitOnewayMode();

    return Result;
}



TKcomExitStatement::TKcomExitStatement(void)
{
}

TKcomExitStatement::~TKcomExitStatement()
{
}

TParaStatement* TKcomExitStatement::Clone(void)
{
    return new TKcomExitStatement();
}

string TKcomExitStatement::FirstToken(void) const
{
    return "exit";
}

void TKcomExitStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("exit");
    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKcomExitStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TMushSignalSender::Raise(SIGTERM);

    TExecResult Result;
    Result.ExecStatus = TParaStatement::esReturn;

    return Result;
}



TKcomComponentStatement::TKcomComponentStatement(void)
{
}

TKcomComponentStatement::~TKcomComponentStatement()
{
}

TParaStatement* TKcomComponentStatement::Clone(void)
{
    return new TKcomComponentStatement();
}

string TKcomComponentStatement::FirstToken(void) const
{
    return "component";
}

void TKcomComponentStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("component");
}

TParaStatement::TExecResult TKcomComponentStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return TExecResult();
}



TKcomSystemScriptMessenger::TKcomSystemScriptMessenger(TKcomManager* Manager)
: TParaObjectPrototype("kcom")
{
    _Manager = Manager;
}

TKcomSystemScriptMessenger::~TKcomSystemScriptMessenger()
{
}

TParaObjectPrototype* TKcomSystemScriptMessenger::Clone(void)
{
    return new TKcomSystemScriptMessenger(_Manager);
}

int TKcomSystemScriptMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (Message == "setTimeout") {
        ReturnValue = SetTimeout(ArgumentList);
    }
    else if (Message == "enableTimeout") {
        ReturnValue = EnableTimeout(ArgumentList);
    }
    else if (Message == "disableTimeout") {
        ReturnValue = DisableTimeout(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKcomSystemScriptMessenger::SetTimeout(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::setTimeout(int)", "too few argument[s]"
	);
    }

    if (! ArgumentList[0]->IsLong() || (ArgumentList[0]->AsLong() <= 0)) {
	throw TScriptException(
	    _InternalClassName + "::setTimeout(int)", "invalid argument[s]"
	);
    }

    int Timeout = ArgumentList[0]->AsLong();
    _Manager->GetEventProcessor()->SetTimeout(Timeout);

    return TParaValue((long) 0);
}

TParaValue TKcomSystemScriptMessenger::EnableTimeout(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    _Manager->GetEventProcessor()->EnableTimeout();

    return TParaValue((long) 0);
}

TParaValue TKcomSystemScriptMessenger::DisableTimeout(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    _Manager->GetEventProcessor()->DisableTimeout();

    return TParaValue((long) 0);
}



TKcomComponentScriptMessenger::TKcomComponentScriptMessenger(TKcomComponentAssembler* ComponentAssembler, TKcomEventProcessor* EventProcessor, TKcomRegistry* Registry)
: TParaObjectPrototype("component")
{
    _ComponentAssembler = ComponentAssembler;
    _EventProcessor = EventProcessor;
    _Registry = Registry; 
}

TKcomComponentScriptMessenger::~TKcomComponentScriptMessenger()
{
}

TParaObjectPrototype* TKcomComponentScriptMessenger::Clone(void)
{
    return new TKcomComponentScriptMessenger(
	_ComponentAssembler, _EventProcessor, _Registry
    );
}

void TKcomComponentScriptMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    _ComponentTypeName = ClassName;
    _ComponentName = this->ObjectName();

    if (ArgumentList.size() > 0) {
	_Host = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	_Control = ArgumentList[1]->AsString();
    }
    else {
	_Control = "null";
    }

    _ComponentAssembler->AddComponent(
	TKcomDeployment(
	    _ComponentName, _ComponentTypeName, _Host, _Control
	)
    );
}

int TKcomComponentScriptMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    TKcomEvent Event(Message);
    Event.SourceName() = "KcomScript";

    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	string Argument = ArgumentList[i]->AsString();
	if (ArgumentList[i]->IsObject()) {
	    Argument = ArgumentList[i]->AsObject()->ObjectName();
	}

	Event.ArgumentList().push_back(Argument);
    }

#ifdef DEBUG
    dumpfile << _ComponentName << ": send event: " << Message << endl;
#endif

    TKcomEventResponse EventResponse;
    try {
	_EventProcessor->EmitEvent(_ComponentName, Event, EventResponse);
    }
    catch (TKcomException &e) {
	throw TScriptException(
	    string("TKcomComponentScriptMessenger::DispatchMessage") +
	    "(\"" + Message + "\")",
	    "kcom exception: " + e.Message()
	);
    }

    if (EventResponse.IsError()) {
#if 0
	throw TScriptException(
	    _ComponentName + "." + Event.Name(), EventResponse.ReturnValue()
	);
#else
	cerr << "ERROR: " << _ComponentName + "." + Event.Name() << "(): ";
	cerr << EventResponse.ReturnValue();
	cerr << " (continue processing)" << endl;
#endif
    }
    else {
	ReturnValue = TParaValue(EventResponse.ReturnValue());
    }

    return 1;
}

int TKcomComponentScriptMessenger::GetPropertyOf(const std::string& PropertyName, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = TParaObjectPrototype::GetPropertyOf(PropertyName, ReturnValue);

    if (Result == 0) {
	string Path = TKcomComponent::PropertyRegistryPathOf(
	    _ComponentName, PropertyName
	);
	string Value;
	if (_Registry->GetValue(Path, Value)) {
	    ReturnValue = TParaValue(Value);
	    Result = 1;
	}
    }

    return Result;
}



TKcomRegistryScriptMessenger::TKcomRegistryScriptMessenger(TKcomRegistry* Registry)
: TParaObjectPrototype("Registry")
{
    _Registry = Registry;
}

TKcomRegistryScriptMessenger::~TKcomRegistryScriptMessenger()
{
}

TParaObjectPrototype* TKcomRegistryScriptMessenger::Clone(void)
{
    return new TKcomRegistryScriptMessenger(_Registry);
}

int TKcomRegistryScriptMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (Message == "getRegistry") {
        ReturnValue = GetRegistry(ArgumentList);
    }
    else if (Message == "setRegistry") {
	ReturnValue = SetRegistry(ArgumentList);
    }
    else if (Message == "saveRegistry") {
	ReturnValue = SaveRegistry(ArgumentList);
    }
    else if (Message == "saveRegistryAsXml") {
	ReturnValue = SaveRegistry(ArgumentList, "xml");
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKcomRegistryScriptMessenger::GetRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::getRegistry()", "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::getRegistry()", "invalid argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value;
    
    _Registry->GetValue(Name, Value);

    return TParaValue(Value);
}

TParaValue TKcomRegistryScriptMessenger::SetRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    _InternalClassName + "::setRegistry()", "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::setRegistry()", "invalid argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    _Registry->SetValue(Name, Value);
    
    return TParaValue(string(""));
}

TParaValue TKcomRegistryScriptMessenger::SaveRegistry(vector<TParaValue*>& ArgumentList, const std::string& FileType) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::" +
            "saveRegistry(string file_name, string base_path=\"/\")", 
	    "too few argument[s]"
	);
    }

    string FileName = ArgumentList[0]->AsString();
    string BasePath = "/";

    if (ArgumentList.size() > 1) {
	BasePath = ArgumentList[1]->AsString();
    }

    _Registry->SaveTo(FileName, FileType, BasePath);
    
    return TParaValue(string(""));
}
