/* KcomScriptedManager.cc */
/* Created by Enomoto Sanshiro on 28 March 2000. */
/* Last updated by Enomoto Sanshiro on 11 April 2000. */


#include <iostream>
#include <fstream>
#include "MushMisc.hh"
#include "KcomEvent.hh"
#include "KcomManager.hh"
#include "KcomScript.hh"
#include "KcomScriptedManager.hh"

using namespace std;


TKcomScriptedManager::TKcomScriptedManager(const string& ScriptFileName, int argc, char** argv)
{
    _ScriptFileName = ScriptFileName;
    _argc = argc;
    _argv = argv;

    _Script = 0;
    _ScriptedEventProcessor = 0;

    _ComponentAssembler = 0;
    _ObjectAssembler = 0;
}

TKcomScriptedManager::~TKcomScriptedManager()
{
    delete _ObjectAssembler;
    delete _ComponentAssembler;
    delete _ScriptedEventProcessor;
    delete _Script;
}

void TKcomScriptedManager::Construct(TKorbCommander* Commander, TKorbOrb* Orb) throw(TKcomException)
{
    ifstream ScriptFile(_ScriptFileName.c_str());
    if (! ScriptFile) {
	throw TKcomException(
	    "TKcomScriptedManager::Construct()",
	    "unable to find file: " + _ScriptFileName
	);
    }

    _ComponentAssembler = new TKcomComponentAssembler(this, Commander, Orb);
    _ObjectAssembler = new TKcomObjectAssembler(Orb);
    _ScriptedEventProcessor = new TKcomScriptedEventProcessor(Orb);

    _Script = new TKcomScript(this, _argc, _argv);

    try {
	_Script->Parse(ScriptFile);
	_Script->Execute();
    }
    catch (TScriptException &e) {
	throw TKcomException(
	    "TKcomScriptedManager::Construct()",
	    "script exception: " + e.Message()
	);
    }
       
    _ScriptedEventProcessor->RegisterScript(_Script);
}

void TKcomScriptedManager::OnStartup(void) throw(TKcomException)
{
    _Script->ExecuteHook("startup");
}

void TKcomScriptedManager::OnShutdown(void) throw(TKcomException)
{
    _Script->ExecuteHook("shutdown");
}

int TKcomScriptedManager::DoTransaction(void) throw(TKcomException)
{
    if (_Script->ExecuteTimeHook(TMushDateTime::SecSinceEpoch())) {
	return 1;
    }
    else {
	return TKcomManager::DoTransaction();
    }
}

TKcomComponentAssembler* TKcomScriptedManager::GetComponentAssembler(void)
{
    return _ComponentAssembler;
}

TKcomObjectAssembler* TKcomScriptedManager::GetObjectAssembler(void)
{
    return _ObjectAssembler;
}

TKcomEventProcessor* TKcomScriptedManager::GetEventProcessor(void)
{
    return _ScriptedEventProcessor;
}



TKcomScriptedEventProcessor::TKcomScriptedEventProcessor(TKorbOrb* Orb)
: TKcomEventProcessor(Orb)
{
    _Script = 0;
}

TKcomScriptedEventProcessor::~TKcomScriptedEventProcessor()
{
}

void TKcomScriptedEventProcessor::RegisterScript(TKcomScript* Script)
{
    _Script = Script;
}

int TKcomScriptedEventProcessor::ProcessEvent(TKcomEvent& Event) throw(TKcomException)
{
#if 0
    Event.Dump(cout);
    cout << " ... " << flush;
#endif

    if (_Script == 0) {
	throw TKcomException(
	    "TKcomScriptedEventProcessor::ProcessEvent()",
	    "sctipt has not been parsed."
	);
    }
    
    try {
#if 1
	_Script->DispatchEvent(Event);
#else
	int Status = _Script->DispatchEvent(Event);
	if (Status  > 0) {
	    cout << "dispatched" << endl;
	}
	else {
	    cout << "ignored" << endl;
	}
#endif
    }
    catch (TKcomException &e) {
	cout << "ERROR: "; 
	Event.Dump(cout);
	cout << ": " << e << " (continue processing)" << endl;
    }

    return 0;
}
