/* dbms-SQLite.cc */
/* Created by Enomoto Sanshiro on 1 October 2002. */
/* Last updated by Enomoto Sanshiro on 1 October 2002. */


#include <string>
#include <vector>
#include <map>
#include <sqlite.h>
#include "KdbcManager.hh"
#include "KdbcDriver.hh"
#include "KdbcResult.hh"
#include "dbms-SQLite.hh"

using namespace std;


static TKdbcDriverCreator Creator("SQLite", new TKdbcDriver_SQLite());


TKdbcDriver_SQLite::TKdbcDriver_SQLite(void)
{
}

TKdbcDriver_SQLite::~TKdbcDriver_SQLite()
{
}

TKdbcDriver* TKdbcDriver_SQLite::Clone(void)
{
    return new TKdbcDriver_SQLite();
}

TKdbcConnection* TKdbcDriver_SQLite::CreateConnection(const string& DatabaseName) throw(TKdbcException)
{
    int Mode = 0;
    char* ErrorMessageBuffer;
    string ErrorMessage;
    sqlite* Sqlite = sqlite_open(
        DatabaseName.c_str(), Mode, &ErrorMessageBuffer
    );

    if (ErrorMessageBuffer != 0) {
        ErrorMessage = ErrorMessageBuffer;
	free(ErrorMessageBuffer);
    }

    if (Sqlite == NULL) {
	throw TKdbcException(
	    "TKdbcDriver_SQLite::CreateConnection()",
	    ErrorMessage + ": " + DatabaseName
	);
    }

    return new TKdbcConnection_SQLite(Sqlite);
}

TKdbcConnection* TKdbcDriver_SQLite::CreateConnection(const string& DatabaseName, const std::string& Host, const std::string& Port, const string& UserName, const string& Password) throw(TKdbcException)
{
    return CreateConnection(DatabaseName);
}



static int QueryCallback(void* pArg, int argc, char** argv, char** columnNames)
{
    TKdbcResult_SQLite* Result = (TKdbcResult_SQLite*) pArg;

    if (Result->NumberOfRows() == 0) {
	Result->SetFieldList(argc, columnNames);
        Result->SetQueryResultFlag();
    }

    Result->AddRow(argv);

    return 0;
}

TKdbcConnection_SQLite::TKdbcConnection_SQLite(sqlite* Sqlite)
{
    _Sqlite = Sqlite;
}

TKdbcConnection_SQLite::~TKdbcConnection_SQLite()
{
    if (_Sqlite != 0) {
	sqlite_close(_Sqlite);
    }
}

void TKdbcConnection_SQLite::Disconnect(void)
{
    if (_Sqlite != 0) {
	sqlite_close(_Sqlite);
	_Sqlite = 0;
    }
}

TKdbcResult* TKdbcConnection_SQLite::ExecuteSql(const string& Statement) throw(TKdbcException)
{
    TKdbcResult_SQLite* Result = new TKdbcResult_SQLite();

    char* ErrorMessageBuffer;
    string ErrorMessage;
    int Status = sqlite_exec(
        _Sqlite, Statement.c_str(), QueryCallback, Result, &ErrorMessageBuffer
    );

    if (ErrorMessageBuffer != 0) {
        ErrorMessage = ErrorMessageBuffer;
	free(ErrorMessageBuffer);
    }

    if (Status != SQLITE_OK) {
	throw TKdbcException(
	    "TKdbcConnection_SQLite::ExecuteSql()",
	    ErrorMessage + ": " + Statement
	);
    }

    if (! Result->IsQueryResult()) {
        int NumberOfAffectedRows = sqlite_changes(_Sqlite);
	Result->SetNumberOfAffectedRows(NumberOfAffectedRows);
    }

    return Result;
}



TKdbcResult_SQLite::TKdbcResult_SQLite(void)
{
    _IsQueryResult = false;
    _NumberOfColumns = 0;
    _NumberOfRows = 0;
    _NumberOfAffectedRows = 0;
}

TKdbcResult_SQLite::~TKdbcResult_SQLite()
{
}

void TKdbcResult_SQLite::Destroy(void)
{
    for (unsigned i = 0; i < _ValueList.size(); i++) {
        delete[] _ValueList[i];
    }

    _ValueList.erase(_ValueList.begin(), _ValueList.end());
    _FieldNameList.erase(_FieldNameList.begin(), _FieldNameList.end());
    _FieldNameTable.erase(_FieldNameTable.begin(), _FieldNameTable.end());
}

string TKdbcResult_SQLite::FieldNameOf(long Column) throw(TKdbcException)
{
    if ((Column >= _NumberOfColumns) || (Column < 0)) {
	throw TKdbcException(
	    "TKdbcResult_SQLite::FieldNameOf()",
	    "invalid column number"
	);
    }

    return _FieldNameList[Column];
}

string TKdbcResult_SQLite::GetValueOf(long Row, long Column) throw(TKdbcException)
{
    if ((Row >= _NumberOfRows) || (Column >= _NumberOfColumns)) {
	throw TKdbcException(
	    "TKdbcResult_SQLite::GetValueOf()",
	    "invalid raw or column number"
	);
    }

    return _ValueList[Row][Column];
}

string TKdbcResult_SQLite::GetValueOf(long Row, const string& FieldName) throw(TKdbcException)
{
    if (_FieldNameTable.count(FieldName)) {
	throw TKdbcException(
	    "TKdbcResult_SQLite::GetValueOf()",
	    "invalid field name: " + FieldName
	);
    }
      
    return GetValueOf(Row, _FieldNameTable[FieldName]);
}

void TKdbcResult_SQLite::SetQueryResultFlag(void)
{
    _IsQueryResult = true;
}

void TKdbcResult_SQLite::SetNumberOfAffectedRows(int NumberOfAffectedRows)
{
    _NumberOfAffectedRows = NumberOfAffectedRows;
}

void TKdbcResult_SQLite::SetFieldList(int NumberOfColumns, char** FieldNameList)
{
    _NumberOfColumns = NumberOfColumns;
    for (int Index = 0; Index < _NumberOfColumns; Index++) {
        string FieldName = FieldNameList[Index];
        _FieldNameList.push_back(FieldName);
	_FieldNameTable[FieldName] = Index;
    }
}

void TKdbcResult_SQLite::AddRow(char** RowData)
{
    string* Row = new string[_NumberOfColumns];
    for (int i = 0; i < _NumberOfColumns; i++) {
        Row[i] = RowData[i];
    }

    _ValueList.push_back(Row);
    _NumberOfRows++;
}
