/* KiscFancyConfigFile.cc */
/* Created by Enomoto Sanshiro on 19 July 2001 */
/* Last updated by Enomoto Sanshiro on 20 July 2001. */


#include <fstream>
#include <string>
#include "ParaTokenizer.hh"
#include "ParaValue.hh"
#include "ParaStatement.hh"
#include "KiscConfigFile.hh"
#include "KiscFancyConfigFile.hh"

using namespace std;


TKiscFancyConfigFile::TKiscFancyConfigFile(const string& FileName)
: TKiscConfigFile(FileName)
{
}

TKiscFancyConfigFile::~TKiscFancyConfigFile()
{
}

void TKiscFancyConfigFile::Construct(void) throw(TScriptException)
{
    ifstream ConfigFile(_FileName.c_str());
    if (! ConfigFile) {
	throw TScriptException(
	    "TKiscFancyConfigFile::Construct()",
	    "unable to open file: " + _FileName
        );
    }

    TKiscFancyConfigFileParser Parser(this);
    Parser.Parse(ConfigFile);
    Parser.Execute();
}

const string& TKiscFancyConfigFile::CurrentSectionName(void)
{
    return _CurrentSectionName;
}

void TKiscFancyConfigFile::SetCurrentSectionName(const string& SectionName)
{
    _CurrentSectionName = SectionName;
}

void TKiscFancyConfigFile::AddEntry(const string& EntryName, const string& Qualifier, const vector<TParaValue>& ValueList)
{
    string EntryPath = _CurrentSectionName + "/" + EntryName + ":" + Qualifier;

    _EntryTable[EntryPath] = ValueList;
}



TKiscFancyConfigFileParser::TKiscFancyConfigFileParser(TKiscFancyConfigFile* ConfigFile)
{
    _ConfigFile = ConfigFile;
}

TKiscFancyConfigFileParser::~TKiscFancyConfigFileParser()
{
}

TParaTokenTable* TKiscFancyConfigFileParser::CreateTokenTable(void)
{
    TParaTokenTable* TokenTable = TKiscScript::CreateTokenTable();

    TokenTable->AddKeyword("section");
    TokenTable->AddKeyword("set");
    TokenTable->AddKeyword("import");
    TokenTable->AddOperator(":=");

    return TokenTable;
}

TParaStatementTable* TKiscFancyConfigFileParser::CreateStatementTable(void)
{
    TParaStatementTable* StatementTable = TKiscScript::CreateStatementTable();

    StatementTable->AddStatement(new TKiscConfigFileSectionStatement(_ConfigFile));
    StatementTable->AddStatement(new TKiscConfigFileSetStatement(_ConfigFile));
    StatementTable->AddStatement(new TKiscConfigFileImportStatement(_ConfigFile));

    return StatementTable;
}



TKiscConfigFileSectionStatement::TKiscConfigFileSectionStatement(TKiscFancyConfigFile* ConfigFile)
{
    _ConfigFile = ConfigFile;
    _Statement = 0;
}

TKiscConfigFileSectionStatement::~TKiscConfigFileSectionStatement()
{
    delete _Statement;
}

TParaStatement* TKiscConfigFileSectionStatement::Clone(void)
{
    return new TKiscConfigFileSectionStatement(_ConfigFile);
}

string TKiscConfigFileSectionStatement::FirstToken(void) const
{
    return string("section");
}

void TKiscConfigFileSectionStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    _SectionName = Tokenizer->Next().RemoveQuotation('"').AsString();
    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKiscConfigFileSectionStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    string NewSectionName;
    string OldSectionName = _ConfigFile->CurrentSectionName();

    if (OldSectionName.empty()) {
	NewSectionName = _SectionName;
    }
    else {
	NewSectionName = OldSectionName + "/" + _SectionName;
    }

    _ConfigFile->SetCurrentSectionName(NewSectionName);
    _Statement->Execute(SymbolTable);
    _ConfigFile->SetCurrentSectionName(OldSectionName);

    return TParaStatement::TExecResult();
}



TKiscConfigFileSetStatement::TKiscConfigFileSetStatement(TKiscFancyConfigFile* ConfigFile)
{
    _ConfigFile = ConfigFile;
}

TKiscConfigFileSetStatement::~TKiscConfigFileSetStatement()
{
    for (unsigned i = 0; i < _ValueExpressionList.size(); i++) {
	delete _ValueExpressionList[i];
    }
}

TParaStatement* TKiscConfigFileSetStatement::Clone(void)
{
    return new TKiscConfigFileSetStatement(_ConfigFile);
}

string TKiscConfigFileSetStatement::FirstToken(void) const
{
    return string("set");
}

void TKiscConfigFileSetStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());
    _EntryName = Tokenizer->Next().RemoveQuotation('"').AsString();

    if (Tokenizer->LookAhead().Is("[")) {
	Tokenizer->Next().MustBe("[");
	_Qualifier = Tokenizer->Next().RemoveQuotation('"').AsString();
	Tokenizer->Next().MustBe("]");
    }

    Tokenizer->Next().MustBe(":=");

    if (Tokenizer->LookAhead().Is("(")) {
	Tokenizer->Next().MustBe("(");
	while (1) {
	    _ValueExpressionList.push_back(
		ExpressionParser->Parse(Tokenizer, SymbolTable)
	    );
	    
	    TParaToken Token = Tokenizer->Next();
	    if (Token.Is(")")) {
		break;
	    }
	    else {
		Token.MustBe(",");
	    }
	}
    }
    else {
	_ValueExpressionList.push_back(
	    ExpressionParser->Parse(Tokenizer, SymbolTable)
	);
    }

    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKiscConfigFileSetStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    vector<TParaValue> ValueList;
    for (unsigned i = 0; i < _ValueExpressionList.size(); i++) {
	TParaValue& Value = _ValueExpressionList[i]->Evaluate(SymbolTable);
	ValueList.push_back(Value);
    }

    _ConfigFile->AddEntry(_EntryName, _Qualifier, ValueList);

    return TParaStatement::TExecResult();
}



TKiscConfigFileImportStatement::TKiscConfigFileImportStatement(TKiscFancyConfigFile* ConfigFile)
{
    _ConfigFile = ConfigFile;
}

TKiscConfigFileImportStatement::~TKiscConfigFileImportStatement()
{
}

TParaStatement* TKiscConfigFileImportStatement::Clone(void)
{
    return new TKiscConfigFileImportStatement(_ConfigFile);
}

string TKiscConfigFileImportStatement::FirstToken(void) const
{
    return string("import");
}

void TKiscConfigFileImportStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("import");
    _ImportedSectionPath = Tokenizer->Next().RemoveQuotation('"').AsString();
    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKiscConfigFileImportStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return TParaStatement::TExecResult();
}
