/* KmlcMatrix.hh */
/* Created by Enomoto Sanshiro on 18 January 2001 */
/* Last updated by Enomoto Sanshiro on 18 January 2001. */


#ifndef __KmlcMatrix_hh__
#define __KmlcMatrix_hh__


#include <complex>


template<class T = double> class TKmlcMatrixBase {
  public:
    inline TKmlcMatrixBase(int NumberOfRows, int NumberOfColumns);
    inline TKmlcMatrixBase(const TKmlcMatrixBase<T>& Matrix);
    inline ~TKmlcMatrixBase();
    inline TKmlcMatrixBase<T>& operator=(const TKmlcMatrixBase<T> Matrix);
    inline int NumberOfRows(void) const;
    inline int NumberOfColumns(void) const;
    inline T* operator[] (int RowIndex);
    inline T* RawArray(void);
    inline T** RawMatrix(void);
  protected:
    int _NumberOfRows, _NumberOfColumns;
    int _Capacity;
    T* _RawArray;
    T** _RawMatrix;
};


typedef TKmlcMatrixBase<double> TKmlcMatrix;
typedef TKmlcMatrixBase<std::complex<double> > TKmlcComplexMatrix;


template<class T> inline TKmlcMatrixBase<T>::TKmlcMatrixBase(int NumberOfRows, int NumberOfColumns)
{
    _NumberOfRows = NumberOfRows;
    _NumberOfColumns = NumberOfColumns;
    
    _Capacity = _NumberOfRows * _NumberOfColumns;
    _RawArray = new T[_Capacity];
    _RawMatrix = 0;
}

template<class T> inline TKmlcMatrixBase<T>::TKmlcMatrixBase(const TKmlcMatrixBase<T>& Matrix)
{
    _NumberOfRows = Matrix._NumberOfRows;
    _NumberOfColumns = Matrix._NumberOfColumns;

    _Capacity = _NumberOfRows * _NumberOfColumns;
    _RawArray = new T[_Capacity];
    _RawMatrix = 0;

    for (unsigned i = 0; i < _NumberOfRows * _NumberOfColumns; i++) {
	_RawArray[i] = Matrix.RawArray[i];
    }
}

template<class T> inline TKmlcMatrixBase<T>::~TKmlcMatrixBase()
{
    delete[] _RawMatrix;
    delete[] _RawArray;
}

template<class T> inline TKmlcMatrixBase<T>& TKmlcMatrixBase<T>::operator=(const TKmlcMatrixBase<T> Matrix)
{
    _NumberOfRows = Matrix._NumberOfRows;
    _NumberOfColumns = Matrix._NumberOfColumns;

    if (_Capacity < _NumberOfRows * _NumberOfColumns) {
	delete[] _RawArray;
	_Capacity = _NumberOfRows * _NumberOfColumns;
	_RawArray = new T[_Capacity];
    }

    delete[] _RawMatrix;
    _RawMatrix = 0;

    for (unsigned i = 0; i < _NumberOfRows * _NumberOfColumns; i++) {
	_RawArray[i] = Matrix.RawArray[i];
    }
}

template<class T> inline int TKmlcMatrixBase<T>::NumberOfRows(void) const
{
    return _NumberOfRows;
}

template<class T> inline int TKmlcMatrixBase<T>::NumberOfColumns(void) const
{
    return _NumberOfColumns;
}

template<class T> inline T* TKmlcMatrixBase<T>::operator[] (int RowIndex)
{
    return _RawArray + RowIndex * _NumberOfColumns;
}

template<class T> inline T* TKmlcMatrixBase<T>::RawArray(void)
{
    return _RawArray;
}

template<class T> inline T** TKmlcMatrixBase<T>::RawMatrix(void)
{
    if (_RawMatrix == 0) {
	_RawMatrix = new (T*)[_NumberOfRows];
	for (int Row = 0; Row < _NumberOfRows; Row++) {
	    _RawMatrix[Row] = _RawArray + Row * _NumberOfColumns;
	}
    }

    return _RawMatrix;
}

#endif
