/* KmlcPrimitive.hh */
/* Created by Enomoto Sanshiro on 26 May 2002. */
/* Last updated by Enomoto Sanshiro on 26 May 2002. */


#ifndef __KmlcPrimitive_hh__
#define __KmlcPrimitive_hh__


#include <string>
#include <complex>
#include "KmlcVector.hh"



template<class T = double> class TKmlcPrimitiveBase {
  public:
    inline TKmlcPrimitiveBase(void);
    inline explicit TKmlcPrimitiveBase(const T& Value);
    inline TKmlcPrimitiveBase(const T& Value, const T& Error);
    inline TKmlcPrimitiveBase(const TKmlcPrimitiveBase<T>& Primitive);
    inline ~TKmlcPrimitiveBase();
    inline TKmlcPrimitiveBase<T>& operator=(const TKmlcPrimitiveBase<T>& Primitive);
    inline TKmlcPrimitiveBase<T>& operator=(const T& Value);
    inline operator T() const;
    inline T& Value(void);
    inline T& Error(void);
    inline std::string& Name(void);
    inline const T& Value(void) const;
    inline const T& Error(void) const;
    inline const std::string& Name(void) const;
    inline T Variance(void) const;
  protected:
    T _Value, _Error;
    std::string _Name;
};


template<int Dimension, class T = double> class TKmlcElementBase {
  public:
    inline TKmlcPrimitiveBase<T>& operator[](int Index);
    inline const TKmlcPrimitiveBase<T>& operator[](int Index) const;
  protected:
    TKmlcPrimitiveBase<T> _PrimitiveArray[Dimension];
};


typedef TKmlcPrimitiveBase<double> TKmlcPrimitive;
typedef TKmlcPrimitiveBase<std::complex<double> > TKmlcComplexPrimitive;

typedef TKmlcElementBase<2, double> TKmlc2dElement;
typedef TKmlcElementBase<2, std::complex<double> > TKmlc2dComplexElement;

typedef TKmlcElementBase<3, double> TKmlc3dElement;
typedef TKmlcElementBase<3, std::complex<double> > TKmlc3dComplexElement;

typedef TKmlcVectorBase<TKmlcPrimitive> TKmlcPrimitiveVector;


template<class T> inline TKmlcPrimitiveBase<T>::TKmlcPrimitiveBase(void)
{
    _Value = 0;
    _Error = 0;
}

template<class T> inline TKmlcPrimitiveBase<T>::TKmlcPrimitiveBase(const T& Value)
{
    _Value = Value;
    _Error = 0;
}

template<class T> inline TKmlcPrimitiveBase<T>::TKmlcPrimitiveBase(const T& Value, const T& Error)
{
    _Value = Value;
    _Error = Error;
}

template<class T> inline TKmlcPrimitiveBase<T>::TKmlcPrimitiveBase(const TKmlcPrimitiveBase<T>& Primitive)
{
    _Value = Primitive._Value;
    _Error = Primitive._Error;
    _Name = Primitive._Name;
}

template<class T> inline TKmlcPrimitiveBase<T>& TKmlcPrimitiveBase<T>::operator=(const TKmlcPrimitiveBase<T>& Primitive)
{
    _Value = Primitive._Value;
    _Error = Primitive._Error;
    _Name = Primitive._Name;

    return *this;
}

template<class T> inline TKmlcPrimitiveBase<T>& TKmlcPrimitiveBase<T>::operator=(const T& Value)
{
    _Value = Value;
    return *this;
}

template<class T> inline TKmlcPrimitiveBase<T>::operator T() const
{
    return _Value;
}

template<class T> inline TKmlcPrimitiveBase<T>::~TKmlcPrimitiveBase()
{
}

template<class T> inline T& TKmlcPrimitiveBase<T>::Value(void)
{
    return _Value;
}

template<class T> inline T& TKmlcPrimitiveBase<T>::Error(void)
{
    return _Error;
}

template<class T> inline std::string& TKmlcPrimitiveBase<T>::Name(void)
{
    return _Name;
}

template<class T> inline const T& TKmlcPrimitiveBase<T>::Value(void) const
{
    return _Value;
}

template<class T> inline const T& TKmlcPrimitiveBase<T>::Error(void) const
{
    return _Error;
}

template<class T> inline T TKmlcPrimitiveBase<T>::Variance(void) const
{
    return _Error * _Error;
}

template<class T> inline const std::string& TKmlcPrimitiveBase<T>::Name(void) const
{
    return _Name;
}


template<int Dimension, class T> inline TKmlcPrimitiveBase<T>& TKmlcElementBase<Dimension, T>::operator[](int Index) 
{ 
    return _PrimitiveArray[Index]; 
}

template<int Dimension, class T> inline const TKmlcPrimitiveBase<T>& TKmlcElementBase<Dimension, T>::operator[](int Index) const
{ 
    return _PrimitiveArray[Index]; 
}


#endif
