/* KorbBroker.cc */
/* Created by Enomoto Sanshiro on 13 February 2000. */
/* Last updated by Enomoto Sanshiro on 29 July 2001. */


#include <fstream>
#include <strstream>
#include <string>
#include <vector>
#include "MushProcess.hh"
#include "MushNetworkSocket.hh"
#include "MushSignal.hh"
#include "MushTimer.hh"
#include "MushFileSystem.hh"
#include "MushMisc.hh"
#include "MushDecoratedSocket.hh"
#include "KiscFancyConfigFile.hh"
#include "KorbNetworkHub.hh"
#include "KorbNetworkAdapter.hh"
#include "KorbNetworkPacket.hh"
#include "KorbOrbPacket.hh"
#include "KorbOrb.hh"
#include "KorbBroker.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbBroker", ios::app);
#endif

//#define DEBUG_SIG
#ifdef DEBUG_SIG
static ofstream sigdumpfile("Dump-KorbBroker-Signal", ios::app);
#endif


static const int MaxNumberOfPortScans = 64;
static const int LaunchTimeOut_sec = 10;
static const int TerminateWait_usec = 100000;


TKorbBroker::TKorbBroker(const string& PathList)
{
    _DomainId = PrimaryDomainId();

    _PathListString = PathList;

    _NameServerProcess = 0;
    _NetworkHub = 0;
}

TKorbBroker::TKorbBroker(int DomainId, int ParentDomainId, const string& ParentHostName, int ParentPortNumber, const string& WorkingDirectory, const string& PathList)
{
    _DomainId = DomainId;
    _ParentDomainId = ParentDomainId;
    _ParentHostName = ParentHostName;
    _ParentPortNumber = ParentPortNumber;

    _WorkingDirectory = WorkingDirectory;
    _PathListString = PathList;

    _NameServerProcess = 0;
    _NetworkHub = 0;
}

TKorbBroker::~TKorbBroker()
{
    for (unsigned i = 0; i < _ChildProcessList.size(); i++) {
	delete _ChildProcessList[i];
    }
    for (unsigned j = 0; j < _SocketList.size(); j++) {
	delete _SocketList[j];
    }

    delete _NameServerProcess;
    delete _NetworkHub;
}

void TKorbBroker::Initialize(void) throw(TKorbException)
{
    if (_NetworkHub != 0) {
	return;
    }

#if 1
    try {
	_RemoteShell = TMushEnvironmentVariable("KINOKO_RSH").AsString();
	_NextPortNumber = TMushEnvironmentVariable("KINOKO_CONTROL_PORT_BASE").AsLong();
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbBroker::Initialize() ",
	    "system call exception: " + e.Message()
	);
    }
#else
    try {
	string ConfigFilePath = TMushEnvironmentVariable("KORB_CONFIG_FILE").AsString();
	TKiscFancyConfigFile ConfigFile(ConfigFilePath);
	ConfigFile.SetTargetSection("korb");
	_RemoteShell = ConfigFile.getValueOf("remote_shell").AsString();
	_NextPortNumber = ConfigFile.getValueOf("port_number").AsLong();
    }
    catch (TSystemCallException &e) {
	throw TKorbException("TKorbBroker::Initialize()", e.Message());
    }
    catch (TScriptException &e) {
	throw TKorbException("TKorbBroker::Initialize()", e.Message());
    }
#endif

    if (_WorkingDirectory.empty()) {
	_WorkingDirectory = TMushFileSystem::CurrentDirectory();
    }
    else {
	try {
	    TMushFileSystem::ChangeDirectory(_WorkingDirectory);
	}
	catch (TSystemCallException &e) {
	    _WorkingDirectory = TMushFileSystem::CurrentDirectory();
	}
    }

    string PathListStringBuffer = _PathListString;
    char Separator = PathListSeparator();
    while (! PathListStringBuffer.empty()) {
	if (PathListStringBuffer[0] == Separator) {
	    PathListStringBuffer.erase(PathListStringBuffer.begin());
	    continue;
	}

	int Length = PathListStringBuffer.find_first_of(Separator);
	string PathEntry = PathListStringBuffer.substr(0, Length);

	_LaunchPathList.push_back(PathEntry);
	PathListStringBuffer.erase(0, Length);
    }

    string ProjectPath = TKorbOrb::ProjectPath();
    int ProjectId = TKorbOrb::ProjectId();
    _NetworkHub = new TKorbNetworkHub(_DomainId, ProjectId, ProjectPath);
    _NetworkHub->Initialize();

    if (_DomainId == PrimaryDomainId()) {
        LaunchNameServer();
    }
    else {
	ConnectParent();
    }
}

void TKorbBroker::Finalize(void) throw(TKorbException)
{
    if (_NameServerProcess != 0) {
	TMushSignalSender::SendSignal(_NameServerProcess->ProcessId(), SIGTERM);
    }

    TMushRealTimeTimer(0, TerminateWait_usec).Suspend();
}

void TKorbBroker::LaunchNameServer(void) throw(TKorbException)
{
    string Path = "korb-nameserver";
    vector<string> ArgumentList;
    ArgumentList.push_back("korb-nameserver");

    _NameServerProcess = new TMushChildProcess(Path, ArgumentList);
    try {
	_NameServerProcess->Run();
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbBroker::LaunchNameServer()",
	    "system call exception: " + e.Message()
	);
    }
}

void TKorbBroker::ConnectParent(void) throw(TKorbException)
{
    TMushNetworkSocket* Socket = 0;
    try {
	Socket = new TMushClientNetworkSocket(_ParentHostName, _ParentPortNumber);
	Socket->Open();
    }
    catch (TSystemCallException &e) {
	delete Socket;
	ostrstream PortNumberStrStream;
	PortNumberStrStream  << _ParentPortNumber << ends;
	throw TKorbException(
            "TKorbBroker::ConnectParent()",
	    "system call exception: " + e.Message() +
            ": " + _ParentHostName + ":" + PortNumberStrStream.str()
	);
    }

    TMushSocket* FramedSocket = new TMushFramedSocket(Socket);
    _NetworkHub->AddConnection(_ParentDomainId, FramedSocket);
    _SocketList.push_back(FramedSocket);
}

void TKorbBroker::Start(void) throw(TKorbException)
{
    TMushSignalHandler *SignalHandler = new TMushSignalHandler();
    TMushSignalCounter *SignalCounter = new TMushSignalCounter();
    SignalHandler->RegisterClient(SIGINT, SignalCounter);
    SignalHandler->RegisterClient(SIGTERM, SignalCounter);
    SignalHandler->RegisterClient(SIGQUIT, SignalCounter);
    SignalHandler->StartHandling();
    
    Initialize();

    while (SignalCounter->SignalCount() == 0) {
	_NetworkHub->DoTransaction();
	
	if (_NetworkHub->Receive(_NetworkPacket) > 0) {
	    TKorbOrbPacket OrbPacket(&_NetworkPacket);
	    ProcessOrbPacket(OrbPacket);
	}
    }

    Finalize();

    delete SignalHandler;
    delete SignalCounter;
}

int TKorbBroker::ProcessOrbPacket(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    OrbPacket.OnReceive();

#ifdef DEBUG
    OrbPacket.Dump(dumpfile);
#endif

    int Result = 0;
    if (OrbPacket.MethodId() == MethodId_LaunchChildBroker) {
	Result = LaunchChildBroker(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_LaunchProcess) {
	Result = LaunchProcess(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_SendSignal) {
	Result = SendSignal(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_SendUrgSignal) {
	Result = SendUrgSignal(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_Exit) {
	Result = Exit(OrbPacket);
    }

    if (Result > 0) {
	if (! OrbPacket.IsOneWay()) {
	    TKorbNetworkPacket* NetworkPacket = OrbPacket.NetworkPacket();
	    int ReceiverDomainId = NetworkPacket->SenderDomainId();
	    int ReceiverAdapterId = NetworkPacket->SenderAdapterId();
	    NetworkPacket->SetSender(_DomainId, this->AdapterId());
	    NetworkPacket->SetReceiver(ReceiverDomainId, ReceiverAdapterId);
	    NetworkPacket->SetDataSize(OrbPacket.PacketSize());
	    OrbPacket.IsRequest() = 0;
	    OrbPacket.OnSend();
	    
#ifdef DEBUG
	    OrbPacket.Dump(dumpfile);
#endif
	    
	    _NetworkHub->Send(*NetworkPacket);
	    _NetworkHub->DoTransaction();
	}
    }
    else {
	// Here we might have to send error status...
#if 0
	throw TKorbException(
	    "TKorbBroker::ProcessOrbPacket()", "invalid method id"
	);
#else
	cerr << "ERROR: KorbBroker: invalid method id" << endl;
	OrbPacket.Dump(cerr);
#endif
    }

    return Result;
}

int TKorbBroker::LaunchChildBroker(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string RemoteHostName;
    int ChildDomainId;
    string ProcessPathList;
    OrbPacket.ArgumentSerializer().GetString(RemoteHostName);
    OrbPacket.ArgumentSerializer().GetInt(ChildDomainId);
    OrbPacket.ArgumentSerializer().GetString(ProcessPathList);

    int PortNumber;
    TMushServerNetworkSocket* Socket = 0;
    for (int i = 0; i < MaxNumberOfPortScans; i++) {
	PortNumber = _NextPortNumber++;
	Socket = new TMushServerNetworkSocket(PortNumber);
	try {
	    Socket->Bind();
	}
	catch (TSystemCallException &e) {
	    delete Socket;
	    Socket = 0;
	    continue;
	}
	break;
    }
    if (Socket == 0) {
	throw TKorbException(
	    "TKorbBroker::LaunchChildBroker()", "unable to bind socket"
	);
    }
    string LocalHostName = Socket->LocalIPAddress();

    ostrstream ParentDomainIdStream;
    ostrstream ChildDomainIdStream;
    ostrstream PortNumberStream;

    ParentDomainIdStream << _DomainId << ends;
    ChildDomainIdStream << ChildDomainId << ends;
    PortNumberStream << PortNumber << ends;

    vector<string> ArgumentList;
    ArgumentList.push_back(_RemoteShell);
    ArgumentList.push_back(RemoteHostName);
    ArgumentList.push_back("korb-broker");
    ArgumentList.push_back(ChildDomainIdStream.str());
    ArgumentList.push_back(ParentDomainIdStream.str());
    ArgumentList.push_back(LocalHostName);
    ArgumentList.push_back(PortNumberStream.str());
    ArgumentList.push_back(_WorkingDirectory);
    ArgumentList.push_back(ProcessPathList);

    TMushChildProcess* Process = new TMushChildProcess(_RemoteShell, ArgumentList);

    int ProcessId = -1;
    string ErrorMessage;
    try {
	Socket->Listen();
	Process->Run();
	ProcessId = Process->ProcessId();

	TMushRealTimeTimer TimeKeeper(LaunchTimeOut_sec, 0);
	TimeKeeper.Start();
	Socket->Accept();
	TimeKeeper.Stop();

	TMushSocket* FramedSocket = new TMushFramedSocket(Socket);
	_SocketList.push_back(FramedSocket);
	_NetworkHub->AddConnection(ChildDomainId, FramedSocket);
	_ChildProcessList.push_back(Process);
    }
    catch (TSystemCallException &e) {
	delete Process; Process = 0;
	delete Socket; Socket = 0;
	ProcessId = -1;

	if (ErrorMessage.empty()) {
	    ErrorMessage = (
		"TKorbBroker::LaunchBroker(): "
		"system call exception: " + e.Message() +
		"\n('interrupted system call' means "
		"'no answer from the remote broker process')"
	    );
	}
    }

    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(ProcessId);
    OrbPacket.ArgumentSerializer().PutString(ErrorMessage);    

    return 1;
}

int TKorbBroker::LaunchProcess(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string Path;
    int NumberOfArguments;
    string Argument;
    vector<string> ArgumentList;

    OrbPacket.ArgumentSerializer().GetString(Path);
    OrbPacket.ArgumentSerializer().GetInt(NumberOfArguments);
    for (int i = 0; i < NumberOfArguments; i++) {
	OrbPacket.ArgumentSerializer().GetString(Argument);
	ArgumentList.push_back(Argument);
    }

    TMushChildProcess* Process;
    int ProcessId = -1;
    string ErrorMessage;
    try {
	if (! _LaunchPathList.empty()) {
	    Process = new TMushChildProcess(_LaunchPathList, Path, ArgumentList);
	}
	else {
	    Process = new TMushChildProcess(Path, ArgumentList);
	}
	Process->Run();

	ProcessId = Process->ProcessId();
	_ChildProcessList.push_back(Process);
    }
    catch (TSystemCallException &e) {
	delete Process;

	ErrorMessage = (
	    "TKorbBroker::LaunchProcess(): "
	    "system call exception: " + e.Message()
	);
    }

    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(ProcessId);
    OrbPacket.ArgumentSerializer().PutString(ErrorMessage);    

    return 1;
}

int TKorbBroker::SendSignal(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    int ProcessId;
    int SignalId;

    OrbPacket.ArgumentSerializer().GetInt(ProcessId);
    OrbPacket.ArgumentSerializer().GetInt(SignalId);

#ifdef DEBUG_SIG
    sigdumpfile << "send signal " << SignalId << " to process " << ProcessId << endl;
#endif
    
    TMushSignalSender::SendSignal(ProcessId, SignalId);

    return 1;
}

int TKorbBroker::SendUrgSignal(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    // the signal ID of SIGURG varies depending on OS.

    int ProcessId;
    int SignalId = SIGURG;

    OrbPacket.ArgumentSerializer().GetInt(ProcessId);

#ifdef DEBUG_SIG
    sigdumpfile << "send signal " << SignalId << " to process " << ProcessId << endl;
#endif
    
    TMushSignalSender::SendSignal(ProcessId, SignalId);

    return 1;
}

int TKorbBroker::Exit(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    TMushSignalSender::Raise(SIGTERM);

    return 1;
}
