/* KorbCommander.cc */
/* Created by Enomoto Sanshiro on 2 March 2000. */
/* Last updated by Enomoto Sanshiro on 2 March 2000. */


#include <unistd.h>
#include "MushProcess.hh"
#include "MushSignal.hh"
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "KorbNetworkAdapter.hh"
#include "KorbNetworkPacket.hh"
#include "KorbOrb.hh"
#include "KorbOrbPacket.hh"
#include "KorbBroker.hh"
#include "KorbCommander.hh"

using namespace std;


TKorbCommander::TKorbCommander(TKorbOrb* Orb)
{
    _Orb = Orb;
    _BrokerProcess = 0;
}

TKorbCommander::~TKorbCommander()
{
    if (_BrokerProcess != 0) {
	if (_BrokerProcess->IsRunning()) {
	    try {
		TerminateBroker();
	    }
	    catch (TKorbException &e) {
		TMushSignalSender::SendSignal(_BrokerProcess->ProcessId(), SIGTERM);
	    }
	}
	_BrokerProcess->WaitToExit();
    }

    delete _BrokerProcess;
}

int TKorbCommander::LaunchBroker(const string& ProcessPathList) throw(TKorbException)
{
    if (_BrokerProcess != 0) {
	return 0;
    }

    string Path = "korb-broker";
    vector<string> ArgumentList;
    ArgumentList.push_back("korb-broker");
    ArgumentList.push_back(ProcessPathList);

    try {
	_BrokerProcess = new TMushChildProcess(Path, ArgumentList);
	_BrokerProcess->Run();
    }
    catch (TSystemCallException &e) {
	delete _BrokerProcess;
	_BrokerProcess = 0;

	throw TKorbException(
	    "TKorbCommander::LaunchBroker()",
	    "system call exception: " + e.Message()
	);
    }

#if 1
    /* wait for the broker process to get launched */
    // The following _Orb->Initialize() waits for the broker
    // by looking for the message queue which will be created
    // by the broker process, but it does not work if an old 
    // message queue is left on the system.
    TMushRealTimeTimer(1, 0).Suspend();
#endif

#if 0
    static const int Provation_sec = 1;
    TMushRealTimeTimer(Provation_sec, 0).Suspend();
    if (! _BrokerProcess->IsRunning()) {
	throw TKorbException(
	    "TKorbCommander::LaunchBroker()",
	    "broker process seems to be crashed."
	);
    }
#endif

    _Orb->Initialize();

    return 1;
}

TKorbOrbPacket TKorbCommander::CreateOrbPacket(int MethodId)
{
    return _Orb->CreateClientOrbPacket(TKorbObjectReference(), MethodId);
}

void TKorbCommander::InvokeBrokerMethod(int DomainId, TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    static const char* ClassName = "KorbBroker";
    static const char* ObjectName = "broker";
    int AdapterId = TKorbBroker::AdapterId();
    int ObjectId = OrbPacket.ObjectId();

    TKorbObjectReference BrokerReference(
	ClassName, ObjectName, 
	DomainId, AdapterId, ObjectId
    );

    _Orb->InvokeMethod(BrokerReference, OrbPacket);
}

int TKorbCommander::LaunchRemoteBroker(int ParentDomainId, const string& HostName, const string& ProcessPathList) throw(TKorbException)
{
    int MethodId = TKorbBroker::MethodId_LaunchChildBroker;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    static int _NextDomainId = 2;
    int ChildDomainId = _NextDomainId++;

    OrbPacket.ArgumentSerializer().PutString(HostName);
    OrbPacket.ArgumentSerializer().PutInt(ChildDomainId);
    OrbPacket.ArgumentSerializer().PutString(ProcessPathList);

    InvokeBrokerMethod(ParentDomainId, OrbPacket);

    int ProcessId;
    OrbPacket.ArgumentSerializer().GetInt(ProcessId);

    if (ProcessId < 0) {
	string Message;
	OrbPacket.ArgumentSerializer().GetString(Message);
	
	throw TKorbException(
	    "TKorbCommander::LaunchRemoteBroker()",
	    Message
	);
    }

    return ChildDomainId;
}

int TKorbCommander::LaunchProcess(int DomainId, const string& Path, const vector<string>& ArgumentList) throw(TKorbException)
{
    int MethodId = TKorbBroker::MethodId_LaunchProcess;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    int NumberOfArguments = ArgumentList.size();
    OrbPacket.ArgumentSerializer().PutString(Path);
    OrbPacket.ArgumentSerializer().PutInt(NumberOfArguments);
    for (int i = 0; i < NumberOfArguments; i++) {
	OrbPacket.ArgumentSerializer().PutString(ArgumentList[i]);
    }

    InvokeBrokerMethod(DomainId, OrbPacket);

    int ProcessId;
    OrbPacket.ArgumentSerializer().GetInt(ProcessId);

    if (ProcessId < 0) {
	string Message;
	OrbPacket.ArgumentSerializer().GetString(Message);
	
	throw TKorbException(
	    "TKorbCommander::LaunchProcess()",
	    Message
	);
    }

    return ProcessId;
}

int TKorbCommander::SendSignal(int DomainId, int ProcessId, int SignalId) throw(TKorbException)
{
    int MethodId = TKorbBroker::MethodId_SendSignal;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    OrbPacket.ArgumentSerializer().PutInt(ProcessId);
    OrbPacket.ArgumentSerializer().PutInt(SignalId);

    InvokeBrokerMethod(DomainId, OrbPacket);

    return 0;
}

int TKorbCommander::TerminateBroker(int DomainId) throw(TKorbException)
{
    int MethodId = TKorbBroker::MethodId_Exit;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

    InvokeBrokerMethod(DomainId, OrbPacket);

    return 0;
}
