/* KorbNamingContext.cc */
/* Created by Enomoto Sanshiro on 25 February 2000. */
/* Last updated by Enomoto Sanshiro on 5 April 2002. */


#include <fstream>
#include <string>
#include <map>
#include "KorbObjectReference.hh"
#include "KorbOrb.hh"
#include "KorbObjectMessenger.hh"
#include "KorbNamingContext.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbNamingContext", ios::app);
#endif


static const int NamingContextDomainId = 1;
static const int NamingContextAdapterId = 2;  // '1' is used by KorbBroker
static const int NamingContextObjectId = 1;
static const char* NamingContextClassName = "KorbNamingContext";
static const char* NamingContextObjectName = "NamingContext";


TKorbNamingContext::TKorbNamingContext(void)
{
}

TKorbNamingContext::~TKorbNamingContext(void)
{
}

TKorbObjectReference TKorbNamingContext::GetInitialReference(void)
{
    return TKorbObjectReference(
	NamingContextClassName, NamingContextObjectName, 
	NamingContextDomainId, NamingContextAdapterId, NamingContextObjectId
    );
}

void TKorbNamingContext::RegisterObject(const TKorbObjectReference& ObjectReference)
{
    string ObjectName = ObjectReference.ObjectName();
    _ObjectTable[ObjectName] = ObjectReference;

#ifdef DEBUG
    string ClassName = ObjectReference.ClassName();
    dumpfile << "register [" << ObjectName << "(" << ClassName << ")]";
    dumpfile << ObjectReference.DomainId() << ".";
    dumpfile << ObjectReference.AdapterId() << ".";
    dumpfile << ObjectReference.ObjectId() << endl;
#endif
}

void TKorbNamingContext::UnregisterObject(const string& ObjectName)
{
    if (_ObjectTable.count(ObjectName) > 0) {
	_ObjectTable.erase(ObjectName);
    }	

#ifdef DEBUG
    dumpfile << "unregister [" << ObjectName << "]" << endl;
#endif
}

int TKorbNamingContext::LookupObject(const string& ObjectName, TKorbObjectReference& ObjectReference)
{
    if (_ObjectTable.count(ObjectName) == 0) {
	return 0;
    }
    ObjectReference = _ObjectTable[ObjectName];
    
    return 1;
}

int TKorbNamingContext::LookupService(const string& ClassName, TKorbObjectReference& ObjectReference)
{
    map<string, TKorbObjectReference>::iterator ObjectReferenceIterator;
    for (
	ObjectReferenceIterator = _ObjectTable.begin(); 
	ObjectReferenceIterator != _ObjectTable.end(); 
	ObjectReferenceIterator++
    ){
	if ((*ObjectReferenceIterator).second.ClassName() == ClassName) {
	    ObjectReference = (*ObjectReferenceIterator).second;
	    return 1;
	}
    }

    return 0;
}



TKorbNamingContextProxy::TKorbNamingContextProxy(TKorbOrb* Orb)
: TKorbObjectProxy(NamingContextClassName, Orb)
{
}

TKorbNamingContextProxy::~TKorbNamingContextProxy()
{
}

void TKorbNamingContextProxy::RegisterObject(const TKorbObjectReference& ObjectReference)
{
    int MethodId = TKorbNamingContextMessenger::MethodId_RegisterObject;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

    OrbPacket.ArgumentSerializer().PutObject(ObjectReference);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw e;
    }
}

void TKorbNamingContextProxy::UnregisterObject(const string& ObjectName)
{
    int MethodId = TKorbNamingContextMessenger::MethodId_UnregisterObject;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

    OrbPacket.ArgumentSerializer().PutString(ObjectName);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw e;
    }
}

int TKorbNamingContextProxy::LookupObject(const string& ObjectName, TKorbObjectReference& ObjectReference)
{
    int MethodId = TKorbNamingContextMessenger::MethodId_LookupObject;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    OrbPacket.ArgumentSerializer().PutString(ObjectName);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw e;
    }

    int Result;
    OrbPacket.ArgumentSerializer().GetInt(Result);
    OrbPacket.ArgumentSerializer().GetObject(ObjectReference);

    return Result;
}

int TKorbNamingContextProxy::LookupService(const string& ClassName, TKorbObjectReference& ObjectReference)
{
    int MethodId = TKorbNamingContextMessenger::MethodId_LookupService;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    OrbPacket.ArgumentSerializer().PutString(ClassName);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException &e) {
	throw e;
    }

    int Result;
    OrbPacket.ArgumentSerializer().GetInt(Result);
    OrbPacket.ArgumentSerializer().GetObject(ObjectReference);

    return Result;
}



TKorbNamingContextMessenger::TKorbNamingContextMessenger(TKorbNamingContext* NamingContext)
: TKorbObjectMessenger("NamingContext")
{
    _NamingContext = NamingContext;
}

TKorbNamingContextMessenger::~TKorbNamingContextMessenger()
{
}

int TKorbNamingContextMessenger::DispatchMethodInvocation(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (OrbPacket.MethodId() == MethodId_RegisterObject) {
	return RegisterObject(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_UnregisterObject) {
	return UnregisterObject(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_LookupObject) {
	return LookupObject(OrbPacket);
    }
    else if (OrbPacket.MethodId() == MethodId_LookupService) {
	return LookupService(OrbPacket);
    }

    return 0;
}

int TKorbNamingContextMessenger::RegisterObject(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    TKorbObjectReference ObjectReference;
    OrbPacket.ArgumentSerializer().GetObject(ObjectReference);
    
    _NamingContext->RegisterObject(ObjectReference);
    
    return 1;
}

int TKorbNamingContextMessenger::UnregisterObject(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string ObjectName;
    OrbPacket.ArgumentSerializer().GetString(ObjectName);
    
    _NamingContext->UnregisterObject(ObjectName);
    
    return 1;
}

int TKorbNamingContextMessenger::LookupObject(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string ObjectName; 
    TKorbObjectReference ObjectReference;
    
    OrbPacket.ArgumentSerializer().GetString(ObjectName);
    
#ifdef DEBUG
    dumpfile << "query " << OrbPacket.RequestId() << ": " << ObjectName << "... " << flush;
#endif

    int Result = _NamingContext->LookupObject(ObjectName, ObjectReference);
    
#ifdef DEBUG
    if (Result > 0) {
        dumpfile << ObjectReference.DomainId() << ".";
	dumpfile << ObjectReference.AdapterId() << ".";
	dumpfile << ObjectReference.ObjectId() << endl;
    }
    else {
	dumpfile << "not found" << endl;
    }
#endif

    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(Result);
    OrbPacket.ArgumentSerializer().PutObject(ObjectReference);
    
    return 1;
}

int TKorbNamingContextMessenger::LookupService(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    string ClassName; 
    TKorbObjectReference ObjectReference;
    
    OrbPacket.ArgumentSerializer().GetString(ClassName);
    int Result = _NamingContext->LookupService(ClassName, ObjectReference);

    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(Result);
    OrbPacket.ArgumentSerializer().PutObject(ObjectReference);
    
    return 1;
}
