/* KorbNetworkHub.cc */
/* Created by Enomoto Sanshiro on 4 December 1999. */
/* Last updated by Enomoto Sanshiro on 7 February 2000. */


#include <iostream>
#include <fstream>
#include "MushTimer.hh"
#include "KorbNetworkHub.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbNetworkHub", ios::app);
#endif


TKorbNetworkHub::TKorbNetworkHub(int DomainId, int MessageQueueProjectId, const string& KorbPath)
{
    _DomainId = DomainId;
    
    _MessageQueueProjectId = MessageQueueProjectId;
    _KorbPath = KorbPath;

    _RoutingTable = new TKorbNetworkRoutingTable(this);
    _SocketSelector = new TMushSocketSelector();

    _MessageQueue = 0;
}

TKorbNetworkHub::~TKorbNetworkHub()
{
    delete _MessageQueue;
    delete _SocketSelector;
    delete _RoutingTable;
}

void TKorbNetworkHub::Initialize(void) throw(TKorbException)
{
    try {
	unsigned AccessPermission = 0600;
	bool IsSelfish = true;
	_MessageQueue = new TMushServerMessageQueue(
	    _MessageQueueProjectId, _KorbPath, AccessPermission, IsSelfish
	); 
	_MessageQueue->SetAddress(MessageQueueAddress);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::Initialize()",
	    "system call exception: " + e.Message()
	);
    }
}

int TKorbNetworkHub::DomainId(void) const
{
    return _DomainId;
}

int TKorbNetworkHub::SendToUplinkPort(int UplinkPortId, TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int SendLength = 0;

    if (_SocketTable.count(UplinkPortId) == 0) {
	throw TKorbException(
	    "TKorbNetworkHub::SendToUplinkPort()",
	    "bad uplink port id"
	);
    }

    try {
	_SocketTable[UplinkPortId]->Send(
	    Packet.ContentsAddress(), 
	    Packet.ContentsSize()
	);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::SendToUplinkPort()",
	    "system call exception: " + e.Message()
	);
    }

#ifdef DEBUG
    dumpfile << "Send to Uplink Port " << UplinkPortId << endl;
    Packet.Dump(dumpfile);
#endif
    
    return SendLength;
}

int TKorbNetworkHub::SendToLocalPort(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    TMushMessageQueuePacket* MessageQueuePacket
	= (TMushMessageQueuePacket*) Packet.ContentsAddress();
    MessageQueuePacket->MessageType = Packet.ReceiverAdapterId();
    int MessageLength = Packet.ContentsSize();

    int SentLength;
    try {
	SentLength = _MessageQueue->Send(MessageQueuePacket, MessageLength);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::SendToLocalPort()",
	    "system call exception: " + e.Message()
	);
    }
    if (SentLength != MessageLength) {
        throw TKorbException(
	    "TKorbNetworkAdapter::Send()",
	    "system call exception: unable to send message"
        );
    }

#ifdef DEBUG
    dumpfile << "Send to Local Port" << endl;
    Packet.Dump(dumpfile);
#endif

    return SentLength;
}

int TKorbNetworkHub::ReceiveFromUplinkPort(TKorbNetworkPacket& Packet, int& UplinkPortId) throw(TKorbException)
{
    int ReceivedLength = 0;
    static const int TimeOut_ms = 50;

    if (_SocketTable.empty()) {
	TMushRealTimeTimer(0, TimeOut_ms * 1000).Suspend();
	return ReceivedLength = 0;
    }

    pair<TMushSocket*, TMushSocketSelector::TSocketEvent> SocketEvent;
    try {
	SocketEvent = _SocketSelector->WaitEvent(TimeOut_ms);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::ReceiveFromUplinkPort()",
	    "system call exception: " + e.Message()
	);
    }

    TMushSocket* Socket = SocketEvent.first;
    if (Socket == 0) {
	return ReceivedLength = 0;
    }
    else if (SocketEvent.second == TMushSocketSelector::SocketEvent_HungUp) {
	RemoveConnection(Socket);
	return ReceivedLength = 0;
    }
    
    UplinkPortId = Socket->FileDescriptor();
    try {
	ReceivedLength = Socket->Receive(
	    Packet.ContentsAddress(), 
	    Packet.ContentsCapacity()
	);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::ReceiveFromUplinkPort()",
	    "system call exception: " + e.Message()
	);
    }
    if (ReceivedLength <= 0) {
	return 0;
    }

    if (Packet.IsByteOrderReversed()) {
	Packet.ReverseByteOrder();
#ifdef DEBUG
	dumpfile << "[byte order was reversed]" << endl;
#endif
    }
    
#ifdef DEBUG
    dumpfile << "Receive from Uplink Port " << UplinkPortId << endl;
    Packet.Dump(dumpfile);
#endif
    
    return ReceivedLength;
}

int TKorbNetworkHub::ReceiveFromLocalPort(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int ReceivedLength = 0;

    TMushMessageQueuePacket* MessageQueuePacket
        = (TMushMessageQueuePacket*) Packet.ContentsAddress();
    int MaxMessageLength = Packet.ContentsCapacity() - sizeof(long);
    try {
	bool NoWait = true;
	ReceivedLength = _MessageQueue->Receive(
	    MessageQueuePacket, MaxMessageLength, NoWait
	);
    }
    catch (TSystemCallException &e) {
        throw TKorbException(
	    "TKorbNetworkHub::ReceiveFromLocalPort()",
	    "system call exception: " + e.Message()
        );
    }

    if (Packet.SenderDomainId() == DomainId_Unknown) {
	Packet.SetSender(_DomainId, Packet.SenderAdapterId());
    }

#ifdef DEBUG
    if (ReceivedLength) {
	dumpfile << "Receive from Local Port " << endl;
	Packet.Dump(dumpfile);
    }
#endif

    return ReceivedLength;
}

void TKorbNetworkHub::AddConnection(int RemoteDomainId, TMushSocket* Socket)
{
    int UplinkPortId = Socket->FileDescriptor();

    _SocketTable[UplinkPortId] = Socket;
    _SocketSelector->AddSocket(Socket);
    _RoutingTable->AddConnection(UplinkPortId, RemoteDomainId);

#ifdef DEBUG
    dumpfile << "Add new connection" << endl;
    dumpfile << "  Uplink Port Id: " << UplinkPortId << endl;
    dumpfile << "  Remote Domain Id: " << RemoteDomainId << endl;
    dumpfile << endl;
#endif
}

void TKorbNetworkHub::RemoveConnection(TMushSocket* Socket)
{
    int UplinkPortId = Socket->FileDescriptor();

    _SocketTable.erase(UplinkPortId);
    _SocketSelector->RemoveSocket(Socket);
    //_RoutingTable->RemoveConnection(UplinkPortId);

#ifdef DEBUG
    dumpfile << "Remove connection" << endl;
    dumpfile << "  Uplink Port Id: " << UplinkPortId << endl;
    dumpfile << endl;
#endif
}

int TKorbNetworkHub::Send(TKorbNetworkPacket& Packet)
{
    _WaitingQueue.push_back(Packet);

    return 1;
}

int TKorbNetworkHub::Receive(TKorbNetworkPacket& Packet)
{
    if (_ReceivedPacketQueue.size() > 0) {
	Packet = _ReceivedPacketQueue.front();
	_ReceivedPacketQueue.pop_front();

	return 1;
    }

    return 0;
}

int TKorbNetworkHub::DoTransaction(void) throw(TKorbException)
{
    if (_MessageQueue == 0) {
	Initialize();
    }

    int NumberOfTransactions = 0;
    int UplinkPortId;

    if (ReceiveFromUplinkPort(_Packet, UplinkPortId) > 0) {
	ProcessPacket(_Packet, UplinkPortId);
	NumberOfTransactions++;
    }

    if (_WaitingQueue.size() > 0) {
	ProcessPacket(_WaitingQueue.front());
	_WaitingQueue.pop_front();
	NumberOfTransactions++;
    }

    if (ReceiveFromLocalPort(_Packet) > 0) {
	ProcessPacket(_Packet, UplinkPortId);
	NumberOfTransactions++;
    }
	
    return NumberOfTransactions;
}

int TKorbNetworkHub::ProcessPacket(TKorbNetworkPacket& Packet, int UplinkPortId) throw(TKorbException)
{
    int Result = 0;
    switch (Packet.PacketType()) {
      case TKorbNetworkPacket::ptRoutingQuery:
      case TKorbNetworkPacket::ptRoutingReply:
	Result = _RoutingTable->ProcessRoutingPacket(Packet, UplinkPortId);
	break;
      case TKorbNetworkPacket::ptEchoRequest:
	Result = ProcessEchoRequestPacket(Packet);
	break;
      case TKorbNetworkPacket::ptEchoReply:
      case TKorbNetworkPacket::ptMessage:
      default:
	Result = ProcessMessagePacket(Packet);
    }

    return Result;
}

int TKorbNetworkHub::ProcessMessagePacket(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int RemoteDomainId = Packet.ReceiverDomainId();

    if (RemoteDomainId == _DomainId) {
	if (Packet.ReceiverAdapterId() == _AdapterId) {
	    _ReceivedPacketQueue.push_back(Packet);
	}
	else {
	    SendToLocalPort(Packet);
	}
    }
    else {
        int UplinkPortId = _RoutingTable->LookupRouteTo(RemoteDomainId);
	if (UplinkPortId >= 0) {
	    SendToUplinkPort(UplinkPortId, Packet);
	}
	else if (UplinkPortId == TKorbNetworkRoutingTable::lrSearching) {
	    _WaitingQueue.push_back(Packet);
	}
	else {
	    Packet.SetDeliverStatus(TKorbNetworkPacket::dsNoRoute);
	    Packet.SetReceiver(
		Packet.SenderDomainId(), Packet.SenderAdapterId()
	    );
	}
    }

    return 0;
}

int TKorbNetworkHub::ProcessEchoRequestPacket(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int RemoteDomainId = Packet.ReceiverDomainId();
    if (RemoteDomainId == DomainId_Unknown) {
	RemoteDomainId = _DomainId;
    }
    
    if (RemoteDomainId == _DomainId) {
	int ReceiverDomianId = Packet.SenderDomainId();
	int ReceiverAdapterId = Packet.SenderAdapterId();
	
	Packet.SetPacketType(TKorbNetworkPacket::ptEchoReply);
	Packet.SetSender(_DomainId, 0);
	Packet.SetReceiver(ReceiverDomianId, ReceiverAdapterId);
    }

    return ProcessMessagePacket(Packet);
}
