/* KorbNetworkPacket.cc */
/* Created by Enomoto Sanshiro on 4 December 1999. */
/* Last updated by Enomoto Sanshiro on 4 February 2000. */


#include <string.h>
#include "MushConfig.hh"
#include "MushDataEncoder.hh"
#include "KorbDefs.hh"
#include "KorbNetworkPacket.hh"

using namespace std;
using namespace mush;


static const int InitialBufferSize = 1024;
static const int ReservedAreaWords = 4;


enum TKorbNetworkPacketOffsetTable {
    ReservedAreaOffset,
    PacketTypeOffset = ReservedAreaWords,
    DeliverStatusOffset
};

enum TKorbNetworkMessagePacketOffsetTable {
    MessagePacketBase = DeliverStatusOffset,
    ReceiverDomainIdOffset,
    ReceiverAdapterIdOffset,
    SenderDomainIdOffset,
    SenderAdapterIdOffset,
    DataSizeOffset,
    NumberOfHeaderWords
};

enum TKorbNetworkRoutingPacketOffsetTable {
    RoutingPacketBase = DeliverStatusOffset,
    LookingDomainIdOffset,
    RouteDomainIdOffset,
    NumberOfStepsOffset,
    RoutingPacketNumberOfWords
};



TKorbNetworkPacket::TKorbNetworkPacket(void)
{
    _BufferCapacity = InitialBufferSize;
    _Buffer = new char[_BufferCapacity];

    _HeaderSize = sizeof(UInt32) * NumberOfHeaderWords;

    SetDeliverStatus(dsOk);
    SetDataSize(0);
}

TKorbNetworkPacket::TKorbNetworkPacket(const TKorbNetworkPacket& Packet)
{
    _BufferCapacity = Packet._BufferCapacity;
    _Buffer = new char[_BufferCapacity];

    _HeaderSize = sizeof(UInt32) * NumberOfHeaderWords;

    memcpy(_Buffer, Packet._Buffer, Packet.ContentsSize());
}

TKorbNetworkPacket& TKorbNetworkPacket::operator= (const TKorbNetworkPacket& Packet)
{
    if (_BufferCapacity < Packet.ContentsSize()) {
        AllocateBuffer((int) (Packet.ContentsSize() * 1.5));
    }

    memcpy(_Buffer, Packet._Buffer, Packet.ContentsSize());

    return *this;
}

TKorbNetworkPacket::~TKorbNetworkPacket()
{
    delete _Buffer;
}

void* TKorbNetworkPacket::ContentsAddress(void)
{
    return _Buffer;
}

int TKorbNetworkPacket::ContentsSize(void) const
{
    return _HeaderSize + DataSize();
}

int TKorbNetworkPacket::ContentsCapacity(void) const
{
    return _BufferCapacity;
}

void TKorbNetworkPacket::AllocateBuffer(int BufferCapacity)
{
    if (_BufferCapacity < BufferCapacity) {
        _BufferCapacity = BufferCapacity;

	delete _Buffer;
	_Buffer = new char[_BufferCapacity];
    }
}

void TKorbNetworkPacket::SetPacketType(int PacketType)
{
    ((UInt32*) _Buffer)[PacketTypeOffset] = PacketType;
}

void TKorbNetworkPacket::SetDeliverStatus(int DeliverStatus)
{
    ((UInt32*) _Buffer)[DeliverStatusOffset] = DeliverStatus;
}

int TKorbNetworkPacket::PacketType(void) const
{
    return ((UInt32*) _Buffer)[PacketTypeOffset];
}

int TKorbNetworkPacket::DeliverStatus(void) const
{
    return ((UInt32*) _Buffer)[DeliverStatusOffset];
}

bool TKorbNetworkPacket::IsByteOrderReversed(void) const
{
    return ! (PacketType() & _PacketTypeMask);
}

void TKorbNetworkPacket::ReverseByteOrder(void)
{
    int WordSize = sizeof(UInt32);
    bool ReverseBytes = true;

    TMushBinaryDataEncoder DataEncoder(WordSize, ReverseBytes);
    DataEncoder.DecodeOn(_Buffer, _HeaderSize);
}

void TKorbNetworkPacket::SetReceiver(int DomainId, int AdapterId)
{
    ((UInt32*) _Buffer)[ReceiverDomainIdOffset] = DomainId;
    ((UInt32*) _Buffer)[ReceiverAdapterIdOffset] = AdapterId;
}

void TKorbNetworkPacket::SetSender(int DomainId, int AdapterId)
{
    ((UInt32*) _Buffer)[SenderDomainIdOffset] = DomainId;
    ((UInt32*) _Buffer)[SenderAdapterIdOffset] = AdapterId;
}

void TKorbNetworkPacket::SetData(const void* Data, int DataSize)
{
    void* DataArea = AllocateDataArea(DataSize);
    memcpy(DataArea, Data, DataSize);
}

void TKorbNetworkPacket::SetDataSize(int DataSize)
{
    ((UInt32*) _Buffer)[DataSizeOffset] = DataSize;    
}

void* TKorbNetworkPacket::AllocateDataArea(int MaxDataSize)
{
    ((UInt32*) _Buffer)[DataSizeOffset] = MaxDataSize;

    int RequiredBufferSize = _HeaderSize + MaxDataSize;
    if (_BufferCapacity < RequiredBufferSize) {
        AllocateBuffer((int) (RequiredBufferSize * 1.5));
    }

    return ((UInt32*) _Buffer) + NumberOfHeaderWords;
}

int TKorbNetworkPacket::ReceiverDomainId(void) const
{
    return ((UInt32*) _Buffer)[ReceiverDomainIdOffset];
}

int TKorbNetworkPacket::ReceiverAdapterId(void) const
{
    return ((UInt32*) _Buffer)[ReceiverAdapterIdOffset];
}

int TKorbNetworkPacket::SenderDomainId(void) const
{
    return ((UInt32*) _Buffer)[SenderDomainIdOffset];
}

int TKorbNetworkPacket::SenderAdapterId(void) const
{
    return ((UInt32*) _Buffer)[SenderAdapterIdOffset];
}

int TKorbNetworkPacket::DataSize(void) const
{
    return ((UInt32*) _Buffer)[DataSizeOffset];
}

void* TKorbNetworkPacket::DataArea(void)
{
    return ((UInt32*) _Buffer) + NumberOfHeaderWords;
}

int TKorbNetworkPacket::DataAreaCapacity(void) const
{
    return _BufferCapacity - _HeaderSize;
}



void TKorbNetworkPacket::SetLookingDomainId(int LookingDomainId)
{
    ((UInt32*) _Buffer)[LookingDomainIdOffset] = LookingDomainId;
}

void TKorbNetworkPacket::SetRouteDomainId(int RouteDomainId)
{
    ((UInt32*) _Buffer)[RouteDomainIdOffset] = RouteDomainId;
}

void TKorbNetworkPacket::SetNumberOfSteps(int NumberOfSteps)
{
    ((UInt32*) _Buffer)[NumberOfStepsOffset] = NumberOfSteps;
}

int TKorbNetworkPacket::LookingDomainId(void)
{
    return ((UInt32*) _Buffer)[LookingDomainIdOffset];
}

int TKorbNetworkPacket::RouteDomainId(void)
{
    return ((UInt32*) _Buffer)[RouteDomainIdOffset];
}

int TKorbNetworkPacket::NumberOfSteps(void)
{
    return ((UInt32*) _Buffer)[NumberOfStepsOffset];
}

void TKorbNetworkPacket::Dump(ostream& os)
{
    os << "  Packet Type Id: " << PacketType();
    switch (PacketType()) {
      case ptMessage:
	os << " (MESSAGE)" << endl;
	break;
      case ptRoutingQuery:
	os << " (ROUTING_QUERY)" << endl;
	break;
      case ptRoutingReply:
	os << " (ROUTING_REPLY)" << endl;
	break;
      case ptEchoRequest:
	os << " (ECHO_REQUEST)" << endl;
	break;
      case ptEchoReply:
	os << " (ECHO_REPLY)" << endl;
	break;
      default:
	os << " (/?/?/?: 0x" << hex << PacketType() << dec << ")" << endl;
    }

    os << "  Deliver Status: " << DeliverStatus();
    switch (DeliverStatus()) {
      case dsOk:
	os << " (OK)" << endl;
	break;
      case dsNoRoute:
	os << " (NO_ROUTE)" << endl;
	break;
      default:
	os << " (/?/?/?)" << endl;
    }

    if (
	(PacketType() == ptMessage) ||
	(PacketType() == ptEchoRequest) ||
	(PacketType() == ptEchoReply)
    ){
	os << "  Sender Domain Id: " << SenderDomainId() << endl;
	os << "  Sender Adapter Id: " << SenderAdapterId() << endl;
	os << "  Receiver Domain Id: " << ReceiverDomainId() << endl;
	os << "  Receiver Adapter Id: " << ReceiverAdapterId() << endl;
	os << "  Data Size: " << DataSize() << endl;
    }
    else if (
	(PacketType() == ptRoutingQuery) ||
	(PacketType() == ptRoutingReply)
    ){
	os << "  Looking Domain Id: " << LookingDomainId() << endl;
	os << "  RouteDomain Id: " << RouteDomainId() << endl;
	os << "  Number of Steps: " << NumberOfSteps() << endl;
    }

    os << endl;
}
