/* KorbNetworkRoutingTable.cc */
/* Created by Enomoto Sanshiro on 4 December 1999. */
/* Last updated by Enomoto Sanshiro on 7 February 2000. */


#include <vector>
#include "KorbNetworkHub.hh"
#include "KorbNetworkRoutingTable.hh"

using namespace std;


class TRoutingInformation {
  public:
    TRoutingInformation(int RouteDomainId, int NumberOfSteps);
    virtual ~TRoutingInformation();
    virtual int RouteDomainId(void) const;
    virtual int NumberOfSteps(void) const;
  protected:
    int _RouteDomainId;
    int _NumberOfSteps;
};


class TSearchingEntry {
  public:
    TSearchingEntry(int NumberOfQueries, int ClientPortId = -1);
    virtual ~TSearchingEntry();
    virtual void IncrementReplyCount(void);
    virtual bool HasFinished(void) const;
    virtual void AddClient(int ClientPortId);
    virtual vector<int> ClientList(void) const;
  protected:
    int _NumberOfQueries;
    int _ReplyCount;
    vector<int> _ClientList;
};



TKorbNetworkRoutingTable::TKorbNetworkRoutingTable(TKorbNetworkHub* NetworkHub)
{
    _NetworkHub = NetworkHub;
    _LocalDomainId = NetworkHub->DomainId();
}

TKorbNetworkRoutingTable::~TKorbNetworkRoutingTable()
{
    map<int, TRoutingInformation*, less<int> >::iterator _RoutingInfoIterator;
    for (
	_RoutingInfoIterator = _CacheTable.begin();
	_RoutingInfoIterator != _CacheTable.end();
	_RoutingInfoIterator++
    ){
	delete (*_RoutingInfoIterator).second;
    }

    map<int, TSearchingEntry*, less<int> >::iterator _SearchingEntryIterator;
    for (
	_SearchingEntryIterator = _SearchingTable.begin();
	_SearchingEntryIterator != _SearchingTable.end();
	_SearchingEntryIterator++
    ){
	delete (*_SearchingEntryIterator).second;
    }
}

void TKorbNetworkRoutingTable::AddConnection(int UplinkPortId, int RemoteDomainId)
{
    _UplinkTable[RemoteDomainId] = UplinkPortId;

    int RouteDomainId = RemoteDomainId;
    int NumberOfSteps = 1;
    AddRouteToCache(RemoteDomainId, RouteDomainId, NumberOfSteps);
}

int TKorbNetworkRoutingTable::LookupRouteTo(int RemoteDomainId) throw(TKorbException)
{
    if (_CacheTable.count(RemoteDomainId) > 0) {
	TRoutingInformation* RoutingInfo = _CacheTable[RemoteDomainId];
	if (RoutingInfo->NumberOfSteps() > 0) {
	    return _UplinkTable[RoutingInfo->RouteDomainId()];
	}
	else {
	    return lrNoRoute;
	}
    }
    
    if (_SearchingTable.count(RemoteDomainId) == 0) {
        TKorbNetworkPacket RoutingPacket;
	RoutingPacket.SetPacketType(TKorbNetworkPacket::ptRoutingQuery);
	RoutingPacket.SetLookingDomainId(RemoteDomainId);
	RoutingPacket.SetNumberOfSteps(0);

	Search(RoutingPacket);
    }

    return lrSearching;
}

int TKorbNetworkRoutingTable::ProcessRoutingPacket(TKorbNetworkPacket& Packet, int UplinkPortId) throw(TKorbException)
{
    if (Packet.PacketType() == TKorbNetworkPacket::ptRoutingQuery) {
        return ProcessRoutingQueryPacket(Packet, UplinkPortId);
    }
    else if (Packet.PacketType() == TKorbNetworkPacket::ptRoutingReply) {
        return ProcessRoutingReplyPacket(Packet, UplinkPortId);
    }
    else {
        /* Unknown Packet */
        return 0;
    }
}

int TKorbNetworkRoutingTable::ProcessRoutingQueryPacket(TKorbNetworkPacket& RoutingPacket, int UplinkPortId) throw(TKorbException)
{
    int DomainId = RoutingPacket.LookingDomainId();

    if (_CacheTable.count(DomainId) > 0) {
        int NumberOfSteps = _CacheTable[DomainId]->NumberOfSteps();
	RoutingPacket.SetPacketType(TKorbNetworkPacket::ptRoutingReply);
	RoutingPacket.SetRouteDomainId(_LocalDomainId);
	RoutingPacket.SetNumberOfSteps(NumberOfSteps);

	_NetworkHub->SendToUplinkPort(UplinkPortId, RoutingPacket);
    }
    else if (_SearchingTable.count(DomainId) > 0) {
	_SearchingTable[DomainId]->AddClient(UplinkPortId);

	int NumberOfSteps = -1;
	RoutingPacket.SetPacketType(TKorbNetworkPacket::ptRoutingReply);
	RoutingPacket.SetRouteDomainId(_LocalDomainId);
	RoutingPacket.SetNumberOfSteps(NumberOfSteps);

	_NetworkHub->SendToUplinkPort(UplinkPortId, RoutingPacket);
    }
    else {
        int NumberOfSteps = RoutingPacket.NumberOfSteps() + 1;
	RoutingPacket.SetNumberOfSteps(NumberOfSteps);

	Search(RoutingPacket, UplinkPortId);
    }

    return 0;
}

int TKorbNetworkRoutingTable::ProcessRoutingReplyPacket(TKorbNetworkPacket& RoutingPacket, int UplinkPortId) throw(TKorbException)
{
    int DomainId = RoutingPacket.LookingDomainId();
    int RouteDomainId = RoutingPacket.RouteDomainId();
    int NumberOfSteps = RoutingPacket.NumberOfSteps();

    if (NumberOfSteps > 0) {
	NumberOfSteps += 1;
        AddRouteToCache(DomainId, RouteDomainId, NumberOfSteps);
    }

    if (_SearchingTable.count(DomainId) > 0) {
        TSearchingEntry* SearchingEntry = _SearchingTable[DomainId];
        SearchingEntry->IncrementReplyCount();

	if (SearchingEntry->HasFinished()) {
	    vector<int> ClientList = SearchingEntry->ClientList();

	    RoutingPacket.SetNumberOfSteps(NumberOfSteps);
	    RoutingPacket.SetRouteDomainId(_LocalDomainId);
	    for (unsigned i = 0; i < ClientList.size(); i++) {
		_NetworkHub->SendToUplinkPort(ClientList[i], RoutingPacket);
	    }

	    delete SearchingEntry;
	    _SearchingTable.erase(DomainId);
	}
    }

    return 0;
}

void TKorbNetworkRoutingTable::Search(TKorbNetworkPacket& RoutingPacket, int ClientPortId) throw(TKorbException)
{
    int NumberOfQueries = 0;
    map<int, int, less<int> >::iterator UplinkTableIterator;
    for (
	UplinkTableIterator = _UplinkTable.begin();
	UplinkTableIterator != _UplinkTable.end();
	UplinkTableIterator++
    ){
        int UplinkPortId = (*UplinkTableIterator).second;
	if (UplinkPortId != ClientPortId) {
	    _NetworkHub->SendToUplinkPort(UplinkPortId, RoutingPacket);
	    NumberOfQueries++;
	}
    }

    int DomainId = RoutingPacket.LookingDomainId();
    TSearchingEntry *SearchingEntry = new TSearchingEntry(
	NumberOfQueries, ClientPortId
    );
    _SearchingTable[DomainId] = SearchingEntry;
}

void TKorbNetworkRoutingTable::AddRouteToCache(int RemoteDomainId, int RouteDomainId, int NumberOfSteps)
{
    if (_CacheTable.count(RemoteDomainId) > 0) {
	TRoutingInformation* OldRoutingInfo = _CacheTable[RemoteDomainId];
	if (OldRoutingInfo->NumberOfSteps() > NumberOfSteps) {
	    delete OldRoutingInfo;
	    _CacheTable[RemoteDomainId] = new TRoutingInformation(
		RouteDomainId, NumberOfSteps
	    );
	}
    }
    else {
	_CacheTable[RemoteDomainId] = new TRoutingInformation(
	    RouteDomainId, NumberOfSteps
	);
    }
}


TRoutingInformation::TRoutingInformation(int RouteDomainId, int NumberOfSteps)
{
    _RouteDomainId = RouteDomainId;
    _NumberOfSteps = NumberOfSteps;
}

TRoutingInformation::~TRoutingInformation()
{
}

int TRoutingInformation::RouteDomainId(void) const
{
    return _RouteDomainId;
}

int TRoutingInformation::NumberOfSteps(void) const
{
    return _NumberOfSteps;
}


TSearchingEntry::TSearchingEntry(int NumberOfQueries, int ClientPortId)
{
    _NumberOfQueries = NumberOfQueries;
    _ReplyCount = 0;

    if (ClientPortId > 0) {
	_ClientList.push_back(ClientPortId);
    }
}

TSearchingEntry::~TSearchingEntry()
{
}

void TSearchingEntry::IncrementReplyCount(void)
{
    _ReplyCount++;
}

bool TSearchingEntry::HasFinished(void) const
{
    return !(_ReplyCount < _NumberOfQueries);
}

void TSearchingEntry::AddClient(int ClientPortId)
{
    _ClientList.push_back(ClientPortId);
}

vector<int> TSearchingEntry::ClientList(void) const
{
    return _ClientList;
}
