/* KorbObjectAdapter.cc */
/* Created by Enomoto Sanshiro on 27 February 2000. */
/* Last updated by Enomoto Sanshiro on 27 February 2000. */


#include <fstream>
#include <string>
#include "MushSignal.hh"
#include "KorbOrb.hh"
#include "KorbObjectMessenger.hh"
#include "KorbObjectAdapter.hh"
#include "KorbNamingContext.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbObjectAdapter", ios::app);
//#define dumpfile cerr
#endif



TKorbObjectAdapter::TKorbObjectAdapter(TKorbOrb* Orb)
{
    _Orb = Orb;
}

TKorbObjectAdapter::~TKorbObjectAdapter()
{
}

void TKorbObjectAdapter::AddMessenger(TKorbObjectMessenger* Messenger, const string& ObjectName) throw(TKorbException)
{
    TKorbObjectReference ObjectReference = _Orb->GenerateObjectReference(
	Messenger->ClassName(), ObjectName
    );

    int ObjectId = ObjectReference.ObjectId();
    _ObjectIdTable[ObjectName] = ObjectId;
    AddMessenger(Messenger, ObjectId);
    
    _Orb->NamingContext()->RegisterObject(ObjectReference);

#ifdef DEBUG
    dumpfile << _Orb->DomainId() << "."
	     << _Orb->AdapterId() << "." 
	     << ObjectId << ": " 
	     << ObjectName << endl;
#endif
}

void TKorbObjectAdapter::AddMessenger(TKorbObjectMessenger* Messenger, int ObjectId)
{
    _MessengerTable[ObjectId] = Messenger;
}

void TKorbObjectAdapter::RemoveMessenger(const string& ObjectName)
{
    if (_ObjectIdTable.count(ObjectName) > 0) {
	int ObjectId = _ObjectIdTable[ObjectName];
	_MessengerTable.erase(ObjectId);
    }

    _Orb->NamingContext()->UnregisterObject(ObjectName);
}

int TKorbObjectAdapter::NumberOfMessengers(void) const
{
    return _MessengerTable.size();
}

void TKorbObjectAdapter::Start(void) throw(TKorbException)
{
    TMushSignalHandler SignalHandler;
    TMushSignalCounter SignalCounter;

    SignalHandler.RegisterClient(SIGINT, &SignalCounter);
    SignalHandler.RegisterClient(SIGTERM, &SignalCounter);
    SignalHandler.RegisterClient(SIGQUIT, &SignalCounter);
    SignalHandler.StartHandling();
    
    while (SignalCounter.SignalCount() == 0) {
	DoTransaction();
    }
}

int TKorbObjectAdapter::DoTransaction(void) throw(TKorbException)
{
    TKorbOrbPacket OrbPacket = _Orb->CreateServerOrbPacket();

    int RequestId = _Orb->WaitRequest(OrbPacket);
    if (RequestId != 0) {
        return ProcessRequest(RequestId, OrbPacket);
    }
    
    return 0;
}

int TKorbObjectAdapter::DoNowaitTransaction(void) throw(TKorbException)
{
    TKorbOrbPacket OrbPacket = _Orb->CreateServerOrbPacket();

    int RequestId = _Orb->GetRequest(OrbPacket);
    if (RequestId != 0) {
        return ProcessRequest(RequestId, OrbPacket);
    }

    return 0;
}

int TKorbObjectAdapter::ProcessRequest(int RequestId, TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    int Result = 0;

    int ObjectId = OrbPacket.ObjectId();    
    bool IsOneWay = OrbPacket.IsOneWay();
    int ClientDomainId = OrbPacket.NetworkPacket()->SenderDomainId();
    int ClientAdapterId = OrbPacket.NetworkPacket()->SenderAdapterId();

#ifdef DEBUG
    dumpfile << "request " << RequestId << ": " << flush;
    int MyDomainId = OrbPacket.NetworkPacket()->ReceiverDomainId();
    int MyAdapterId = OrbPacket.NetworkPacket()->ReceiverAdapterId();
    dumpfile << ClientDomainId << "." << ClientAdapterId << ".?" << " => "
             << MyDomainId << "." << MyAdapterId << "." << ObjectId << ": "
             << "method " << OrbPacket.MethodId() << ", "
             << "request " << RequestId 
	     << (IsOneWay ? ", oneway" : "") <<endl;
#endif
	    
    if (_MessengerTable.count(ObjectId) > 0) {
	if (! _MessengerTable[ObjectId]->DispatchMethodInvocation(OrbPacket)) {
#ifdef DEBUG
	    dumpfile << "UNKNOWN METHOD" << endl;
#endif
	}
    }
    else {
#ifdef DEBUG
        dumpfile << "UNKNOWN OBJECT" << endl;
#endif
    }

    if (! IsOneWay) {
	_Orb->SendReplyFor(
	    RequestId, ClientDomainId, ClientAdapterId, OrbPacket
	);
    }

    return Result = 1;
}
