/* KorbOrb.cc */
/* Created by Enomoto Sanshiro on 26 February 2000. */
/* Last updated by Enomoto Sanshiro on 29 April 2002. */


#include <fstream>
#include <string>
#include <deque>
#include <climits>
#include "MushProcess.hh"
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "MushFileSystem.hh"
#include "KorbNetworkHub.hh"
#include "KorbNetworkAdapter.hh"
#include "KorbOrbPacket.hh"
#include "KorbObjectReference.hh"
#include "KorbObjectAdapter.hh"
#include "KorbNamingContext.hh"
#include "KorbBroker.hh"
#include "KorbOrb.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbOrb", ios::app);
#endif


static const int MessageQueueProjectId = 1;
static const int InitialObjectId = 0x0100;
static const int MaxRequestCount = 100;
static const int MaxRequestIdBase = 100000000;

// SendRequest() is defered when number of pending packets 
// in the message queue reaches this number.
// This is a temporary measure: data size in the message queue
// should be used instead.
static const int MaxNumberOfPendingPackets = 100;


TKorbOrb::TKorbOrb(void)
{
    _NetworkDomainId = TKorbNetworkHub::DomainId_Unknown;
    _NetworkAdapterId = TMushProcess().ProcessId();

    _ObjectAdapter = 0;
    _NamingContext = 0;
    _NetworkAdapter = 0;

    _NextObjectId = InitialObjectId;

    _RequestIdBase = (_NetworkAdapterId % (MaxRequestIdBase / MaxRequestCount)) * MaxRequestCount;
    _RequestCount = 0;
}

TKorbOrb::TKorbOrb(int NetworkAdapterId)
{
    _NetworkDomainId = TKorbNetworkHub::DomainId_Unknown;
    _NetworkAdapterId = NetworkAdapterId;

    _ObjectAdapter = 0;
    _NamingContext = 0;
    _NetworkAdapter = 0;

    _NextObjectId = InitialObjectId;

    _RequestIdBase = (_NetworkAdapterId % (MaxRequestIdBase / MaxRequestCount)) * MaxRequestCount;
    _RequestCount = 0;
}

TKorbOrb::~TKorbOrb()
{
    delete _NamingContext;
    delete _ObjectAdapter;
    delete _NetworkAdapter;
}

int TKorbOrb::ProjectId(void)
{
    return MessageQueueProjectId;
}

string TKorbOrb::ProjectPath(void) throw(TKorbException)
{
    string ProjectPath;
    try {
	ProjectPath = TMushEnvironmentVariable("KORB_PROJECT_PATH").AsString();
    }
    catch (TSystemCallException &e) {
	ProjectPath = "/tmp/korb";

	try {
	    if (! TMushFileAttribute(ProjectPath).IsReadable()) {
		TMushFileSystem::MakeDirectory("/tmp/korb");
	    }
	}
	catch (TSystemCallException &e) {
	    throw TKorbException(
		"TKorbOrb::ProjectPath()", 
		"system call exception: " + e.Message()
	    );
	}
    }    

    return ProjectPath;
}

int TKorbOrb::DomainId(void) const
{
    return _NetworkDomainId;
}

int TKorbOrb::AdapterId(void) const
{
    return _NetworkAdapterId;
}

TKorbObjectAdapter* TKorbOrb::ObjectAdapter(void)
{
    if (_ObjectAdapter == 0) {
	_ObjectAdapter = new TKorbObjectAdapter(this);
    }
    
    return _ObjectAdapter;
}

TKorbNamingContext* TKorbOrb::NamingContext(void)
{
    if (_NamingContext == 0) {
	_NamingContext = new TKorbNamingContextProxy(this);
	_NamingContext->Bind(
	    TKorbNamingContext::GetInitialReference()
	);
    }
    
    return _NamingContext;
}

TKorbObjectReference TKorbOrb::GenerateObjectReference(const string& ClassName, const string& ObjectName, int ObjectId) throw(TKorbException)
{
    if (_NetworkAdapter == 0) {
	Initialize();
    }

    if (ObjectId == 0) {
	ObjectId = _NextObjectId;
	_NextObjectId++;
    }
    
    return TKorbObjectReference(
	ClassName, ObjectName, 
	_NetworkDomainId, _NetworkAdapterId, ObjectId
    );
}

void TKorbOrb::Initialize(void) throw(TKorbException)
{
    if (_NetworkAdapter != 0) {
	return;
    }

    _NetworkAdapter = new TKorbNetworkAdapter(
	_NetworkAdapterId, ProjectId(), ProjectPath()
    );

    static const int AttachWait_usec = 100000;
    static const int MaxNumberOfTries = 300;  // i.e. 30 sec.
    int NumberOfTries = 0;
    while (1) {
	try {
	    _NetworkAdapter->Attach();
	}
	catch (TKorbException& e) {
	    NumberOfTries++;
	    if (NumberOfTries < MaxNumberOfTries) {
		TMushRealTimeTimer(0, AttachWait_usec).Suspend();
		continue;
	    }
	    throw e;
	}
	break;
    }

    _NetworkDomainId = _NetworkAdapter->DomainId();
}

TKorbOrbPacket TKorbOrb::CreateServerOrbPacket(void)
{
    TKorbOrbPacket OrbPacket(&_ServerNetworkPacket);

    return OrbPacket;
}

TKorbOrbPacket TKorbOrb::CreateClientOrbPacket(const TKorbObjectReference& ObjectReference, int MethodId)
{
    TKorbOrbPacket OrbPacket(&_ClientNetworkPacket);

    OrbPacket.ObjectId() = ObjectReference.ObjectId();
    OrbPacket.MethodId() = MethodId;
    OrbPacket.IsOneWay() = 0;

    return OrbPacket;
}

int TKorbOrb::InvokeMethod(const TKorbObjectReference& ObjectReference, TKorbOrbPacket& OrbPacket, int TimeOut_sec, bool IsAsynchronous) throw(TKorbException)
{
    int RemoteDomainId = ObjectReference.DomainId();
    int RemoteAdapterId = ObjectReference.AdapterId();

    int RequestId = SendRequest(RemoteDomainId, RemoteAdapterId, OrbPacket);

    int Result;
    if (OrbPacket.IsOneWay() || IsAsynchronous) {
	Result = RequestId;
    }
    else {
	Result = WaitReplyFor(RequestId, OrbPacket, TimeOut_sec);
    }
    
    return Result;
}

void TKorbOrb::SendSignal(const TKorbObjectReference& ObjectReference, int SignalId, bool IsOneWay) throw(TKorbException)
{
    TKorbOrbPacket OrbPacket(&_ClientNetworkPacket);
    OrbPacket.MethodId() = TKorbBroker::MethodId_SendSignal;
    if (IsOneWay) {
	OrbPacket.IsOneWay() = 1;
    }

    int ProcessId = ObjectReference.AdapterId();

    OrbPacket.ArgumentSerializer().PutInt(ProcessId);
    OrbPacket.ArgumentSerializer().PutInt(SignalId);

    int RemoteDomainId = ObjectReference.DomainId();
    int RemoteAdapterId = TKorbBroker::AdapterId();

    int RequestId = SendRequest(RemoteDomainId, RemoteAdapterId, OrbPacket);

    if (! IsOneWay) {
	WaitReplyFor(RequestId, OrbPacket);
    }
}

void TKorbOrb::SendUrgSignal(const TKorbObjectReference& ObjectReference, bool IsOneWay) throw(TKorbException)
{
    // the value of SIGURG varies depending on OS.

    TKorbOrbPacket OrbPacket(&_ClientNetworkPacket);
    OrbPacket.MethodId() = TKorbBroker::MethodId_SendUrgSignal;
    if (IsOneWay) {
	OrbPacket.IsOneWay() = 1;
    }

    int ProcessId = ObjectReference.AdapterId();

    OrbPacket.ArgumentSerializer().PutInt(ProcessId);

    int RemoteDomainId = ObjectReference.DomainId();
    int RemoteAdapterId = TKorbBroker::AdapterId();

    int RequestId = SendRequest(RemoteDomainId, RemoteAdapterId, OrbPacket);
    if (! IsOneWay) {
	WaitReplyFor(RequestId, OrbPacket);
    }
}

int TKorbOrb::SendRequest(int DomainId, int AdapterId, TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (_NetworkAdapter == 0) {
	Initialize();
    }

    while (_NetworkAdapter->NumberOfPendingPackets() > MaxNumberOfPendingPackets) {
	TMushRealTimeTimer(0, 100000).Suspend();
    }

    _RequestCount = _RequestCount % (MaxRequestCount - 1) + 1;
    int RequestId = _RequestIdBase + _RequestCount;

    OrbPacket.RequestId() = RequestId;
    OrbPacket.IsRequest() = 1;

    return Send(DomainId, AdapterId, OrbPacket) ? RequestId : 0;
}

int TKorbOrb::SendReplyFor(int RequestId, int DomainId, int AdapterId, TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    OrbPacket.RequestId() = RequestId;
    OrbPacket.IsRequest() = 0;

    return Send(DomainId, AdapterId, OrbPacket) ? RequestId : 0;
}

int TKorbOrb::WaitRequest(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (! _RequestPacketQueue.empty()) {
	*(OrbPacket.NetworkPacket()) = _RequestPacketQueue.front();
	_RequestPacketQueue.pop_front();
	OrbPacket.OnReceive();

#ifdef DEBUG
	dumpfile << _NetworkDomainId << "." << _NetworkAdapterId << ": ";
	dumpfile << "pop request: " << OrbPacket.RequestId() << endl;
	OrbPacket.Dump(dumpfile);
#endif

	return OrbPacket.RequestId();
    }

    while (Receive(OrbPacket) > 0) {
	if (! OrbPacket.IsRequest()) {
	    _ReplyPacketTable[OrbPacket.RequestId()] = *OrbPacket.NetworkPacket();
	    continue;
	}

#ifdef DEBUG
	dumpfile << _NetworkDomainId << "." << _NetworkAdapterId << ": ";
	dumpfile << "receive request: " << OrbPacket.RequestId() << endl;
#endif

	return OrbPacket.RequestId();
    }

    return 0;
}

int TKorbOrb::GetRequest(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (! _RequestPacketQueue.empty()) {
	return WaitRequest(OrbPacket);
    }

    while (ReceiveIfAvailable(OrbPacket) > 0) {
	if (! OrbPacket.IsRequest()) {
	    _ReplyPacketTable[OrbPacket.RequestId()] = *OrbPacket.NetworkPacket();
	    continue;
	}

#ifdef DEBUG
	dumpfile << _NetworkDomainId << "." << _NetworkAdapterId << ": ";
	dumpfile << "receive request: " << OrbPacket.RequestId() << endl;
#endif

	return OrbPacket.RequestId();
    }

    return 0;
}

int TKorbOrb::WaitReplyFor(int RequestId, TKorbOrbPacket& OrbPacket, int TimeOut_sec) throw(TKorbException)
{
    int StartTime = TMushDateTime::SecSinceEpoch();
    int Result = 0;

    std::map<int, TKorbNetworkPacket>::iterator ReplyPacketIterator;
    ReplyPacketIterator = _ReplyPacketTable.find(RequestId);
    if (ReplyPacketIterator != _ReplyPacketTable.end()) {
	*(OrbPacket.NetworkPacket()) = (*ReplyPacketIterator).second;
	_ReplyPacketTable.erase(ReplyPacketIterator);
	OrbPacket.OnReceive();
	return 1;
    }

    while (1) {
	if (ReceiveIfAvailable(OrbPacket) > 0) {
	    if (OrbPacket.IsRequest()) {
		_RequestPacketQueue.push_back(*OrbPacket.NetworkPacket());
#ifdef DEBUG
		dumpfile << _NetworkDomainId << ".";
		dumpfile << _NetworkAdapterId << ": ";
		dumpfile << "push request: " << OrbPacket.RequestId() << endl;
		OrbPacket.Dump(dumpfile);
#endif
		continue;
	    }
	    else if (OrbPacket.RequestId() != RequestId) {
		_ReplyPacketTable[OrbPacket.RequestId()] = *OrbPacket.NetworkPacket();
		continue;
	    }
	    else {
#ifdef DEBUG
		dumpfile << _NetworkDomainId << ".";
		dumpfile << _NetworkAdapterId << ": ";
		dumpfile << "receive reply: " << OrbPacket.RequestId() << endl;
#endif
		Result = RequestId;
		break;
	    }
	}

	if (
	    (TimeOut_sec > 0) &&
	    (TMushDateTime::SecSince(StartTime) > TimeOut_sec)
	){
	    Result = 0;
	    break;
	}
	
	TMushRealTimeTimer(0, 10000).Suspend();
    }

    return Result;
}

int TKorbOrb::Send(int NetworkDomainId, int NetworkAdapterId, TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (_NetworkAdapter == 0) {
	Initialize();
    }

    TKorbNetworkPacket* NetworkPacket = OrbPacket.NetworkPacket();
    NetworkPacket->SetSender(_NetworkDomainId, _NetworkAdapterId);
    NetworkPacket->SetReceiver(NetworkDomainId, NetworkAdapterId);
    NetworkPacket->SetDataSize(OrbPacket.PacketSize());

    OrbPacket.OnSend();
    _NetworkAdapter->Send(*NetworkPacket);

#ifdef DEBUG
    dumpfile << _NetworkDomainId << "." << _NetworkAdapterId << ": ";
    dumpfile << "send request: " << OrbPacket.RequestId() << endl;
#endif

    return 1;
}

int TKorbOrb::Receive(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (_NetworkAdapter == 0) {
	Initialize();
    }

    TKorbNetworkPacket* NetworkPacket = OrbPacket.NetworkPacket();
    int Length = _NetworkAdapter->Receive(*NetworkPacket);
    
    if (Length > 0) {
	OrbPacket.OnReceive();
    }

    return (Length > 0) ? 1 : 0;
}

int TKorbOrb::ReceiveIfAvailable(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    if (_NetworkAdapter == 0) {
	Initialize();
    }

    static const int ReceiveFlag = TKorbNetworkAdapter::rfNoWait;
    TKorbNetworkPacket* NetworkPacket = OrbPacket.NetworkPacket();
    int Length = _NetworkAdapter->Receive(*NetworkPacket, ReceiveFlag);
    
    if (Length > 0) {
	OrbPacket.OnReceive();
    }

    return (Length > 0) ? 1 : 0;
}
