/* MushDataEncoder.hh */
/* Created by Enomoto Sanshiro on 23 January 2000. */
/* Last updated by Enomoto Sanshiro on 28 January 2002. */


#ifndef __MushDataEncoder_hh__
#define __MushDataEncoder_hh__

#include "MushDefs.hh"

#if USE_ZLIB
extern "C" {
#include <zlib.h>
};
#endif


class TMushDataEncoder {
  public:
    TMushDataEncoder(void);
    virtual ~TMushDataEncoder();
    virtual size_t Encode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException) = 0;
    virtual size_t Decode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException) = 0;
};


/* Binary to binary encoder (corrects byte order) */
class TMushBinaryDataEncoder: public TMushDataEncoder {
  public:
    TMushBinaryDataEncoder(size_t WordSize, bool ReverseBytes = false);
    virtual ~TMushBinaryDataEncoder();
    virtual size_t Encode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException);
    virtual size_t Decode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException);
    virtual size_t EncodeOn(void* Data, size_t DataSize) throw(TSystemCallException);
    virtual size_t DecodeOn(void* Data, size_t DataSize) throw(TSystemCallException);
  protected:
    virtual size_t ConvertData(void* Data, size_t Size);
  protected:
    size_t _WordSize;
    bool _ReverseBytes;
};


/* GZIP encoder */
#if USE_ZLIB
class TMushZlibDataEncoder: public TMushDataEncoder {
  public:
    TMushZlibDataEncoder(int CompressionLevel = 1);
    virtual ~TMushZlibDataEncoder();
    virtual size_t Encode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException);
    virtual size_t Decode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException);
  protected:
    virtual void ReallocateBuffer(size_t NewSize, size_t DataSizeInBuffer = 0);
  protected:
    int _CompressionLevel;
    char* _Buffer;
    size_t _BufferSize;
    z_stream _DeflateZ;
    z_stream _InflateZ;
};
#else
class TMushZlibDataEncoder: public TMushDataEncoder {
  public:
    TMushZlibDataEncoder(int CompressionLevel = 1) { 
	throw TSystemCallException(
	    "TMushZlibDataEncoder::TMushZlibDataEncoder()", 
	    "function not supported"
	); 
    }
    virtual ~TMushZlibDataEncoder() {}
    virtual size_t Encode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException) { return 0; }
    virtual size_t Decode(void* Data, size_t DataSize, void*& Result, size_t& ResultSize) throw(TSystemCallException) { return 0; }
};
#endif



/* Base64 encoder */
class TMushBase64Encoder: public TMushDataEncoder {
};


#endif
