/* MushDecoratedFile.cc */
/* Created by Enomoto Sanshiro on 25 October 1999. */
/* Last updated by Enomoto Sanshiro on 30 April 2002. */


#include "MushFile.hh"
#include "MushDecoratedFile.hh"

using namespace std;
using namespace mush;


TMushDecoratedFile::TMushDecoratedFile(TMushFile* ComponentFile)
: TMushFile("decorated_file")
{
    _ComponentFile = ComponentFile;
}

TMushDecoratedFile::~TMushDecoratedFile()
{
    delete _ComponentFile;
}

void TMushDecoratedFile::OpenReadMode(void) throw(TSystemCallException)
{
    _ComponentFile->OpenReadMode();
}

void TMushDecoratedFile::OpenWriteMode(unsigned AccessMode, bool IsExclusive) throw(TSystemCallException)
{
    _ComponentFile->OpenWriteMode(AccessMode, IsExclusive);
}

void TMushDecoratedFile::OpenAppendMode(unsigned AccessMode) throw(TSystemCallException)
{
    _ComponentFile->OpenAppendMode(AccessMode);
}

void TMushDecoratedFile::OpenReadWriteMode(unsigned AccessMode) throw(TSystemCallException)
{
    _ComponentFile->OpenReadWriteMode(AccessMode);
}

void TMushDecoratedFile::Close(void)
{
    _ComponentFile->Close();
}

int TMushDecoratedFile::Read(void* Buffer, size_t Length) throw(TSystemCallException)
{
    return _ComponentFile->Read(Buffer, Length);
}

int TMushDecoratedFile::Write(void* Buffer, size_t Length) throw(TSystemCallException)
{
    return _ComponentFile->Write(Buffer, Length);
}

Int64 TMushDecoratedFile::Position(void) throw(TSystemCallException)
{
    return _ComponentFile->Position();
}

void TMushDecoratedFile::SeekTo(Int64 Offset) throw(TSystemCallException)
{
    _ComponentFile->SeekTo(Offset);
}

void TMushDecoratedFile::SeekBy(Int64 Offset) throw(TSystemCallException)
{
    _ComponentFile->SeekBy(Offset);
}

void TMushDecoratedFile::Unlink(void) throw(TSystemCallException)
{
    _ComponentFile->Unlink();
}

string TMushDecoratedFile::Name(void) const
{
    return _ComponentFile->Name();
}

long TMushDecoratedFile::Size(void) const throw(TSystemCallException)
{
    return _ComponentFile->Size();
}

bool TMushDecoratedFile::IsOpened(void) const
{
    return _ComponentFile->IsOpened();
}

int TMushDecoratedFile::FileDescriptor(void) const
{
    return _ComponentFile->FileDescriptor();
}



TMushEncodedFile::TMushEncodedFile(TMushFile* ComponentFile, TMushDataEncoder* DataEncoder)
: TMushDecoratedFile(ComponentFile)
{
    _DataEncoder = DataEncoder;

    _WorkAreaSize = 0;
    _WorkArea = 0;
}

TMushEncodedFile::~TMushEncodedFile()
{
    delete[] _WorkArea;
}

int TMushEncodedFile::Read(void* Buffer, size_t Length) throw(TSystemCallException)
{
    int ReadLength = _ComponentFile->Read(Buffer, Length);
    if (ReadLength <= 0) {
	return ReadLength;
    }

    void* DecodedData;
    size_t DecodedDataSize;
    _DataEncoder->Decode(Buffer, ReadLength, DecodedData, DecodedDataSize);

    return DecodedDataSize;
}

int TMushEncodedFile::Write(void* Buffer, size_t Length) throw(TSystemCallException)
{
    if (_WorkAreaSize < Length) {
        delete[] _WorkArea;
	_WorkAreaSize = Length + Length / 2;
	_WorkArea = new char[_WorkAreaSize];
    }

    void* EncodedData;
    size_t EncodedDataSize;
    _DataEncoder->Encode(Buffer, Length, EncodedData, EncodedDataSize);

    _ComponentFile->Write(EncodedData, EncodedDataSize);

    return Length;
}



TMushCompressedFile::TMushCompressedFile(TMushFile* ComponentFile, TMushDataCompressor* DataCompressor)
: TMushDecoratedFile(ComponentFile)
{
    if (DataCompressor) {
	_DataCompressor = DataCompressor;
	_IsOwningCompressor = false;
    }
    else {
	_DataCompressor = new TMushZlibDataCompressor();
	_IsOwningCompressor = true;
    }

    _TargetBufferSize = 4096;
    _TargetBuffer = new char[_TargetBufferSize];

    _IsFirstRead = true;
    _IsOutputPending = false;
}

TMushCompressedFile::~TMushCompressedFile()
{
    TMushCompressedFile::Close();
    
    if (_IsOwningCompressor) {
	delete _DataCompressor;
    }

    delete[] _TargetBuffer;
}

void TMushCompressedFile::Close(void)
{
    if (_IsOutputPending) {
	void* Buffer = 0;
	size_t Length = 0;
	bool IsLast = true;
	size_t CompressedSize;

	_DataCompressor->SetCompressInput(Buffer, Length, IsLast);
	while ((CompressedSize = _DataCompressor->Compress(_TargetBuffer, _TargetBufferSize)) > 0) {
	    _ComponentFile->Write(_TargetBuffer, CompressedSize);
	}

	_IsOutputPending = false;
    }

    TMushDecoratedFile::Close();
}

int TMushCompressedFile::Read(void* Buffer, size_t Length) throw(TSystemCallException)
{
    size_t InputSize;
    size_t DecompressedSize;

    if (_IsFirstRead) {
	DecompressedSize = 0;
    }
    else {
	DecompressedSize = _DataCompressor->Decompress(Buffer, Length);
    }

    if (DecompressedSize == 0) {
	InputSize = _ComponentFile->Read(_TargetBuffer, _TargetBufferSize);
	if (InputSize == 0) {
	    return 0;
	}

	bool IsLast = false;
	_DataCompressor->SetDecompressInput(_TargetBuffer, InputSize, IsLast);
	_IsFirstRead = false;

	return Read(Buffer, Length);
    }
    
    return DecompressedSize;
}

int TMushCompressedFile::Write(void* Buffer, size_t Length) throw(TSystemCallException)
{
    bool IsLast = false;
    _DataCompressor->SetCompressInput(Buffer, Length, IsLast);

    size_t CompressedSize;
    while ((CompressedSize = _DataCompressor->Compress(_TargetBuffer, _TargetBufferSize)) > 0) {
	_ComponentFile->Write(_TargetBuffer, CompressedSize);
    }
    _IsOutputPending = true;

    return Length;
}

Int64 TMushCompressedFile::Position(void) throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushCompressedFile::Position()", "function not implemented yet"
    );
}

void TMushCompressedFile::SeekTo(Int64 Offset) throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushCompressedFile::SeekTo()", "function not implemented yet"
    );
}

void TMushCompressedFile::SeekBy(Int64 Offset) throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushCompressedFile::SeekBy()", "function not implemented yet"
    );
}

long TMushCompressedFile::Size(void) const throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushCompressedFile::Size()", "function not implemented yet"
    );
}
