/* MushFileSystem.cc */
/* Created by Enomoto Sanshiro on 17 July 2001. */
/* Last updated by Enomoto Sanshiro on 26 April 2002. */


#include <string>
#include <fstream>
#include <sys/stat.h>
#include <unistd.h>
#include "MushDefs.hh"
#include "MushCheckSum.hh"
#include "MushFileSystem.hh"

#if IS_SYSVFS_AVAILABLE
#include <sys/vfs.h>
#endif

using namespace std;


TMushFileSystem::TMushFileSystem(const string& Path)
{
    if (! Path.empty()) {
	_Path = Path;
    }
    else {
	_Path = TMushFileSystem::CurrentDirectory();
    }

    _BlockSize = -1;
}

TMushFileSystem::~TMushFileSystem()
{
}

void TMushFileSystem::ChangeDirectory(const string& Path) throw(TSystemCallException)
{
    if (chdir(Path.c_str()) < 0) {
	throw TSystemCallException(
	    "TMushFileSystem::ChangeDirectory(): chdir(2)"
	);
    }
}

string TMushFileSystem::CurrentDirectory(void) throw(TSystemCallException)
{
    char Buffer[256];
    if (getcwd(Buffer, sizeof(Buffer)) == NULL) {
	throw TSystemCallException(
	    "TMushFileSystem::CurrentDirectory(): getcwd(2)"
	);
    }

    return string(Buffer);
}

int TMushFileSystem::MakeDirectory(const string& Path, int AccessMode) throw(TSystemCallException)
{
    int Result = 1;

    if (mkdir(Path.c_str(), AccessMode) < 0) {
	TMushFileAttribute Attribute(Path);
	if (Attribute.Exists() && Attribute.IsDirectory()) {
	    // directory already exists //
	    return 0;
	}

	string ParentDirectoryName = Attribute.PathName();
	if (
	    ! ParentDirectoryName.empty() &&
	    ! TMushFileAttribute(ParentDirectoryName).Exists()
	){
	    // parent directory does not exist //
	    MakeDirectory(ParentDirectoryName, AccessMode);
	    Result = MakeDirectory(Path, AccessMode);
	}
	else {
	    throw TSystemCallException(
		"TMushFileSystem::MakeDirectory(): mkdir(2): " + Path
	    );
	}
    }

    return Result;
}

unsigned TMushFileSystem::SetUMask(unsigned UMask)
{
    return umask(UMask);
}

void TMushFileSystem::Unlink(const string& FileName) throw(TSystemCallException)
{
    if (unlink(FileName.c_str()) < 0) {
	throw TSystemCallException("TMushFileSystem::Unlink(): unlink(2)");
    }
}

long TMushFileSystem::FileSystemType(void) throw(TSystemCallException)
{
#if IS_SYSVFS_AVAILABLE
    struct statfs FileSystemStat;
    if (statfs((char*) _Path.c_str(), &FileSystemStat) < 0) {
        throw TSystemCallException(
            "TMushFileSystem::FileSystemType(): statfs(2)"
	);
    }
    return FileSystemStat.f_type;
#else
    throw TSystemCallException(
	"TMushFileSystem::FileSystemType(): statfs(2)",
	"function not available"
    );
#endif
}

long TMushFileSystem::BlockSize(void) throw(TSystemCallException)
{
    if (_BlockSize < 0) {
#if IS_SYSVFS_AVAILABLE
	struct statfs FileSystemStat;
	if (statfs((char*) _Path.c_str(), &FileSystemStat) < 0) {
	    throw TSystemCallException(
		"TMushFileSystem::BlockSize(): statfs(2)"
	    );
	}
	_BlockSize = FileSystemStat.f_bsize;
#else
	throw TSystemCallException(
	    "TMushFileSystem::BlockSize(): statfs(2)",
	    "function not available"
	);
#endif
    }

    return _BlockSize;
}

long TMushFileSystem::TotalNumberOfBlocks(void) throw(TSystemCallException)
{
#if IS_SYSVFS_AVAILABLE
    struct statfs FileSystemStat;
    if (statfs((char*) _Path.c_str(), &FileSystemStat) < 0) {
	throw TSystemCallException(
	    "TMushFileSystem::TotalNumberOfBlocks(): statfs(2)"
	);
    }
    return FileSystemStat.f_blocks;
#else
    throw TSystemCallException(
	"TMushFileSystem::TotalNumberOfBlocks(): statfs(2)",
	"function not available"
    );
#endif
}

long TMushFileSystem::NumberOfFreeBlocks(void) throw(TSystemCallException)
{
#if IS_SYSVFS_AVAILABLE
    struct statfs FileSystemStat;
    if (statfs((char*) _Path.c_str(), &FileSystemStat) < 0) {
	throw TSystemCallException(
	    "TMushFileSystem::NumberOfFreeBlocks(): statfs(2)"
	);
    }
    return FileSystemStat.f_bfree;
#else
    throw TSystemCallException(
	"TMushFileSystem::NumberOfFreeBlocks(): statfs(2)",
	"function not available"
    );
#endif
}

long TMushFileSystem::NumberOfAvailableBlocks(void) throw(TSystemCallException)
{
#if IS_SYSVFS_AVAILABLE
    struct statfs FileSystemStat;
    if (statfs((char*) _Path.c_str(), &FileSystemStat) < 0) {
	throw TSystemCallException(
	    "TMushFileSystem::NumberOfAvailableBlocks(): statfs(2)"
	);
    }
    return FileSystemStat.f_bavail;
#else
    throw TSystemCallException(
	"TMushFileSystem::NumberOfAvailableBlocks(): statfs(2)",
	"function not available"
    );
#endif
}

long TMushFileSystem::TotalNumberOfNodes(void) throw(TSystemCallException)
{
#if IS_SYSVFS_AVAILABLE
    struct statfs FileSystemStat;
    if (statfs((char*) _Path.c_str(), &FileSystemStat) < 0) {
	throw TSystemCallException(
	    "TMushFileSystem::TotalNumberOfNodes(): statfs(2)"
	);
    }
    return FileSystemStat.f_files;
#else
    throw TSystemCallException(
	"TMushFileSystem::TotalNumberOfNodes(): statfs(2)",
	"function not available"
    );
#endif
}

long TMushFileSystem::NumberOfFreeNodes(void) throw(TSystemCallException)
{
#if IS_SYSVFS_AVAILABLE
    struct statfs FileSystemStat;
    if (statfs((char*) _Path.c_str(), &FileSystemStat) < 0) {
	throw TSystemCallException(
	    "TMushFileSystem::NumberOfFreeNodes(): statfs(2)"
	);
    }
    return FileSystemStat.f_ffree;
#else
    throw TSystemCallException(
	"TMushFileSystem::NumberOfFreeNodes(): statfs(2)",
	"function not available"
    );
#endif
}



TMushFileAttribute::TMushFileAttribute(const string& FilePathName)
{
    _FilePathName = FilePathName;
}

TMushFileAttribute::~TMushFileAttribute()
{
}

long TMushFileAttribute::Size(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException("TMushFileAttribute::Size(): stat(2)");
    }

    return FileStat.st_size;
}

long TMushFileAttribute::AccessMode(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::AccessMode(): stat(2)"
	);
    }

    return FileStat.st_mode & 0777;
}

long TMushFileAttribute::OwnerUserId(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::OwnerUserId(): stat(2)"
	);
    }

    return FileStat.st_uid;
}

long TMushFileAttribute::OwnerGroupId(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::OwnerGroupId(): stat(2)"
	);
    }

    return FileStat.st_gid;
}

long TMushFileAttribute::LastAccessTime(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::LastAccessTime(): stat(2)"
	);
    }

    return FileStat.st_atime;
}

long TMushFileAttribute::LastModificationTime(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::LastModificationTime(): stat(2)"
	);
    }

    return FileStat.st_mtime;
}

long TMushFileAttribute::LastChangeTime(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::LastChangeTime(): stat(2)"
	);
    }

    return FileStat.st_ctime;
}

bool TMushFileAttribute::IsFile(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException("TMushFileAttribute::IsFile(): stat(2)");
    }

    return S_ISREG(FileStat.st_mode);
}

bool TMushFileAttribute::IsDirectory(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::IsDirectory(): stat(2)"
	);
    }

    return S_ISDIR(FileStat.st_mode);
}

bool TMushFileAttribute::IsSymbolicLink(void) throw(TSystemCallException)
{
    struct stat FileStat;
    if (lstat(_FilePathName.c_str(), &FileStat) < 0) {
	throw TSystemCallException(
	    "TMushFileAttribute::IsSymbolicLink(): stat(2)"
	);
    }

    return S_ISLNK(FileStat.st_mode);
}

bool TMushFileAttribute::Exists(void)
{
    return (access(_FilePathName.c_str(), F_OK) == 0);
}

bool TMushFileAttribute::IsReadable(void)
{
    return (access(_FilePathName.c_str(), R_OK) == 0);
}

bool TMushFileAttribute::IsWritable(void)
{
    return (access(_FilePathName.c_str(), W_OK) == 0);
}

bool TMushFileAttribute::IsExecutable(void)
{
    return (access(_FilePathName.c_str(), X_OK) == 0);
}

string TMushFileAttribute::PathName(void)
{
    int End = _FilePathName.find_last_of('/');
    if (End >= 0) {
	return _FilePathName.substr(0, End + 1);
    }
    else {
	return string("");
    }
}

string TMushFileAttribute::FileName(void)
{
    int Start = _FilePathName.find_last_of('/');
    if (Start >= 0) {
	return _FilePathName.substr(Start + 1, string::npos);
    }
    else {
	return _FilePathName;
    }
}

string TMushFileAttribute::FileRootName(void)
{
    string EntireFileName = FileName();
    int End = EntireFileName.find_last_of('.');

    if (End >= 0) {
	return EntireFileName.substr(0, End);
    }
    else {
	return EntireFileName;
    }
}

string TMushFileAttribute::Extension(void)
{
    string EntireFileName = FileName();
    int Start = EntireFileName.find_last_of('.');
    if (Start >= 0) {
	return EntireFileName.substr(Start + 1, string::npos);
    }
    else {
	return string("");
    }
}

unsigned long TMushFileAttribute::CheckSum(void) throw(TSystemCallException)
{
    if (! IsReadable()) {
	throw TSystemCallException(
	    "TMushFileAttribute::CheckSum()",
	    "unable to open file: " + _FilePathName
	);
    }

    if (IsDirectory()) {
	throw TSystemCallException(
	    "TMushFileAttribute::CheckSum()",
	    _FilePathName + " is a directory"
	);
    }

    ifstream File(_FilePathName.c_str());
    if (! File) {
	throw TSystemCallException(
	    "TMushFileAttribute::CheckSum()",
	    "unable to open file: " + _FilePathName
	);
    }

    TMushCrc32CheckSum CheckSum;

    int Size;
    char Buffer[4096];
    while (File.read(Buffer, sizeof(Buffer)), (Size = File.gcount()) > 0) {
	CheckSum.ReadData(Buffer, Size);
    }

    // include file size (to keep compatibility with cksum(1))
    int FileSize = CheckSum.DataLength();
    for (int SizeData = FileSize; SizeData != 0; SizeData >>= 8) {
	char Byte = SizeData & 0xff;
	CheckSum.ReadData(&Byte, 1);
    }

    unsigned long Result = 0;
    for (int i = 0; i < CheckSum.ResultLength(); i++) {
	Result |= CheckSum.ByteValueAt(i) << (8 * i);
    }

    return Result;
}
