/* MushFramedFile.cc */
/* Created by Enomoto Sanshiro on 25 October 1999. */
/* Last updated by Enomoto Sanshiro on 30 April 2002. */


#include "MushFile.hh"
#include "MushDecoratedFile.hh"
#include "MushDataEncoder.hh"
#include "MushFramedFile.hh"


using namespace std;
using namespace mush;


TMushFramedFile::TMushFramedFile(TMushFile *ComponentFile, bool IsIndexEnabled)
: TMushDecoratedFile(ComponentFile)
{
    _IsIndexEnabled = IsIndexEnabled;
    _Index = 0;

    _NextPacketSize = -1;
    _CurrentPosition = 0;
}

TMushFramedFile::~TMushFramedFile()
{
    TMushFramedFile::Close();

    delete _Index;
}

void TMushFramedFile::OpenReadMode(void) throw(TSystemCallException)
{
    TMushDecoratedFile::OpenReadMode();
    _IsReadMode = true;
}

void TMushFramedFile::OpenWriteMode(unsigned AccessMode, bool IsExclusive) throw(TSystemCallException)
{
    TMushDecoratedFile::OpenWriteMode(AccessMode, IsExclusive);
    _IsReadMode = false;

    if (_IsIndexEnabled) {
	_Index = new TMushFramedFileIndex(this);
	_Index->OpenWriteMode(AccessMode);
    }
}

void TMushFramedFile::OpenAppendMode(unsigned AccessMode) throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushFramedFile::Open()", 
	"opening in 'append' mode is not implemented yet"
    ); 
}

void TMushFramedFile::OpenReadWriteMode(unsigned AccessMode) throw(TSystemCallException)
{
    throw TSystemCallException(
	"TMushFramedFile::Open()", 
	"opening in 'append' mode is not implemented yet"
    ); 
}

void TMushFramedFile::Close(void)
{
    if (_Index) {
	_Index->Close();
    }

    TMushDecoratedFile::Close();
}

int TMushFramedFile::Write(void *Buffer, size_t Length) throw(TSystemCallException)
{
    if (_Index) {
	_Index->AddNext(_ComponentFile->Position(), Length);
    }

    for (int Byte = 0; Byte < _HeaderSize; Byte++) {
	_HeaderBuffer[Byte] = (Length >> (8 * Byte)) & 0x000000ff;
    }
    _ComponentFile->WriteBlock(_HeaderBuffer, _HeaderSize);

    _ComponentFile->WriteBlock(Buffer, Length);

    _CurrentPosition++;

    return Length;
}

int TMushFramedFile::NextPacketSize(void) throw(TSystemCallException)
{
    if (_NextPacketSize >= 0) {
	return _NextPacketSize;
    }

    int ReadLength = _ComponentFile->ReadBlock(_HeaderBuffer, _HeaderSize);

    if (ReadLength == 0) {
	_NextPacketSize = -1;
	return 0;
    }
    else if (ReadLength != _HeaderSize) {
	throw TSystemCallException(
	    "TMushFramedFile::NextPacketSize()", "unexpected end-of-file"
	);
    }
    else {
	_NextPacketSize = 0;
	for (int Byte = 0; Byte < _HeaderSize; Byte++) {
	    _NextPacketSize += _HeaderBuffer[Byte] << (8 * Byte);
	}
    }

    return _NextPacketSize;
}

int TMushFramedFile::Read(void *Buffer, size_t Length) throw(TSystemCallException)
{
    if (_NextPacketSize < 0) {
	NextPacketSize();
    }

    if (_NextPacketSize <= 0) {
	_NextPacketSize = -1;
	return 0;
    }
    else if (_NextPacketSize > (int) Length) {
	throw TSystemCallException(
	    "TMushFramedFile::Read()", "buffer is too small"
	);
    }
   
    int ReadLength = _ComponentFile->ReadBlock(Buffer, _NextPacketSize, true);

    if (ReadLength != _NextPacketSize) {
	throw TSystemCallException(
	    "TMushFramedFile::Read()", "unexpected end-of-file"
	);
    }

    _CurrentPosition++;
    _NextPacketSize = -1;

    return ReadLength;
}

int TMushFramedFile::SkipNext(void) throw(TSystemCallException)
{
    if (_NextPacketSize < 0) {
	NextPacketSize();
    }

    if (_NextPacketSize <= 0) {
	_NextPacketSize = -1;
	return 0;
    }
   
    int SkippedSize = _NextPacketSize;
    try {
	_ComponentFile->SeekBy(SkippedSize);
    }
    catch (TSystemCallException &e) {
	throw TSystemCallException("TMushFramedFile::SkipNext()", e.Message());
    }

    _CurrentPosition++;
    _NextPacketSize = -1;

    return SkippedSize;
}

Int64 TMushFramedFile::Position(void) throw(TSystemCallException)
{
    return _CurrentPosition;
}

void TMushFramedFile::SeekTo(Int64 Offset) throw(TSystemCallException)
{
    if (! _IsReadMode) {
	throw TSystemCallException(
	    "TMushFramedFile::SeekTo()",
	    "seeking is supported only in ReadOnly mode"
	);
    }

    if (! _IsIndexEnabled) {
	throw TSystemCallException(
	    "TMushFramedFile::SeekTo()", 
	    "function not enabled (index is required for random access)"
	);
    }

    if (_Index == 0) {
	_Index = new TMushFramedFileIndex(this);
	_Index->OpenReadMode();
    }

    _ComponentFile->SeekTo(_Index->OffsetOf(Offset));
    _CurrentPosition = Offset;
}

void TMushFramedFile::SeekBy(Int64 Offset) throw(TSystemCallException)
{
    return SeekTo(_CurrentPosition + Offset);
}

long TMushFramedFile::Size(void) const throw(TSystemCallException)
{
    if (! _IsReadMode) {
	throw TSystemCallException(
	    "TMushFramedFile::Size()", 
	    "function not supported in Write mode"
	);
    }

    if (! _IsIndexEnabled) {
	throw TSystemCallException(
	    "TMushFramedFile::Size()", 
	    "function not enabled (index is required)"
	);
    }

    if (_Index == 0) {
	_Index = new TMushFramedFileIndex(this);
	_Index->OpenReadMode();
    }

    return _Index->NumberOfPackets();
}


#define FullVersion(Major, Minor) (Major << 16 | Minor)
#define MajorVersion(FullVersion) ((FullVersion >> 16) & 0xffffffff)
#define MinorVersion(FullVersion) ((FullVersion >> 0) & 0xffffffff)

static const UInt64 MagicNumber = 0x6502488383ull;
static const UInt64 Version = FullVersion(1, 1);

static const UInt64 ShortOffsetMax = 0x0001 << 16;
static const UInt64 MediumOffsetMax = ((UInt64) 0x0001) << 32;
static const UInt64 LongOffsetMax = ((UInt64) 0x0001) << 63;

static const size_t ShortOffsetLength = sizeof(UInt16);
static const size_t MediumOffsetLength = sizeof(UInt32);
static const size_t LongOffsetLength = sizeof(UInt64);


TMushFramedFileIndex::TMushFramedFileIndex(const TMushFramedFile* FramedFile)
{
    _FileName = FramedFile->Name();
    _IndexFile = new TMushFile(_FileName + ".index");

    _BufferSize = 256;  // should be a small number
    _Buffer = new UInt8[_BufferSize];

    _CurrentIndex = 0;
}

TMushFramedFileIndex::~TMushFramedFileIndex()
{
    delete _IndexFile;
    delete[] _Buffer;
}

void TMushFramedFileIndex::AddNext(TFileOffset Offset, size_t Size) throw(TSystemCallException)
{
    if ((_BufferIndex >= _BufferSize) || (Offset > _CurrentOffsetMax)) {
	if (_BufferIndex > 0){
	    _IndexFile->WriteBlock(_Buffer, _BufferIndex);
	    _BufferIndex = 0;
	}
    }

    if (Offset > _CurrentOffsetMax) {
	UInt64 AlignMask = 0x0007;
	UInt64 Position = (_IndexFile->Position() & ~AlignMask) + AlignMask+1;
	_IndexFile->SeekTo(Position);

	if (_CurrentOffsetMax == ShortOffsetMax) {
	    _MediumOffsetBase = Position;
	    _MediumOffsetStartIndex = _CurrentIndex;
	    _CurrentOffsetMax = MediumOffsetMax;
	    _CurrentOffsetLength = MediumOffsetLength;
	}
	else if (_CurrentOffsetMax == MediumOffsetMax) {
	    _CurrentOffsetMax = LongOffsetMax;
	    _LongOffsetBase = Position;
	    _LongOffsetStartIndex = _CurrentIndex;
	    _CurrentOffsetLength = LongOffsetLength;
	}
	else {
	    throw TSystemCallException(
		"TMushFramedFileIndex::AddNext()", "too large file offset"
	    );
	}
    }

    _CurrentIndex++;
    if (_MaxPacketSize < Size) {
	_MaxPacketSize = Size;
    }

    for (unsigned Byte = 0; Byte < _CurrentOffsetLength; Byte++) {
	_Buffer[_BufferIndex++] = (Offset >> (8 * Byte)) & 0x00ff;
    }
}

TMushFramedFileIndex::TFileOffset TMushFramedFileIndex::OffsetOf(UInt64 Index) throw(TSystemCallException)
{
    UInt64 IndexOffset;
    size_t OffsetLength;

    if (Index < _MediumOffsetStartIndex) {
	OffsetLength = ShortOffsetLength;
	IndexOffset = _ShortOffsetBase + OffsetLength * (Index - _ShortOffsetStartIndex);
    }
    else if (Index < _LongOffsetStartIndex) {
	OffsetLength = MediumOffsetLength;
	IndexOffset = _MediumOffsetBase + OffsetLength * (Index - _MediumOffsetStartIndex);
    }
    else {
	OffsetLength = LongOffsetLength;
	IndexOffset = _LongOffsetBase + OffsetLength * (Index - _LongOffsetStartIndex);
    }

    if (Index >= _NumberOfPackets) {
	throw TSystemCallException(
	    "TMushFramedFileIndex::OffsetOf()", "too large index"
	);
    }

    _IndexFile->SeekTo(IndexOffset);
    if (_IndexFile->ReadBlock(_Buffer, OffsetLength) != (int) OffsetLength) {
	throw TSystemCallException(
	    "TMushFramedFileIndex::OffsetOf()", "unable to read index file"
	);
    }

    TFileOffset Offset = 0;
    for (unsigned Byte = 0; Byte < OffsetLength; Byte++) {
	Offset += _Buffer[Byte] << (8 * Byte);
    }

    return Offset;
}

void TMushFramedFileIndex::OpenWriteMode(int AccessMode) throw(TSystemCallException)
{
    _IndexFile->OpenWriteMode(AccessMode);

    _ShortOffsetStartIndex = 0;
    _ShortOffsetBase = sizeof(UInt64) * _NumberOfPreambleWords;

    _MediumOffsetStartIndex = ~((UInt64) 0);
    _MediumOffsetBase =  ~((UInt64) 0);

    _LongOffsetStartIndex = ~((UInt64) 0);
    _LongOffsetBase = ~((UInt64) 0);

    _NumberOfPackets = 0;
    _MaxPacketSize = 0;

    _CurrentOffsetLength = ShortOffsetLength;
    _CurrentOffsetMax = ShortOffsetMax;

    _BufferIndex = 0;
    _CurrentIndex = 0;

    _IndexFile->SeekTo(_ShortOffsetBase);
}

void TMushFramedFileIndex::OpenReadMode(void) throw(TSystemCallException)
{
    _IndexFile->OpenReadMode();
    ReadPreamble();
}

void TMushFramedFileIndex::Close(void) throw(TSystemCallException)
{
    if (_CurrentIndex > 0) {
	if (_BufferIndex > 0) {
	    _IndexFile->WriteBlock(_Buffer, _BufferIndex);
	    _BufferIndex = 0;
	}

	_NumberOfPackets = _CurrentIndex;

	WritePreamble();
    }
}

void TMushFramedFileIndex::WritePreamble(void) throw(TSystemCallException)
{
    UInt64 PreambleBuffer[_NumberOfPreambleWords];

    PreambleBuffer[MagicNumberOffset] = MagicNumber;
    PreambleBuffer[VersionOffset] = Version;
    PreambleBuffer[FormatFlagsOffset] = 0x0000;

    PreambleBuffer[NumberOfPacketsOffset] = _NumberOfPackets;
    PreambleBuffer[MaxPacketSizeOffset] = _MaxPacketSize;

    PreambleBuffer[ShortOffsetStartIndexOffset] = _ShortOffsetStartIndex;
    PreambleBuffer[ShortOffsetBaseOffset] = _ShortOffsetBase;
    PreambleBuffer[MediumOffsetStartIndexOffset] = _MediumOffsetStartIndex;
    PreambleBuffer[MediumOffsetBaseOffset] = _MediumOffsetBase;
    PreambleBuffer[LongOffsetStartIndexOffset] = _LongOffsetStartIndex;
    PreambleBuffer[LongOffsetBaseOffset] = _LongOffsetBase;

    _IndexFile->SeekTo(0);
    _IndexFile->WriteBlock(PreambleBuffer, sizeof(PreambleBuffer));
}

void TMushFramedFileIndex::ReadPreamble(void) throw(TSystemCallException)
{
    UInt64 PreambleBuffer[_NumberOfPreambleWords];
    size_t PreambleSize = sizeof(PreambleBuffer);

    _IndexFile->SeekTo(0);
    if (_IndexFile->ReadBlock(PreambleBuffer, PreambleSize) != (int) PreambleSize) {
	throw TSystemCallException(
	    "TMushFramedFileIndex::ReadPreamble()",
	    "unexpected end-of-file"
	);
    }

    UInt64 ThisMagicNumber = PreambleBuffer[MagicNumberOffset];
    if (ThisMagicNumber != MagicNumber) {
        bool ReverseBytes = true;
        TMushBinaryDataEncoder DataEncoder(sizeof(UInt64), ReverseBytes);
	DataEncoder.DecodeOn(PreambleBuffer, sizeof(PreambleBuffer));
	ThisMagicNumber = PreambleBuffer[MagicNumberOffset];
    }

    if (ThisMagicNumber != MagicNumber) {
	throw TSystemCallException(
	    "TMushFramedFileIndex::ReadPreamble()",
	    "bad index file (inconsistent magic number)"
	);
    }

    UInt64 ThisVersion = PreambleBuffer[VersionOffset];
    if (MajorVersion(ThisVersion) > MajorVersion(Version)) {
	throw TSystemCallException(
	    "TMushFramedFileIndex::ReadPreamble()",
	    "version number mismatch: use newer version of the library"
	);
    }

    _FormatFlags = PreambleBuffer[FormatFlagsOffset];
    _NumberOfPackets = PreambleBuffer[NumberOfPacketsOffset];
    _MaxPacketSize = PreambleBuffer[MaxPacketSizeOffset];
    _ShortOffsetStartIndex = PreambleBuffer[ShortOffsetStartIndexOffset];
    _ShortOffsetBase = PreambleBuffer[ShortOffsetBaseOffset];
    _MediumOffsetStartIndex = PreambleBuffer[MediumOffsetStartIndexOffset];
    _MediumOffsetBase = PreambleBuffer[MediumOffsetBaseOffset];
    _LongOffsetStartIndex = PreambleBuffer[LongOffsetStartIndexOffset];
    _LongOffsetBase = PreambleBuffer[LongOffsetBaseOffset];
}

UInt64 TMushFramedFileIndex::NumberOfPackets(void) const
{
    return _NumberOfPackets;
}

UInt64 TMushFramedFileIndex::MaxPacketSize(void) const
{
    return _MaxPacketSize;
}
