/* MushIpc.cc */
/* Created by Enomoto Sanshiro on 21 January 1998. */
/* Last updated by Enomoto Sanshiro on 3 July 1998. */


#include <iostream>
#include <string>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/shm.h>
#include "MushIpc.hh"

using namespace std;


TMushSystemVIpc::TMushSystemVIpc(int ProjectId, const string& ProjectPath)
{
    _ProjectId = ProjectId;

    if (ProjectPath.empty()) {
	char CurrentWorkingDirectry[256];
	getcwd(CurrentWorkingDirectry, sizeof(CurrentWorkingDirectry) - 1);
	_ProjectPath = CurrentWorkingDirectry;
    }
    else {
	_ProjectPath = ProjectPath;
    }

    _IpcKey = -1;
    _IpcId = -1;
}

TMushSystemVIpc::~TMushSystemVIpc()
{
}

int TMushSystemVIpc::ProjectId()
{
    return _ProjectId;
}

string TMushSystemVIpc::ProjectPath(void)
{
    return _ProjectPath;
}

key_t TMushSystemVIpc::IpcKey(void) throw(TSystemCallException)
{
    if (_IpcKey < 0) {
	char ProjectPath[256];
	strncpy(ProjectPath, _ProjectPath.c_str(), sizeof(ProjectPath) - 1);
	if ((_IpcKey = ftok(ProjectPath, _ProjectId)) < 0) {
	    throw TSystemCallException(
		"TMushSystemVIpc::IpcKey(): ftok(3C)"
	    );
	}
    }

    return _IpcKey;
}

int TMushSystemVIpc::IpcId(void) throw(TSystemCallException)
{
    if (_IpcId < 0) {
	throw TSystemCallException(
	    "TMushSystemVIpc::IpcId()", "IPC is not attached"
	);
    }

    return _IpcId;
}



TMushMessageQueue::TMushMessageQueue(int ProjectId, const string& ProjectPath)
: TMushSystemVIpc(ProjectId, ProjectPath)
{
    _Capacity = -1;
    SetAddress();
}

TMushMessageQueue::~TMushMessageQueue()
{
}

int TMushMessageQueue::Send(const TMushMessageQueuePacket *MessageQueuePacket, int MessageLength) throw(TSystemCallException)
{
    int MessageFlag = 0;
    int Result = msgsnd(
        _IpcId, (void*) MessageQueuePacket, MessageLength, MessageFlag
    );

    int SentLength = MessageLength;
    if (Result < 0) {
	if ((errno == EINTR) || (errno == EAGAIN)) {
	    SentLength = 0;
	}
	else {
	    throw TSystemCallException(
		"TMushMessageQueue::Send(): msgsnd(2)"
	    );
	}
    }

    return SentLength;
}

int TMushMessageQueue::Receive(TMushMessageQueuePacket *MessageQueuePacket, int MaxMessageLength, bool NoWait) throw(TSystemCallException)
{
    long MessageType = _Address;
    int MessageFlag = (NoWait) ? IPC_NOWAIT : 0;

    int Result = msgrcv(
        _IpcId, 
        (void*) MessageQueuePacket, MaxMessageLength, MessageType, MessageFlag
    );

    int ReceivedLength = Result;
    if (Result < 0) {
	if ((errno == EINTR) || (errno == EAGAIN) || (errno == ENOMSG)) {
	    ReceivedLength = 0;
	}
	else {
	    throw TSystemCallException(
		"TMushMessageQueue::Receive(): msgrcv(2)"
	    );
	}
    }

    return ReceivedLength;
}

int TMushMessageQueue::QueueCapacity(void) throw(TSystemCallException)
{
    if (_Capacity < 0) {
	struct msqid_ds StatusBuffer;
	if (msgctl(_IpcId, IPC_STAT, &StatusBuffer) < 0) {
	    throw TSystemCallException(
		"TMushMessageQueue::QueueCapacity(): msgctl(2)"
	    );
	}
	
	_Capacity = StatusBuffer.msg_qbytes;
    }

    return _Capacity;
}

int TMushMessageQueue::NumberOfPendingPackets(void) throw(TSystemCallException)
{
    struct msqid_ds StatusBuffer;
    if (msgctl(_IpcId, IPC_STAT, &StatusBuffer) < 0) {
	throw TSystemCallException(
	    "TMushMessageQueue::NumberOfPendingPackets(): msgctl(2)"
	);
    }

    return StatusBuffer.msg_qnum;
}

void TMushMessageQueue::SetAddress(long Address)
{
    if (Address < 0) {
	_Address = getpid();
    }
    else {
	_Address = Address;
    }
}

long TMushMessageQueue::Address(void)
{
    return _Address;
}



TMushServerMessageQueue::TMushServerMessageQueue(int ProjectId, const string& ProjectPath, unsigned AccessPermission, bool IsSelfish) throw(TSystemCallException)
: TMushMessageQueue(ProjectId, ProjectPath)
{
    _IpcId = msgget(IpcKey(), IPC_CREAT | IPC_EXCL | AccessPermission);

    if ((_IpcId < 0) && (errno == EEXIST) && IsSelfish) {
	int OldIpcId = msgget(IpcKey(), 0);
	if (OldIpcId >= 0) {
	    struct msqid_ds StatusBuffer;
	    msgctl(OldIpcId, IPC_RMID, &StatusBuffer);
	    _IpcId = msgget(IpcKey(), IPC_CREAT | IPC_EXCL | AccessPermission);
	}
    }

    if (_IpcId < 0) {
	throw TSystemCallException(
            "TMushServerMessageQueue::TMushServerMessageQueue(): msgget(2)"
        );
    }
}

TMushServerMessageQueue::~TMushServerMessageQueue()
{
    if (_IpcId >= 0) {
	struct msqid_ds StatusBuffer;
	msgctl(_IpcId, IPC_RMID, &StatusBuffer);
    }
}



TMushClientMessageQueue::TMushClientMessageQueue(int ProjectId, const string& ProjectPath) throw(TSystemCallException)
: TMushMessageQueue(ProjectId, ProjectPath)
{
    _IpcId = msgget(IpcKey(), 0);
    if (_IpcId < 0) {
	throw TSystemCallException(
            "TMushClientMessageQueue::TMushClientMessageQueue(): msgget(2)"
        );
    }
}

TMushClientMessageQueue::~TMushClientMessageQueue()
{
}



TMushSharedMemory::TMushSharedMemory(int ProjectId, const string& ProjectPath)
: TMushSystemVIpc(ProjectId, ProjectPath)
{
    _SharedMemoryAddress = 0;
}

TMushSharedMemory::~TMushSharedMemory()
{
}

volatile void* TMushSharedMemory::BasePointer(void) 
{
    return _SharedMemoryAddress;
}

void TMushSharedMemory::Attach(void) throw(TSystemCallException)
{
    void *Address = 0; // MapAddress will be assigned by kernel.
    int Flags = 0;
    _SharedMemoryAddress = shmat(_IpcId, Address, Flags);
    if ((int) _SharedMemoryAddress == -1) {
	_SharedMemoryAddress = 0;
	throw TSystemCallException("TMushSharedMemory::Attach(): shmat(2)");
    }
}

void TMushSharedMemory::Detach(void)
{
    if (_SharedMemoryAddress != 0) {
	shmdt((char *) _SharedMemoryAddress);
    }
}



TMushServerSharedMemory::TMushServerSharedMemory(size_t Size, int ProjectId, const string& ProjectPath, unsigned AccessPermission, bool IsSelfish) throw(TSystemCallException)
: TMushSharedMemory(ProjectId, ProjectPath)
{
    _IpcId = shmget(IpcKey(), Size, IPC_CREAT | IPC_EXCL | AccessPermission);

    if ((_IpcId < 0) && (errno == EEXIST) && IsSelfish) {
	int OldIpcId = shmget(IpcKey(), 0, 0);
	if (OldIpcId >= 0) {
	    struct shmid_ds StatusBuffer;
	    shmctl(OldIpcId, IPC_RMID, &StatusBuffer);
	    _IpcId = shmget(IpcKey(), Size, IPC_CREAT | IPC_EXCL | AccessPermission);
	}
    }

    if (_IpcId < 0) {
	throw TSystemCallException(
            "TMushServerSharedMemory::TMushServerSharedMemory(): shmget(2)"
        );
    }

    Attach();
}

TMushServerSharedMemory::~TMushServerSharedMemory()
{
    Detach();

    if (_IpcId >= 0) {
	struct shmid_ds StatusBuffer;
	shmctl(_IpcId, IPC_RMID, &StatusBuffer);
    }
}



TMushClientSharedMemory::TMushClientSharedMemory(int ProjectId, const string& ProjectPath) throw(TSystemCallException)
: TMushSharedMemory(ProjectId, ProjectPath)
{
    size_t Size = 0;
    _IpcId = shmget(IpcKey(), Size, 0);
    if (_IpcId < 0) {
	throw TSystemCallException(
            "TMushClientSharedMemory::TMushClientSharedMemory(): shmget(2)"
        );
    }

    Attach();
}

TMushClientSharedMemory::~TMushClientSharedMemory()
{
    Detach();
}
