/* MushMisc.cc */
/* Created by Enomoto Sanshiro on 28 May 1998. */
/* Last updated by Enomoto Sanshiro on 24 September 1998. */


#include <iostream>
#include <strstream>
#include <string>
#include <unistd.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/time.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include "MushMisc.hh"


using namespace std;


TMushDateTime::TMushDateTime(TTimeZone TimeZone)
{
    _TimeZone = TimeZone;
    Update();
}

TMushDateTime::TMushDateTime(long TimeValue, TTimeZone TimeZone)
{
    _TimeZone = TimeZone;
    
    _TimeValue.tv_sec = TimeValue;
    _TimeValue.tv_usec = 0;

    if (_TimeZone == TimeZone_Local) {
	_TimeStruct = *(localtime((time_t *) &_TimeValue.tv_sec));
    }
    else {
	_TimeStruct = *(gmtime((time_t *) &_TimeValue.tv_sec));
    }
}

TMushDateTime& TMushDateTime::Update(void)
{
    gettimeofday(&_TimeValue, 0);
    if (_TimeZone == TimeZone_Local) {
	_TimeStruct = *(localtime((time_t *) &_TimeValue.tv_sec));
    }
    else {
	_TimeStruct = *(gmtime((time_t *) &_TimeValue.tv_sec));
    }

    return *this;
}

long TMushDateTime::AsLong(void)
{
    return _TimeValue.tv_sec;
}

long TMushDateTime::PassedTime(void)
{
    static struct timeval CurrentTimeValue;
    gettimeofday(&CurrentTimeValue, 0);

    return CurrentTimeValue.tv_sec - _TimeValue.tv_sec;
}

long TMushDateTime::PassedTimeMicroSec(void)
{
    static struct timeval CurrentTimeValue;
    gettimeofday(&CurrentTimeValue, 0);

    long PassedSec = CurrentTimeValue.tv_sec - _TimeValue.tv_sec;
    long PassedUSec = CurrentTimeValue.tv_usec - _TimeValue.tv_usec;

    return (1000000 * PassedSec + PassedUSec);
}

string TMushDateTime::AsString(string TimeFormat)
{
    if (TimeFormat.empty()) {
	TimeFormat = "%d %b %Y %H:%M:%S %Z";
    }

    char Buffer[256];
    strftime(Buffer, sizeof(Buffer), TimeFormat.c_str(), &_TimeStruct);

    return Buffer;
}

long TMushDateTime::USec(void)
{
    return _TimeValue.tv_usec;
}

int TMushDateTime::Sec(void)
{
    return _TimeStruct.tm_sec;
}

int TMushDateTime::Min(void)
{
    return _TimeStruct.tm_min;
}

int TMushDateTime::Hour(void)
{
    return _TimeStruct.tm_hour;
}

int TMushDateTime::Day(void)
{
    return _TimeStruct.tm_mday;
}

int TMushDateTime::Month(void)
{
    return 1 + _TimeStruct.tm_mon;   // 1: Jan, 2: Feb, ...
}

int TMushDateTime::Year(void)
{
    return 1900 + _TimeStruct.tm_year;
}

int TMushDateTime::DayOfWeek(void)
{
    return _TimeStruct.tm_wday;   // 0: Sun, 1: Mon, ...
}

int TMushDateTime::SecInDay(void)
{
    return (_TimeStruct.tm_hour * 60 + _TimeStruct.tm_min) * 60 + _TimeStruct.tm_sec;
}

int TMushDateTime::DayInYear(void)
{
    return _TimeStruct.tm_yday;
}



TMushUser::TMushUser(void)
{
    _UserId = getuid();
}

TMushUser::TMushUser(long UserId) throw(TSystemCallException)
{
    struct passwd *PassWordEntry = getpwuid(UserId);
    if (PassWordEntry == 0) { 
	throw TSystemCallException(
            "TMushUser::TMushUser(): getpwuid(3C)",
            "No such user."
        );
    }

    _UserId = PassWordEntry->pw_uid;
}

TMushUser::TMushUser(const string& LoginName) throw(TSystemCallException)
{
    struct passwd *PassWordEntry = getpwnam(LoginName.c_str());
    if (PassWordEntry == 0) { 
	throw TSystemCallException(
            "TMushUser::TMushUser(): getpwnam(3C)",
            string("No such user: ") + LoginName
        );
    }

    _UserId = PassWordEntry->pw_uid;
}

long TMushUser::Id(void)
{
    return _UserId;
}

const char *TMushUser::Name(void)
{
    return getpwuid(_UserId)->pw_name;
}

long TMushUser::GroupId(void)
{
    return getpwuid(_UserId)->pw_gid;
}

const char *TMushUser::GroupName(void)
{
    return getgrgid(getpwuid(_UserId)->pw_gid)->gr_name;
}

const char *TMushUser::HomeDir(void)
{
    return getpwuid(_UserId)->pw_dir;
}

const char *TMushUser::Shell(void)
{
    return getpwuid(_UserId)->pw_shell;
}
    


TMushGroup::TMushGroup(void)
{
    _GroupId = getgid();
}

TMushGroup::TMushGroup(long GroupId) throw(TSystemCallException)
{
    _GroupId = GroupId;
}

TMushGroup::TMushGroup(const string& GroupName) throw(TSystemCallException)
{
    struct group* GroupEntry = getgrnam(GroupName.c_str());
    if (GroupEntry == 0) {
	throw TSystemCallException(
            "TMushGroup::TMushGroup(): getgrnam(3C)",
            "No such group: " + GroupName
        );
    }

    _GroupId = GroupEntry->gr_gid;
}

long TMushGroup::Id(void)
{
    return _GroupId;
}

const char* TMushGroup::Name(void)
{
    return getgrgid(_GroupId)->gr_name;
}



static struct utsname UtsName;
const char *TMushLocalHost::SystemName(void)
{
    uname(&UtsName);
    return UtsName.sysname;
}

const char *TMushLocalHost::SystemRelease(void)
{
    uname(&UtsName);
    return UtsName.release;
}

const char *TMushLocalHost::SystemVersion(void)
{
    uname(&UtsName);
    return UtsName.version;
}

const char *TMushLocalHost::HostName(void)
{
    uname(&UtsName);
    return UtsName.nodename;
}

const char *TMushLocalHost::Machine(void)
{
    uname(&UtsName);
    return UtsName.machine;
}


bool TMushCpu::IsLittleEndian(void)
{
    static const int IntValue = 1;
    return ((char*) &IntValue)[0] == 1;
}



TMushEnvironmentVariable::TMushEnvironmentVariable(const string& Name)
{
    _Name = Name;
    _Value = Get(Name);
}

TMushEnvironmentVariable::~TMushEnvironmentVariable()
{
}

string TMushEnvironmentVariable::AsString(void) throw(TSystemCallException)
{
    if (_Value.empty()) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsString()",
            string("undefined variable: ") + _Name
        );
    }

    return _Value;
}

long TMushEnvironmentVariable::AsLong(void) throw(TSystemCallException)
{
    if (_Value.empty()) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsLong()",
            string("undefined variable: ") + _Name
        );
    }

    istrstream ValueStream(_Value.c_str());
    long LongValue;
    if (! (ValueStream >> LongValue)) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsLong()",
            string("variable(") + _Name + string(") is not long: ") + _Value
        );
    }

    return LongValue;
}

double TMushEnvironmentVariable::AsDouble(void) throw(TSystemCallException)
{
    if (_Value.empty()) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsDouble()",
            string("undefined variable: ") + _Name
        );
    }

    istrstream ValueStream(_Value.c_str());
    double DoubleValue;
    if (! (ValueStream >> DoubleValue)) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsDouble()",
            string("variable(") + _Name + string(") is not double: ") + _Value
        );
    }

    return DoubleValue;
}

TMushEnvironmentVariable::operator const char* ()
{
    return _Value.c_str();
}

TMushEnvironmentVariable::operator const string& ()
{
    return _Value;
}

string TMushEnvironmentVariable::Get(const string& Name)
{
    const char* Value = getenv(Name.c_str());
    if (Value == 0) {
	return "";
    }
    else {
	return Value;
    }
}

void TMushEnvironmentVariable::Set(const string& Name, const string& Value)
{
    string Command = (Name + '=' + Value);
    putenv((char*) Command.c_str());
}

bool TMushEnvironmentVariable::IsDefined(const string& Name)
{
    return (getenv(Name.c_str()) != 0);
}
