/* MushNetworkSocket.cc */
/* Created by Enomoto Sanshiro on 21 January 1998. */
/* Last updated by Enomoto Sanshiro on 19 November 1999. */


#include <iostream>
#include <strstream>
#include <string>
#include <unistd.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <errno.h>

#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "MushNetworkSocket.hh"

using namespace std;


unsigned long TMushNetworkSocket::_LocalAddress = 0;
string TMushNetworkSocket::_LocalHostName;
string TMushNetworkSocket::_LocalIPAddress;


TMushNetworkSocket::TMushNetworkSocket(int PortNumber)
{
    _PortNumber = PortNumber;
}

TMushNetworkSocket::~TMushNetworkSocket() 
{
}

unsigned long TMushNetworkSocket::HostAddressByName(string HostName) throw(TSystemCallException)
{
    struct hostent *Host;
    Host = gethostbyname(HostName.c_str());

    if (Host == 0) {
	unsigned long Address = inet_addr(HostName.c_str());
	if ((signed long) Address < 0) {
	    throw TSystemCallException(
                string("TMushNetworkSocket::HostAddressByName") +
                "(\"" + HostName + "\"): " +
                "TMushNetworkSocket::HostAddressByName(): " +
                "inet_addr(2)"
            );
	}
	else {
            Host = gethostbyaddr((char *) &Address, sizeof(Address), AF_INET);
	}
    }

    return (* (unsigned long *) Host->h_addr_list[0]);
}

string TMushNetworkSocket::LocalIPAddress(void)
{
    if (_LocalIPAddress.empty()) {
	LookupLocalHostName();
    }

    return _LocalIPAddress;
}

string TMushNetworkSocket::LocalHostName(void)
{
    if (_LocalHostName.empty()) {
	LookupLocalHostName();
    }

    return _LocalHostName;
}

unsigned long TMushNetworkSocket::LocalAddress(void) const
{
    if (_LocalAddress == 0) {
	LookupLocalHostName();
    }
	
    return _LocalAddress;
}

string TMushNetworkSocket::RemoteIPAddress(void) const 
{
    return _RemoteIPAddress;
}

string TMushNetworkSocket::RemoteHostName(void) const
{
    return _RemoteHostName;
}

unsigned long TMushNetworkSocket::RemoteAddress(void) const
{
    return _RemoteAddress;
}

string TMushNetworkSocket::AddressStringOf(unsigned long Address)
{
    char Buffer[128];
    ostrstream StringStream(Buffer, sizeof(Buffer));

    /* Note that h_addr_list is network byte order */
    StringStream << (int) ((unsigned char*)&_LocalAddress)[0] << ".";
    StringStream << (int) ((unsigned char*)&_LocalAddress)[1] << ".";
    StringStream << (int) ((unsigned char*)&_LocalAddress)[2] << ".";
    StringStream << (int) ((unsigned char*)&_LocalAddress)[3] << ends;
    
    return string(Buffer);
}

void TMushNetworkSocket::LookupLocalHostName(void) throw(TSystemCallException)
{
    static struct utsname UtsName;
    uname(&UtsName);
    _LocalHostName = UtsName.nodename;

    struct hostent *LocalHost;
    LocalHost = gethostbyname(UtsName.nodename);

    if (LocalHost == 0) {
	throw TSystemCallException(
            "TMushNetworkSocket::LookupLocalHostName(): gethostbyname(2)"
        );
    }

    _LocalAddress = * (unsigned long *) LocalHost->h_addr_list[0];
    
    /* Note that h_addr_list is network byte order */
    ostrstream LocalIPStream;
    LocalIPStream << (int) ((unsigned char*)&_LocalAddress)[0] << ".";
    LocalIPStream << (int) ((unsigned char*)&_LocalAddress)[1] << ".";
    LocalIPStream << (int) ((unsigned char*)&_LocalAddress)[2] << ".";
    LocalIPStream << (int) ((unsigned char*)&_LocalAddress)[3] << ends;
    
    _LocalIPAddress = LocalIPStream.str();
}

void TMushNetworkSocket::LookupRemoteHostName(unsigned long RemoteAddress) throw(TSystemCallException)
{
    _RemoteAddress = RemoteAddress;

    /* Note that h_addr_list is network byte order */
    ostrstream RemoteIPStream;
    RemoteIPStream << (int) ((unsigned char*)&_RemoteAddress)[0] << ".";
    RemoteIPStream << (int) ((unsigned char*)&_RemoteAddress)[1] << ".";
    RemoteIPStream << (int) ((unsigned char*)&_RemoteAddress)[2] << ".";
    RemoteIPStream << (int) ((unsigned char*)&_RemoteAddress)[3] << ends;
    _RemoteIPAddress = RemoteIPStream.str();

    struct hostent *RemoteHost = gethostbyaddr(
        (char *) &_RemoteAddress, sizeof(_RemoteAddress), AF_INET
    );    
    if (RemoteHost == 0) {
	_RemoteHostName = _RemoteIPAddress;
    }
    else {
	_RemoteHostName = RemoteHost->h_name;
    }
}



TMushServerNetworkSocket::TMushServerNetworkSocket(int PortNumber)
: TMushNetworkSocket(PortNumber)
{
}

void TMushServerNetworkSocket::Create(void) throw(TSystemCallException)
{
    if (_SocketStatus == ssInitial) {
	LookupLocalHostName();

	_ListenSocket = socket(AF_INET, SOCK_STREAM, 0);
	if (_ListenSocket < 0) {
	    throw TSystemCallException(
		"TMushServerNetworkSocket::Create(): socket(2)"
	    );
	}

	_SocketStatus = ssCreated;
    }

    if (_SocketStatus != ssCreated) {
	throw TSystemCallException(
            "TMushServerNetworkSocket::Create()",
            "bad socket status to create"
        );
    }
}

void TMushServerNetworkSocket::Bind(void) throw(TSystemCallException)
{
    if (_SocketStatus == ssInitial) {
	Create();
    }

    if (_SocketStatus == ssCreated) {
	struct sockaddr_in SocketAddress;
	SocketAddress.sin_family = AF_INET;
	SocketAddress.sin_port = htons(_PortNumber);
	SocketAddress.sin_addr.s_addr = INADDR_ANY;

        int BindResult = bind(
            _ListenSocket,
            (struct sockaddr *) &SocketAddress,
            sizeof(SocketAddress)
        );
	if (BindResult < 0) {
	    throw TSystemCallException(
                "TMushServerNetworkSocket::Bind(): bind(2)"
            );
	}	
	_SocketStatus = ssBound;
    }

    if (_SocketStatus != ssBound) {
	throw TSystemCallException(
            "TMushServerNetworkSocket::Bind()",
            "bad socket status to bind"
        );
    }
}

void TMushServerNetworkSocket::Accept(void) throw(TSystemCallException)
{
    if (_SocketStatus == ssListening) {
	struct sockaddr_in RemoteSocketAddress;
#if ADDRLEN_IS_UNSIGNED
        size_t AddressLength = sizeof(RemoteSocketAddress);
#else
        int AddressLength = sizeof(RemoteSocketAddress);
#endif
	_ConnectSocket = accept(
	    _ListenSocket, 
	    (struct sockaddr *) &RemoteSocketAddress, 
	    &AddressLength
	);
	if (_ConnectSocket < 0) {
	    throw TSystemCallException(
                "TMushServerNetworkSocket::Accept(): accept(2)"
            );
	}

	unsigned long RemoteAddress = RemoteSocketAddress.sin_addr.s_addr;
	LookupRemoteHostName(RemoteAddress);

	_SocketStatus = ssOpened;
	close(_ListenSocket);
    }
    else {
	throw TSystemCallException(
            "TMushServerNetworkSocket::Accept()",
            "bad socket status to accept"
        );
    }
}



TMushClientNetworkSocket::TMushClientNetworkSocket(string RemoteHostName, int PortNumber)
: TMushNetworkSocket(PortNumber)
{
    _RemoteHostName = RemoteHostName;
}

void TMushClientNetworkSocket::Create(void) throw(TSystemCallException)
{
    if (_SocketStatus == ssInitial) {
	LookupLocalHostName();

	unsigned long RemoteAddress = HostAddressByName(_RemoteHostName);
	LookupRemoteHostName(RemoteAddress);
	
	_ConnectSocket = socket(AF_INET, SOCK_STREAM, 0);
	if (_ConnectSocket < 0) {
	    throw TSystemCallException(
		"TMushClientNetworkSocket::Create(): socket(2)"
	    );
	}

	_SocketStatus = ssCreated;
    }

    if (_SocketStatus != ssCreated) {
	throw TSystemCallException(
            "TMushClientNetworkSocket::Create()",
            "bad socket status to create"
        );
    }

}

void TMushClientNetworkSocket::Connect(void) throw(TSystemCallException)
{
    if (_SocketStatus == ssInitial) {
	Create();
    }

    if (_SocketStatus != ssCreated) {
	throw TSystemCallException(
            "TMushClientNetworkSocket::Connect()",
            "bad socket status to connect"
        );
    }

    struct sockaddr_in RemoteSocketAddress;
    RemoteSocketAddress.sin_family = AF_INET;
    RemoteSocketAddress.sin_port = htons(_PortNumber);
    RemoteSocketAddress.sin_addr.s_addr = _RemoteAddress;
    
    if (
        connect(
	    _ConnectSocket, 
	    (struct sockaddr *) &RemoteSocketAddress, 
            sizeof(RemoteSocketAddress)
	) < 0
    ){
	throw TSystemCallException(
            "TMushClientNetworkSocket::Connect(): connect(2)"
        );
    }

    _SocketStatus = ssOpened;
}
