/* ParaExpression.cc */
/* Created by Enomoto Sanshiro on 22 August 1999. */
/* Last updated by Enomoto Sanshiro on 3 July 2002. */


#include <iostream>
#include <strstream>
#include <string>
#include <vector>
#include "ParaObject.hh"
#include "ParaValue.hh"
#include "ParaOperator.hh"
#include "ParaSymbolTable.hh"
#include "ParaExpression.hh"
#include "ParaFunction.hh"
#include "ParaBuiltinFunction.hh"

using namespace std;


TParaExpressionParser::TParaExpressionParser(TParaOperatorTable* OperatorTable)
{
    _OperatorTable = OperatorTable;
}

TParaExpressionParser::~TParaExpressionParser()
{
}

TParaExpression* TParaExpressionParser::Parse(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable, int PriorityRank) throw(TScriptException)
{
    TParaExpression* ThisNode = 0;

    if (PriorityRank < 0) {
        /* default priority: start from the lowest priority (highest level) */
        PriorityRank = _OperatorTable->LowestPriorityRank();
    }
    if (PriorityRank < _OperatorTable->HighestPriorityRank()) {
        return ParsePrepositional(Tokenizer, SymbolTable);
    }
    
    ThisNode = Parse(Tokenizer, SymbolTable, PriorityRank - 1);

    while (1) {
	string OperatorSymbol = Tokenizer->LookAhead().AsString();
	if (_OperatorTable->PriorityRankOf(OperatorSymbol) != PriorityRank) {
	    break;
	}

	TParaExpression* LeftNode = ThisNode;
	TParaExpression* RightNode = 0;
	TParaOperator* Operator = _OperatorTable->CreateOperator(OperatorSymbol);

	long LineNumber = Tokenizer->LineNumber();
	Operator->Parse(Tokenizer, this, SymbolTable);
    
	int RightNodePriorityRank = PriorityRank - 1;
	if (! Operator->IsLeftAssociative()) {
	    RightNodePriorityRank = PriorityRank;
	}

	try {
	    RightNode = Parse(Tokenizer, SymbolTable, RightNodePriorityRank);
	}
	catch (TScriptException &e) {
	    delete Operator;
	    delete LeftNode;
	    throw e;
	}
	ThisNode = new TParaOperatorNode(Operator, LeftNode, RightNode);
	ThisNode->SetLineNumber(LineNumber);

	if (! Operator->IsLeftAssociative()) {
	    break;
	}
    }
    
    return ThisNode;
}

TParaExpression* TParaExpressionParser::ParsePrepositional(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpression* ThisNode = 0;
    long LineNumber = Tokenizer->LineNumber();

    TParaToken Token = Tokenizer->LookAhead();
    string OperatorSymbol = Token.AsString();
    TParaOperator* PrepositionalOperator = _OperatorTable->CreatePrepositionalOperator(OperatorSymbol);
    
    // normal prepositional operator //
    if (PrepositionalOperator != 0) {
	PrepositionalOperator->Parse(Tokenizer, this, SymbolTable);
        TParaExpression* RightNode = ParsePrepositional(Tokenizer, SymbolTable);
	ThisNode = new TParaOperatorNode(PrepositionalOperator, 0, RightNode);
	ThisNode->SetLineNumber(LineNumber);
    }

    // typecast //
    // (int), (Scanner), (int*), (int***) etc.
    // the third condition is to make distinction from (Scanner(line).get())
    else if (
	Tokenizer->LookAhead(1).Is("(") &&
	SymbolTable->IsTypeName(Tokenizer->LookAhead(2).AsString()) &&
	! Tokenizer->LookAhead(3).Is("(")
    ){
	ThisNode = ParseTypeCast(Tokenizer, SymbolTable);
    }

    // go lower: element //
    else {
        ThisNode = ParseElement(Tokenizer, SymbolTable);
    }

    return ThisNode;
}

TParaExpression* TParaExpressionParser::ParseElement(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpression* ThisNode = 0;
    TParaToken Token = Tokenizer->Next();
    long LineNumber = Tokenizer->LineNumber();
    
    string OperatorSymbol = Token.AsString();
    TParaOperator* ElementaryOperator = _OperatorTable->CreateElementaryOperator(OperatorSymbol);

    if (ElementaryOperator != 0) {
	Tokenizer->Unget(Token);
	ElementaryOperator->Parse(Tokenizer, this, SymbolTable);
	ThisNode = new TParaOperatorNode(ElementaryOperator, 0, 0);
	ThisNode->SetLineNumber(LineNumber);
    }

    else if (Token.Is("(")) {
	ThisNode = Parse(Tokenizer, SymbolTable);
	Tokenizer->Next().MustBe(")");
    }

    // primitive literals //
    else if (Token.IsInteger()) {
        ThisNode = new TParaLiteralNode(TParaValue(Token.AsLong()));
	ThisNode->SetLineNumber(LineNumber);
    }
    else if (Token.IsFloating()) {
        ThisNode = new TParaLiteralNode(TParaValue(Token.AsDouble()));
	ThisNode->SetLineNumber(LineNumber);
    }
    else if (Token.IsQuote()) {
        Token.RemoveQuotation();
        ThisNode = new TParaLiteralNode(TParaValue(Token.AsString()));
	ThisNode->SetLineNumber(LineNumber);
    }

    // temporary object creation //
    else if (SymbolTable->IsTypeName(Token.AsString())) {
	Tokenizer->Unget(Token);
	ThisNode = ParseTemporaryObjectCreation(Tokenizer, SymbolTable);
    }

    // identifier: function call or variable //
    else if (Token.IsIdentifier()) {
        if (Tokenizer->LookAhead().Is("(")) {
            Tokenizer->Unget(Token);
            ThisNode = ParseFunctionCall(Tokenizer, SymbolTable);
        }
        else {
            long VariableId = SymbolTable->NameToId(Token.AsString());
            ThisNode = new TParaVariableNode(VariableId);
	    ThisNode->SetLineNumber(LineNumber);
        }
    }

    // list literal //
    else if (Token.Is("{")) {
	Tokenizer->Unget(Token);
        ThisNode = ParseListExpression(Tokenizer, SymbolTable);
	ThisNode->SetLineNumber(LineNumber);
    }
    
    else {
	Token.ThrowUnexpected();
    }

    return ParsePostpositional(ThisNode, Tokenizer, SymbolTable);
}

TParaExpression* TParaExpressionParser::ParseTypeCast(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long LineNumber = Tokenizer->LineNumber();

    Tokenizer->Next().MustBe("(");
    string TypeName = Tokenizer->Next().AsString();
    while (Tokenizer->LookAhead().Is("*")) {
	TypeName = "pointer";
	Tokenizer->Next();
    }
    Tokenizer->Next().MustBe(")");

    TParaExpression* RightNode = ParsePrepositional(Tokenizer, SymbolTable);

    TParaExpression* ThisNode = new TParaTypeCastNode(TypeName, RightNode);
    ThisNode->SetLineNumber(LineNumber);

    return ThisNode;
}

TParaExpression* TParaExpressionParser::ParseTemporaryObjectCreation(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long LineNumber = Tokenizer->LineNumber();

    TParaToken Token = Tokenizer->Next();
    string TypeName = Token.AsString();
    
    if (! SymbolTable->IsTypeName(Token.AsString())) {
	Token.ThrowUnexpected("type name");
    }
        
    vector<TParaExpression*> ArgumentExpressionList;
    ArgumentExpressionList = ParseExpressionList(
	Tokenizer, SymbolTable, "(", ")", ","
    );

    TParaExpression* ThisNode = new TParaTemporaryObjectCreationNode(
	TypeName, ArgumentExpressionList
    );
    ThisNode->SetLineNumber(LineNumber);

    return ThisNode;
}

TParaExpression* TParaExpressionParser::ParseFunctionCall(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long LineNumber = Tokenizer->LineNumber();

    TParaToken Token = Tokenizer->Next();
    if (! Token.IsIdentifier()) {
	Token.ThrowUnexpected("function name");
    }
    long FunctionId = SymbolTable->NameToId(Token.AsString());
        
    vector<TParaExpression*> ArgumentExpressionList;
    ArgumentExpressionList = ParseExpressionList(
	Tokenizer, SymbolTable, "(", ")", ","
    );

    TParaExpression* ThisNode = new TParaFunctionCallNode(
	FunctionId, ArgumentExpressionList
    );
    ThisNode->SetLineNumber(LineNumber);

    return ThisNode;
}

TParaExpression* TParaExpressionParser::ParseListExpression(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    vector<TParaExpression*> KeyExpressionList;
    vector<TParaExpression*> ValueExpressionList;

    TParaToken Token;
    Tokenizer->Next().MustBe("{");
    
    if ((Token = Tokenizer->Next()).IsNot("}")) {
        Tokenizer->Unget(Token);

	TParaExpression* Expression;
        while (1) {
	    Expression = Parse(Tokenizer, SymbolTable);

            Token = Tokenizer->Next();
	    if (Token.Is("=>")) {
		KeyExpressionList.push_back(Expression);
		Expression = Parse(Tokenizer, SymbolTable);
		Token = Tokenizer->Next();
	    }
	    else {
		KeyExpressionList.push_back((TParaExpression*) 0);
	    }
	    
	    ValueExpressionList.push_back(Expression);

            if (Token.Is("}")) {
                break;
            }
            else {
                Token.MustBe(",");
            }
        }
    }

    return new TParaListNode(KeyExpressionList, ValueExpressionList);
}

TParaExpression* TParaExpressionParser::ParsePostpositional(TParaExpression* Expression, TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpression* ThisNode = Expression;
    TParaOperator* Operator;

    TParaToken Token = Tokenizer->LookAhead();
    Operator = _OperatorTable->CreatePostpositionalOperator(Token.AsString());

    if (Operator != 0) {
	Operator->Parse(Tokenizer, this, SymbolTable);
	ThisNode = new TParaOperatorNode(Operator, ThisNode, 0);
	ThisNode->SetLineNumber(Tokenizer->LineNumber());
    }
    else if (Token.Is(".") || Token.Is("->")) {
        ThisNode = ParseMethodInvocation(ThisNode, Tokenizer, SymbolTable);
    }
    else if (Token.Is("[")) {
        ThisNode = ParseArrayReference(ThisNode, Tokenizer, SymbolTable);
    }
    else if (Token.Is("{")) {
        ThisNode = ParseTableReference(ThisNode, Tokenizer, SymbolTable);
    }
    else {
	return ThisNode;
    }

    return ParsePostpositional(ThisNode, Tokenizer, SymbolTable);
}

TParaExpression* TParaExpressionParser::ParseMethodInvocation(TParaExpression* ObjectNode, TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long LineNumber = Tokenizer->LineNumber();

    if (Tokenizer->Next().Is("->")) {
        TParaOperator* Operator = new TParaOperatorPointerReference();
        ObjectNode = new TParaOperatorNode(Operator, 0, ObjectNode);
	ObjectNode->SetLineNumber(LineNumber);
    }    

    TParaToken Token = Tokenizer->Next();
    if (! Token.IsIdentifier()) {
	Token.ThrowUnexpected("method or property name");
    }
    string MethodName = Token.AsString();
    long MethodId = SymbolTable->NameToId(MethodName);

    TParaExpression* ThisNode = 0;
    if (Tokenizer->LookAhead().IsNot("(")) {
	ThisNode = new TParaPropertyAccessNode(ObjectNode, MethodName);
    }
    else {
	vector<TParaExpression*> ArgumentExpressionList;
	ArgumentExpressionList = ParseExpressionList(
	    Tokenizer, SymbolTable, "(", ")", ","
	);
	
	ThisNode = new TParaMethodInvocationNode(
	    ObjectNode, MethodId, ArgumentExpressionList
	);
    }

    ThisNode->SetLineNumber(LineNumber);
    
    return ThisNode;
}

TParaExpression* TParaExpressionParser::ParseArrayReference(TParaExpression* ObjectNode, TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long LineNumber = Tokenizer->LineNumber();

    vector<TParaExpression*> IndexExpressionList = ParseExpressionList(
	Tokenizer, SymbolTable, "[", "]", ","
    );
    
    TParaExpression* ThisNode = new TParaArrayReferenceNode(
        ObjectNode, IndexExpressionList
    );
    ThisNode->SetLineNumber(LineNumber);
    
    return ThisNode;
}

TParaExpression* TParaExpressionParser::ParseTableReference(TParaExpression* ObjectNode, TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long LineNumber = Tokenizer->LineNumber();

    Tokenizer->Next().MustBe("{");
    TParaExpression* ThisNode = new TParaTableReferenceNode(
        ObjectNode, Parse(Tokenizer, SymbolTable)
    );
    Tokenizer->Next().MustBe("}");

    ThisNode->SetLineNumber(LineNumber);
    
    return ThisNode;
}

vector<TParaExpression*> TParaExpressionParser::ParseExpressionList(TParaTokenizer* Tokenizer, TParaSymbolTable* SymbolTable, const string& StartToken, const string& StopToken, const string& SeparatorToken) throw(TScriptException)
{
    vector<TParaExpression*> ArgumentExpressionList;

    TParaToken Token;
    Tokenizer->Next().MustBe(StartToken);
    
    if ((Token = Tokenizer->Next()).IsNot(StopToken)) {
        Tokenizer->Unget(Token);

        while (1) {
            TParaExpression* Argument = Parse(Tokenizer, SymbolTable);
            ArgumentExpressionList.push_back(Argument);

            Token = Tokenizer->Next();
            if (Token.Is(StopToken)) {
                break;
            }
            else {
                Token.MustBe(SeparatorToken);
            }
        }
    }

    return ArgumentExpressionList;
}



TParaExpression::TParaExpression()
{
    _LeftNode = 0;
    _RightNode = 0;
}
    
TParaExpression::~TParaExpression()
{
    delete _LeftNode;
    delete _RightNode;
}
    
void TParaExpression::SetLineNumber(long LineNumber)
{
    _LineNumber = LineNumber;
}

string TParaExpression::Position(void) const
{
    if (_LineNumber == 0) {
	return string("");
    }

    char Buffer[256];
    ostrstream Stream(Buffer, sizeof(Buffer));
    Stream << "line " << _LineNumber << ": " << ends;

    return string(Buffer);
}

void TParaExpression::Dump(ostream &os, int IndentLevel) const
{
    if (_LeftNode != 0) {
        _LeftNode->Dump(os, IndentLevel + 1);
    }

    for (int i = 0; i < IndentLevel; i++) {
        os << "  ";
    }
    DumpThis(os);
    os << endl;

    if (_RightNode != 0) {
        _RightNode->Dump(os, IndentLevel + 1);
    }
}



TParaOperatorNode::TParaOperatorNode(TParaOperator* Operator, TParaExpression* LeftNode, TParaExpression* RightNode)
{
    _Operator = Operator;
    _LeftNode = LeftNode ? LeftNode : new TParaLiteralNode(TParaValue());
    _RightNode = RightNode ? RightNode : new TParaLiteralNode(TParaValue());
}   

TParaOperatorNode::~TParaOperatorNode()
{
    delete _Operator;
}

TParaValue& TParaOperatorNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& LeftValue = _LeftNode->Evaluate(SymbolTable);
    TParaValue& RightValue = _RightNode->Evaluate(SymbolTable);
    
    try {
	//...
	if (LeftValue.IsVariant() && (_Operator->Symbol() == "=")) {
	    return _Operator->Evaluate(
		LeftValue, RightValue, SymbolTable, _Value
	    );
	}

	if (LeftValue.IsObject()) {
	    return LeftValue.AsObject()->EvaluateOperator(
		_Operator, LeftValue, RightValue, SymbolTable, _Value
	    );
	}
	else if (RightValue.IsObject()) {
	    return RightValue.AsObject()->EvaluateOperator(
		_Operator, LeftValue, RightValue, SymbolTable, _Value
	    );
	}
	else {
	    return _Operator->Evaluate(
		LeftValue, RightValue, SymbolTable, _Value
	    );
	}
    }
    catch (TScriptException &e) {
	throw TScriptException(Position() + e.Message());
    }
}

void TParaOperatorNode::DumpThis(ostream &os) const
{
    os << _Operator->Symbol();
}



TParaTypeCastNode::TParaTypeCastNode(const string& TypeName, TParaExpression* RightNode)
{
    _TypeName = TypeName;

    _LeftNode = 0;
    _RightNode = RightNode;
}   

TParaTypeCastNode::~TParaTypeCastNode()
{
}

TParaValue& TParaTypeCastNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& Right = _RightNode->Evaluate(SymbolTable);

    try {
	if ((_TypeName == "int") || (_TypeName == "long")) {
	    _Value = TParaValue(Right.AsLong());
	}
	else if ((_TypeName == "float") || (_TypeName == "double")) {
	    _Value = TParaValue(Right.AsDouble());
	}
	else if (_TypeName == "string") {
	    _Value = TParaValue(Right.AsString());
	}
	else if ((_TypeName == "pointer") || (_TypeName == "*")) {
	    _Value = TParaValue(Right.AsPointer());
	}
	else{ 
	    throw TScriptException("invalid type name");
	}
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "invalid type cast: " + _TypeName + ": " + e.Message()
	);
    }

    return _Value;
}

void TParaTypeCastNode::DumpThis(ostream &os) const
{
    os << "(" << _TypeName << ")";
}



TParaLiteralNode::TParaLiteralNode(const TParaValue& Value)
: _Value(Value)
{
}

TParaLiteralNode::~TParaLiteralNode()
{
}

TParaValue& TParaLiteralNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _Value;
}

void TParaLiteralNode::DumpThis(ostream &os) const
{
    os << _Value.AsString();
}



TParaVariableNode::TParaVariableNode(long VariableId)
{
    _VariableId = VariableId;
}

TParaVariableNode::~TParaVariableNode()
{
}

TParaValue& TParaVariableNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue* Variable = SymbolTable->GetVariable(_VariableId);
    if (Variable == 0) {
        string Name = SymbolTable->IdToName(_VariableId);
        throw TScriptException(Position() + "undefined variable: " + Name);
    }

    return *Variable;
}

void TParaVariableNode::DumpThis(ostream &os) const
{
    os << TParaNameTable::GetInstance()->IdToName(_VariableId);
    os << "{" << _VariableId << "}";
}



TParaListNode::TParaListNode(const vector<TParaExpression*>& KeyExpressionList, const vector<TParaExpression*>& ValueExpressionList)
{
    _KeyExpressionList = KeyExpressionList;
    _ValueExpressionList = ValueExpressionList;
}

TParaListNode::~TParaListNode()
{
    for (unsigned i = 0; i < _KeyExpressionList.size(); i++) {
	delete _KeyExpressionList[i];
	delete _ValueExpressionList[i];
    }
}

TParaValue& TParaListNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _Value = TParaValue(TParaListValue());
    TParaListValue& ListValue = _Value.AsList();
    vector<TParaValue>& ValueList = _Value.AsValueList();

    for (unsigned i = 0; i < _ValueExpressionList.size(); i++) {
	TParaValue& Value = _ValueExpressionList[i]->Evaluate(SymbolTable);
	if (_KeyExpressionList[i] != 0) {
	    TParaValue& Key = _KeyExpressionList[i]->Evaluate(SymbolTable);
	    ListValue.ValueOf(Key.AsString()) = Value;
	}
	else {
	    ValueList.push_back(Value);
	}
    }
    
    return _Value;
}

void TParaListNode::DumpThis(ostream &os) const
{
    os << "list[" << _ValueExpressionList.size() << "]";
}



TParaFunctionCallNode::TParaFunctionCallNode(long FunctionId, vector<TParaExpression*>& ArgumentExpressionList)
: _Value((long) 0)
{
    _FunctionId = FunctionId;
    _ArgumentExpressionList = ArgumentExpressionList;
    
    _BuiltinFunctionTable = 0;
}

TParaFunctionCallNode::~TParaFunctionCallNode()
{
    for (unsigned i = 0; i < _ArgumentExpressionList.size(); i++) {
	delete _ArgumentExpressionList[i];
    }
}

TParaValue& TParaFunctionCallNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    EvaluateArguments(SymbolTable);

    int Result = 0;

    TParaFunction* Function = SymbolTable->GetFunction(_FunctionId);
    if (Function != 0) {
	long ImportDepth;
	TParaSymbolTable* GlobalSymbolTable;
	GlobalSymbolTable = new TParaSymbolTable(SymbolTable, ImportDepth = 1);
	_Value = Function->Execute(_ArgumentList, GlobalSymbolTable);
	delete GlobalSymbolTable;    
	Result = 1;
    }
    else {
	if (_BuiltinFunctionTable == 0) {
	    _BuiltinFunctionTable = SymbolTable->BuiltinFunctionTable();
	    if (_BuiltinFunctionTable != 0) {
		_BuiltinFunctionTable->RegisterFunctionId(
		    SymbolTable->IdToName(_FunctionId), _FunctionId
		);
	    }
	}

	if (_BuiltinFunctionTable != 0) {
	    try {
		Result = _BuiltinFunctionTable->Execute(
		    _FunctionId, _ArgumentList, _Value
		);
	    }
	    catch (TScriptException &e) {
		throw TScriptException(Position() + e.Message());
	    }
	}
    }
    
    if (Result == 0) {
	string FunctionName= SymbolTable->IdToName(_FunctionId);
	throw TScriptException(
	    Position() + "unknown function: " + FunctionName + "()"
	);
    }

    return _Value;
}

void TParaFunctionCallNode::EvaluateArguments(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _ArgumentList.erase(
        _ArgumentList.begin(), _ArgumentList.end()
    );
    
    vector<TParaExpression*>::iterator Expression;
    for (
        Expression = _ArgumentExpressionList.begin();
        Expression != _ArgumentExpressionList.end();
        Expression++
    ){
        TParaValue& Value = (*Expression)->Evaluate(SymbolTable);
        _ArgumentList.push_back(&Value);
    }
}

void TParaFunctionCallNode::DumpThis(ostream &os) const
{
    string FunctionName = TParaNameTable::GetInstance()->IdToName(_FunctionId);
    os << FunctionName << "(...)";
}



TParaMethodInvocationNode::TParaMethodInvocationNode(TParaExpression* ObjectExpression, long FunctionId, vector<TParaExpression*>& ArgumentExpressionList)
: TParaFunctionCallNode(FunctionId, ArgumentExpressionList)
{
    _ObjectExpression = ObjectExpression;
    _MethodId = -1;
}

TParaMethodInvocationNode::~TParaMethodInvocationNode()
{
    delete _ObjectExpression;
}

TParaValue& TParaMethodInvocationNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& ObjectValue = _ObjectExpression->Evaluate(SymbolTable);
    TParaObjectPrototype* Object;
    try {
	Object = ObjectValue.AsObject();
    }
    catch (TScriptException &e) {
	throw TScriptException(Position() + e.Message());
    }

    EvaluateArguments(SymbolTable);
    
    if (_MethodId < 0) {
	//... BUG: 'Object' can be a instance of a different class 
        // from one in the previous call (when _ObjectExpression is 
        // a pointer reference, for example). 
	// _MethodId becomes wrong in such cases.

	_MethodName = SymbolTable->IdToName(_FunctionId);
	_MethodId = Object->MethodIdOf(_MethodName);
    }

    try {
	int Result = 0;
	if (_MethodId > 0) {
	    Result = Object->InvokeMethod(_MethodId, _ArgumentList, _Value);
	}
	if (Result == 0) {
	    Result = Object->DispatchMessage(_MethodName, _ArgumentList, _Value);
	}
	if (Result == 0) {
	    throw TScriptException("unknown method: " + _MethodName);
	}
    }
    catch (TScriptException &e) {
	throw TScriptException(Position() + e.Message());
    }

    return _Value;
}



TParaPropertyAccessNode::TParaPropertyAccessNode(TParaExpression* ObjectExpression, const string& PropertyName)
{
    _ObjectExpression = ObjectExpression;
    _PropertyName = PropertyName;
}

TParaPropertyAccessNode::~TParaPropertyAccessNode()
{
    delete _ObjectExpression;
}

TParaValue& TParaPropertyAccessNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& ObjectValue = _ObjectExpression->Evaluate(SymbolTable);
    TParaObjectPrototype* Object;
    try {
	Object = ObjectValue.AsObject();
    }
    catch (TScriptException &e) {
	throw TScriptException(Position() + e.Message());
    }

    try {
	//... TODO: Object->GetProperty(PropertyId, _Value)
	int Result = Object->GetPropertyOf(_PropertyName, _Value);
	if (Result == 0) {
	    throw TScriptException("unknown property: " + _PropertyName);
	}
    }
    catch (TScriptException &e) {
	throw TScriptException(Position() + e.Message());
    }

    return _Value;
}

void TParaPropertyAccessNode::DumpThis(ostream &os) const
{
    os << "." << _PropertyName;
}



TParaTemporaryObjectCreationNode::TParaTemporaryObjectCreationNode(const string& TypeName, vector<TParaExpression*>& ArgumentExpressionList)
: TParaFunctionCallNode(0, ArgumentExpressionList)
{
    _TypeName = TypeName;
}

TParaTemporaryObjectCreationNode::~TParaTemporaryObjectCreationNode()
{
}

TParaValue& TParaTemporaryObjectCreationNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    EvaluateArguments(SymbolTable);
    
    TParaValue* Variable = SymbolTable->CreateObject(_TypeName);
    if (Variable == 0) {
        throw TScriptException(
	    Position() + "unknown variable type: " + _TypeName
	);
    }
    long VariableId = -1;  // unused value for temporary objects
    SymbolTable->RegisterVariable(VariableId, Variable);

    if (Variable->IsObject()) {
	Variable->AsObject()->Construct(_TypeName, _ArgumentList);
    }
    else {
        if (_ArgumentList.size() > 1) {
	    throw TScriptException("too many initial-value arguments");
	}
	else if (! _ArgumentList.empty()) {
	    Variable->Assign(*_ArgumentList[0]);
	}
    }

    return *Variable;
}



TParaArrayReferenceNode::TParaArrayReferenceNode(TParaExpression* ObjectExpression, const vector<TParaExpression*>& IndexExpressionList)
{
    _ObjectExpression = ObjectExpression;
    _IndexExpressionList = IndexExpressionList;
}

TParaArrayReferenceNode::~TParaArrayReferenceNode()
{
    for (unsigned i = 0; i < _IndexExpressionList.size(); i++) {
	delete _IndexExpressionList[i];
    }
    delete _ObjectExpression;
}

TParaValue& TParaArrayReferenceNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& ObjectValue = _ObjectExpression->Evaluate(SymbolTable);

    int NumberOfIndexes = _IndexExpressionList.size();
    int BeginIndex = -1, EndIndex = -1;
    try {
	if ((NumberOfIndexes < 1) || (NumberOfIndexes > 2)) {
	    throw TScriptException("bad number of argument[s]");
	}

	BeginIndex = _IndexExpressionList[0]->Evaluate(SymbolTable).AsLong();
	if (BeginIndex < 0) {
	    throw TScriptException("invalid index (negative)");
	}
	if (_IndexExpressionList.size() == 2) {
	    EndIndex = _IndexExpressionList[1]->Evaluate(SymbolTable).AsLong();
	    if (EndIndex < 0) {
		throw TScriptException("invalid index (negative)");
	    }
	}
    }
    catch (TScriptException &e) {
	throw TScriptException(Position() + "operator[]: " + e.Message());
    }

    if (ObjectValue.IsList()) {
	if (NumberOfIndexes == 1) {
	    return ListItemOf(ObjectValue, BeginIndex);
	}
	else if (NumberOfIndexes == 2) {
	    return PartialListOf(ObjectValue, BeginIndex, EndIndex);
	}
    }
    else if (ObjectValue.IsPointer()) {
	if (NumberOfIndexes != 1) {
	    throw TScriptException(
		Position() + "operator[]: bad number of argument[s]"
	    );
	}
	if (BeginIndex >= ObjectValue.AsPointer()->ArrayLength()) {
	    throw TScriptException(
		Position() + "operator[]: index out of range"
	    );
	}
	return ObjectValue.AsPointer()[BeginIndex];
    }
    else if (ObjectValue.IsString()) {
	if (NumberOfIndexes == 1) {
	    EndIndex = BeginIndex + 1;
	}
	return PartialStringOf(ObjectValue, BeginIndex, EndIndex);
    }
    else {
	throw TScriptException(
	    Position() + "operator[]: list, array or string is expected"
	);
    }

    return _Value;
}

TParaValue& TParaArrayReferenceNode::ListItemOf(TParaValue& ListValue, int Index) throw(TScriptException)
{
    vector<TParaValue>& List = ListValue.AsValueList();

    if ((int) List.size() <= Index) {
	bool IsVariant;
	List.resize(Index + 1, TParaValue(IsVariant = true));
	if (ListValue.IsLeftValue()) {
	    ListValue.SetLeftValueFlag();
	}
    }
    
    return List[Index];
}

TParaValue& TParaArrayReferenceNode::PartialListOf(TParaValue& ListValue, int BeginIndex, int EndIndex) throw(TScriptException)
{
    vector<TParaValue>& InputList = ListValue.AsValueList();
    int InputListSize = InputList.size();

    if ((BeginIndex > InputListSize) || (EndIndex > InputListSize)){
	throw TScriptException(Position() + "operator[]: index out of range");
    }
    
    _Value = TParaValue(TParaListValue());
    vector<TParaValue>& List = _Value.AsValueList();

    if (BeginIndex < EndIndex) {
	List.insert(
	    List.begin(), 
	    InputList.begin() + BeginIndex, InputList.begin() + EndIndex
	);
    }
    else {
	List.insert(
	    List.begin(), 
	    InputList.rend() - BeginIndex, InputList.rend() - EndIndex
	);
    }

    return _Value;
}    

TParaValue& TParaArrayReferenceNode::PartialStringOf(TParaValue& StringValue, int BeginIndex, int EndIndex) throw(TScriptException)
{
    string String = StringValue.AsString();
    int StringSize = (int) String.size();

    if ((BeginIndex > StringSize) || (EndIndex > StringSize)){
	throw TScriptException(Position() + "operator[]: index out of range");
    }

    if (BeginIndex < EndIndex) {
	string SubString(
	    String.begin() + BeginIndex, String.begin() + EndIndex
	);
	_Value = TParaValue(SubString);
    }
    else {
	string SubString(
	    String.rend() - BeginIndex, String.rend() - EndIndex
	);
	_Value = TParaValue(SubString);
    }

    return _Value;
}

void TParaArrayReferenceNode::DumpThis(std::ostream &os) const
{
    os << "[]";
}



TParaTableReferenceNode::TParaTableReferenceNode(TParaExpression* ObjectExpression, TParaExpression* IndexExpression)
{
    _ObjectExpression = ObjectExpression;
    _IndexExpression = IndexExpression;
}

TParaTableReferenceNode::~TParaTableReferenceNode()
{
    delete _ObjectExpression;
    delete _IndexExpression;
}

TParaValue& TParaTableReferenceNode::Evaluate(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& ObjectValue = _ObjectExpression->Evaluate(SymbolTable);
    if (! ObjectValue.IsList()) {
	throw TScriptException(
	    Position() + "operator{}: list value is expected"
	);
    }
    TParaListValue& ListValue = ObjectValue.AsList();

    string Key;
    try {
	Key = _IndexExpression->Evaluate(SymbolTable).AsString();
    }
    catch (TScriptException &e) {
	throw TScriptException(Position() + "operator{}: " + e.Message());
    }

    return ListValue.ValueOf(Key);
}

void TParaTableReferenceNode::DumpThis(std::ostream &os) const
{
    os << "{}";
}
