/* ParaStandardLibrary.cc */
/* Created by Enomoto Sanshiro on 22 August 1999. */
/* Last updated by Enomoto Sanshiro on 24 March 2002. */


#include <iostream>
#include <fstream>
#include <strstream>
#include <iomanip>
#include <string>
#include <vector>
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include "ParaObject.hh"
#include "ParaStandardLibrary.hh"

using namespace std;


TParaConsoleObject::TParaConsoleObject(void)
: TParaObjectPrototype("Console")
{
}

TParaConsoleObject::~TParaConsoleObject()
{
}

TParaObjectPrototype* TParaConsoleObject::Clone(void)
{
    return new TParaConsoleObject();
}

int TParaConsoleObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "print") {
        return MethodId_Print;
    }
    else if ((MethodName == "printLine") || (MethodName == "println")) {
        return MethodId_PrintLine;
    }
    else if ((MethodName == "getLine") || (MethodName == "getln")) {
        return MethodId_GetLine;
    }
    else if (MethodName == "getString") {
        return MethodId_GetString;
    }
    else if ((MethodName == "getByte") || (MethodName == "getc")) {
        return MethodId_GetByte;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaConsoleObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_Print:
        Result = Print(ArgumentList, ReturnValue);
	break;
      case MethodId_PrintLine:
        Result = PrintLine(ArgumentList, ReturnValue);
	break;
      case MethodId_GetLine:
        Result = GetLine(ArgumentList, ReturnValue);
	break;
      case MethodId_GetString:
        Result = GetString(ArgumentList, ReturnValue);
	break;
      case MethodId_GetByte:
        Result = GetByte(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TParaConsoleObject::Print(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
        cout << ArgumentList[i]->AsString();
    }
    cout << flush;

    return 1;
}

int TParaConsoleObject::PrintLine(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    Print(ArgumentList, ReturnValue);
    cout << endl;
    
    return 1;
}

int TParaConsoleObject::GetLine(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string Line;
    if (getline(cin, Line, '\n')) {
	ReturnValue = TParaValue(Line + '\n');
    }
    else {
	ReturnValue = TParaValue(string(""));
    }

    return 1;
}

int TParaConsoleObject::GetString(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string String;
    cin >> String;

    ReturnValue = TParaValue(String);

    return 1;
}

int TParaConsoleObject::GetByte(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    long Value = -1;

    char ch;
    if (cin.get(ch)) {
	Value = (unsigned char) ch;
    }

    ReturnValue = TParaValue(Value);

    return 1;
}



TParaInputFileObject::TParaInputFileObject(void)
: TParaObjectPrototype("InputFile")
{
    _FileStream = 0;
}

TParaInputFileObject::~TParaInputFileObject()
{
    delete _FileStream;
}

TParaObjectPrototype* TParaInputFileObject::Clone(void)
{
    return new TParaInputFileObject();
}

void TParaInputFileObject::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::" + _InternalClassName + "()",
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::" + _InternalClassName + "()",
	    "invalid argument[s]"
	);
    }

    string FileName = ArgumentList[0]->AsString();
    _FileStream = new ifstream(FileName.c_str());
    if (! *_FileStream) {
	throw TScriptException(
	    _InternalClassName + "::" + _InternalClassName + "()",
	    "unable to open file: " + FileName
	);
    }
}

int TParaInputFileObject::MethodIdOf(const string& MethodName)
{
    if ((MethodName == "getLine") || (MethodName == "getln")) {
        return MethodId_GetLine;
    }
    else if (MethodName == "getString") {
        return MethodId_GetString;
    }
    else if ((MethodName == "getByte") || (MethodName == "getc")) {
        return MethodId_GetByte;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaInputFileObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;
    
    switch (MethodId) {
      case MethodId_GetLine:
        Result = GetLine(ArgumentList, ReturnValue);
	break;
      case MethodId_GetString:
        Result = GetString(ArgumentList, ReturnValue);
	break;
      case MethodId_GetByte:
        Result = GetByte(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TParaInputFileObject::GetLine(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string Line;
    if (getline(*_FileStream, Line, '\n')) {
	ReturnValue = TParaValue(Line + '\n');
    }
    else {
	ReturnValue = TParaValue(string(""));
    }

    return 1;
}

int TParaInputFileObject::GetString(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string String;
    (*_FileStream) >> String;

    ReturnValue = TParaValue(String);

    return 1;
}

int TParaInputFileObject::GetByte(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    long Value = -1;

    char ch;
    if (_FileStream->get(ch)) {
	Value = (unsigned char) ch;
    }

    ReturnValue = TParaValue(Value);

    return 1;
}



TParaOutputFileObject::TParaOutputFileObject(void)
: TParaObjectPrototype("OutputFile")
{
    _FileStream = 0;
}

TParaOutputFileObject::~TParaOutputFileObject()
{
    delete _FileStream;
}

TParaObjectPrototype* TParaOutputFileObject::Clone(void)
{
    return new TParaOutputFileObject();
}

void TParaOutputFileObject::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::" + _InternalClassName + "()",
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::" + _InternalClassName + "()",
	    "invalid argument[s]"
	);
    }

    string FileName = ArgumentList[0]->AsString();
    _FileStream = new ofstream(FileName.c_str());
    if (! *_FileStream) {
	throw TScriptException(
	    _InternalClassName + "::" + _InternalClassName + "()",
	    "unable to open file: " + FileName
	);
    }
}

int TParaOutputFileObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "print") {
        return MethodId_Print;
    }
    else if ((MethodName == "printLine") || (MethodName == "println")) {
        return MethodId_PrintLine;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaOutputFileObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;
    
    switch (MethodId) {
      case MethodId_Print:
	Result = Print(ArgumentList, ReturnValue);
	break;
      case MethodId_PrintLine:
        Result = PrintLine(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TParaOutputFileObject::Print(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
        (*_FileStream) << ArgumentList[i]->AsString();
    }
    (*_FileStream) << flush;

    return 1;
}

int TParaOutputFileObject::PrintLine(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    Print(ArgumentList, ReturnValue);
    (*_FileStream) << endl;
    
    return 1;
}



TParaFormatterObject::TParaFormatterObject(void)
: TParaObjectPrototype("Formatter")
{
    _FormatStream = new ostrstream();
}

TParaFormatterObject::~TParaFormatterObject()
{
    delete _FormatStream;
}

TParaObjectPrototype* TParaFormatterObject::Clone(void)
{
    return new TParaFormatterObject();
}

int TParaFormatterObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "put") {
        return MethodId_Put;
    }
    else if (MethodName == "flush") {
        return MethodId_Flush;
    }
    else if ((MethodName == "setWidth") || (MethodName == "setw")) {
        return MethodId_SetWidth;
    }
    else if ((MethodName == "setPrecision") || (MethodName == "setprecision")) {
        return MethodId_SetPrecision;
    }
    else if ((MethodName == "setFill") || (MethodName == "setfill")) {
        return MethodId_SetFill;
    }
    else if ((MethodName == "setBase") || (MethodName == "setbase")) {
        return MethodId_SetBase;
    }
    else if (MethodName == "hex") {
        return MethodId_Hex;
    }
    else if (MethodName == "dec") {
        return MethodId_Dec;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaFormatterObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;
    
    switch (MethodId) {
      case MethodId_Put:
        Result = Put(ArgumentList, ReturnValue);
	break;
      case MethodId_Flush:
        Result = Flush(ArgumentList, ReturnValue);
	break;
      case MethodId_SetWidth:
        Result = SetWidth(ArgumentList, ReturnValue);
	break;
      case MethodId_SetPrecision:
        Result = SetPrecision(ArgumentList, ReturnValue);
	break;
      case MethodId_SetFill:
        Result = SetFill(ArgumentList, ReturnValue);
	break;
      case MethodId_SetBase:
        Result = SetBase(ArgumentList, ReturnValue);
	break;
      case MethodId_Hex:
        Result = Hex(ArgumentList, ReturnValue);
	break;
      case MethodId_Dec:
        Result = Dec(ArgumentList, ReturnValue);
	break;
      default:
	break;
    }

    return Result;
}

int TParaFormatterObject::Flush(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    (*_FormatStream) << ends;

    string Value = _FormatStream->str(); 
    // _FormatStream->freeze(0);

    delete _FormatStream;
    _FormatStream = new ostrstream();

    ReturnValue = TParaValue(Value);

    return 1;
}

int TParaFormatterObject::Put(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	if (ArgumentList[i]->IsLong()) {
	    (*_FormatStream) << ArgumentList[i]->AsLong();
	}
	else if (ArgumentList[i]->IsDouble()) {
	    (*_FormatStream) << ArgumentList[i]->AsDouble();
	}
	else {
	    (*_FormatStream) << ArgumentList[i]->AsString();
	}	
    }
    
    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaFormatterObject::SetWidth(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("setWidth(): incorrect argument");
    }

    int Width = ArgumentList[0]->AsLong();
    (*_FormatStream) << setw(Width);

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaFormatterObject::SetPrecision(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("setPrecision(): incorrect argument");
    }

    int Precision = ArgumentList[0]->AsLong();
    (*_FormatStream) << setprecision(Precision);

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaFormatterObject::SetFill(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("setFill(): incorrect argument");
    }

    char Fill = ArgumentList[0]->AsString()[0];
    (*_FormatStream) << setfill(Fill);

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaFormatterObject::SetBase(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("setBase(): incorrect argument");
    }

    int Base = ArgumentList[0]->AsLong();
    if (Base == 10) {
	(*_FormatStream) << dec;
    }
    else if (Base == 16) {
	(*_FormatStream) << hex;
    }
    else {
	(*_FormatStream) << setbase(Base);
    }

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaFormatterObject::Hex(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    (*_FormatStream) << hex;

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaFormatterObject::Dec(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    (*_FormatStream) << dec;

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}



TParaScannerObject::TParaScannerObject(void)
: TParaObjectPrototype("Scanner")
{
    _Source = 0;
    _SourceStream = 0;
}

TParaScannerObject::~TParaScannerObject()
{
    delete _SourceStream;
    delete[] _Source;
}

TParaObjectPrototype* TParaScannerObject::Clone(void)
{
    return new TParaScannerObject();
}

void TParaScannerObject::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() > 0) {
	if (! ArgumentList[0]->IsString()) {
	    throw TScriptException(
		_InternalClassName + "::" + _InternalClassName + "()", 
		"invalid argumrnt[s]"
	    );
	}
	
	TParaValue ReturnValue;
	Load(ArgumentList, ReturnValue);
    }
}

int TParaScannerObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "load") {
        return MethodId_Load;
    }
    else if (MethodName == "get") {
        return MethodId_Get;
    }
    else if ((MethodName == "setBase") || (MethodName == "setbase")) {
        return MethodId_SetBase;
    }
    else if ((MethodName == "isGood") || (MethodName == "good")) {
        return MethodId_IsGood;
    }
    else if ((MethodName == "LastGetCount") || (MethodName == "gcount")) {
        return MethodId_LastGetCount;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaScannerObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;
    
    switch (MethodId) {
      case MethodId_Load:
        Result = Load(ArgumentList, ReturnValue);
	break;
      case MethodId_Get:
        Result = Get(ArgumentList, ReturnValue);
	break;
      case MethodId_SetBase:
        Result = SetBase(ArgumentList, ReturnValue);
	break;
      case MethodId_IsGood:
        Result = IsGood(ArgumentList, ReturnValue);
	break;
      case MethodId_LastGetCount:
        Result = LastGetCount(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TParaScannerObject::Load(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::load()", "too few argumrnt[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::load()", "invalid argumrnt[s]"
	);
    }

    string SourceString = ArgumentList[0]->AsString();

    delete[] _Source;
    delete _SourceStream;

    size_t Length = SourceString.size();
    _Source = new char[Length + 1];
    SourceString.copy(_Source, SourceString.size());
    _Source[Length] = '\0';

    _SourceStream = new istrstream(_Source);

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaScannerObject::Get(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (_SourceStream == 0) {
	throw TScriptException(
	    _InternalClassName + "::get()", "empty buffer"
	);
    }

    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsLeftValue())) {
	throw TScriptException(
	    _InternalClassName + "::get(...)", "invalid argument"
	);
    }

    if (ArgumentList[0]->IsVariant() || ArgumentList[0]->IsString()) {
	string Value;
	(*_SourceStream) >> Value;
	ArgumentList[0]->Assign(TParaValue(Value));
    }
    else if (ArgumentList[0]->IsLong()) {
	long Value;
	(*_SourceStream) >> Value;
	ArgumentList[0]->Assign(TParaValue(Value));
    }
    else if (ArgumentList[0]->IsDouble()) {
	double Value;
	(*_SourceStream) >> Value;
	ArgumentList[0]->Assign(TParaValue(Value));
    }
    else {
	throw TScriptException(
	    _InternalClassName + "::get(...)", "invalid argument type"
	);
    }

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaScannerObject::SetBase(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("setBase(): incorrect argument");
    }

    int Base = ArgumentList[0]->AsLong();
    if (Base == 10) {
	(*_SourceStream) >> dec;
    }
    else if (Base == 16) {
	(*_SourceStream) >> hex;
    }
    else {
	(*_SourceStream) >> setbase(Base);
    }

    ReturnValue = TParaValue((TParaObjectPrototype*) this);

    return 1;
}

int TParaScannerObject::IsGood(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue((long) _SourceStream->good());

    return 1;
}

int TParaScannerObject::LastGetCount(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue((long) _SourceStream->gcount());

    return 1;
}



TParaArgumentObject::TParaArgumentObject(int argc, char** argv)
: TParaObjectPrototype("Argument")
{
    _argc = argc;
    _argv = argv;

    _IsParsed = false;
}

TParaArgumentObject::~TParaArgumentObject()
{
}

TParaObjectPrototype* TParaArgumentObject::Clone(void)
{
    return new TParaArgumentObject(_argc, _argv);
}

int TParaArgumentObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "numberOfArguments") {
        return MethodId_NumberOfArguments;
    }
    else if (MethodName == "getArgumentOf") {
        return MethodId_GetArgumentOf;
    }
    else if (MethodName == "numberOfParameters") {
        return MethodId_NumberOfParameters;
    }
    else if (MethodName == "getParameterOf") {
        return MethodId_GetParameterOf;
    }
    else if (MethodName == "isOptionSpecified") {
        return MethodId_IsOptionSpecified;
    }
    else if (MethodName == "getOptionValueOf") {
        return MethodId_GetOptionValueOf;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaArgumentObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (! _IsParsed) {
	Parse();
    }

    int Result = 0;
    
    switch (MethodId) {
      case MethodId_NumberOfArguments:
        Result = NumberOfArguments(ArgumentList, ReturnValue);
	break;
      case MethodId_GetArgumentOf:
        Result = GetArgumentOf(ArgumentList, ReturnValue);
	break;
      case MethodId_NumberOfParameters:
        Result = NumberOfParameters(ArgumentList, ReturnValue);
	break;
      case MethodId_GetParameterOf:
        Result = GetParameterOf(ArgumentList, ReturnValue);
	break;
      case MethodId_IsOptionSpecified:
        Result = IsOptionSpecified(ArgumentList, ReturnValue);
	break;
      case MethodId_GetOptionValueOf:
        Result = GetOptionValueOf(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TParaArgumentObject::NumberOfArguments(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue((long) _ArgumentList.size());

    return 1;
}

int TParaArgumentObject::GetArgumentOf(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("getArgumentOf(int index): incorrect argument");
    }

    int Index = ArgumentList[0]->AsLong();
    string Argument;
    if (Index < (int) _ArgumentList.size()) {
	Argument = _ArgumentList[Index];
    }

    ReturnValue = TParaValue(Argument);

    return 1;
}

int TParaArgumentObject::IsOptionSpecified(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("isOptionSpecified(): incorrect argument");
    }

    string Name = ArgumentList[0]->AsString();

    ReturnValue = TParaValue((long) (_OptionTable.count(Name) > 0));

    return 1;
}

int TParaArgumentObject::GetOptionValueOf(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("getOptionValueOf(): incorrect argument");
    }

    string Name = ArgumentList[0]->AsString();
    string Value;
    if (_OptionTable.count(Name) > 0) {
	Value = _OptionTable[Name];
    }

    ReturnValue = TParaValue(Value);

    return 1;
}

int TParaArgumentObject::NumberOfParameters(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue((long) _ParameterList.size());

    return 1;
}

int TParaArgumentObject::GetParameterOf(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("getParameterOf(int index): incorrect argument");
    }

    int Index = ArgumentList[0]->AsLong();
    string Parameter;
    if (Index < (int) _ParameterList.size()) {
	Parameter = _ParameterList[Index];
    }

    ReturnValue = TParaValue(Parameter);

    return 1;
}

void TParaArgumentObject::Parse(void)
{
    if (_argc > 0) {
	_ArgumentList.push_back(_argv[0]);
    }

    for (int i = 1; i < _argc; i++) {
	string Argument = _argv[i];
	_ArgumentList.push_back(Argument);

	if ((Argument[0] == '-') && (Argument != "-") && (Argument != "--")) {
	    unsigned NameLength = Argument.find_first_of('=');
	    string Name = Argument.substr(0, NameLength);
	    string Value = "";
	    if (NameLength != string::npos) {
		Value = Argument.substr(NameLength + 1, Argument.size());
	    }

	    _OptionTable[Name] = Value;
	}
	else {
	    _ParameterList.push_back(Argument);
	}
    }

    _IsParsed = true;
}



TParaSystemObject::TParaSystemObject(void)
: TParaObjectPrototype("System")
{
}

TParaSystemObject::~TParaSystemObject()
{
}

TParaObjectPrototype* TParaSystemObject::Clone(void)
{
    return new TParaSystemObject();
}

int TParaSystemObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "system") {
        return MethodId_System;
    }
    else if (MethodName == "shell") {
        return MethodId_Shell;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaSystemObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_System:
        Result = System(ArgumentList, ReturnValue);
	break;
      case MethodId_Shell:
        Result = Shell(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TParaSystemObject::System(std::vector<TParaValue*>& ArgumentList, TParaValue& Result) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsString())) {
	throw TScriptException(
	    "system(string command)", "invalid argument[s]"
	);
    }

    string Command = ArgumentList[0]->AsString();
    long ReturnValue = system(Command.c_str());

    Result = TParaValue(ReturnValue);

    return 1;
}

int TParaSystemObject::Shell(std::vector<TParaValue*>& ArgumentList, TParaValue& Result) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsString())) {
	throw TScriptException(
	    "shell(string command)", "invalid argument[s]"
	);
    }

    string Command = ArgumentList[0]->AsString();

    FILE* Pipe = popen(Command.c_str(), "r");

    string ReturnValue;
    char Buffer[1024];
    while (fgets(Buffer, sizeof(Buffer), Pipe) != NULL) {
	ReturnValue += Buffer;
    }

    pclose(Pipe);

    Result = TParaValue(ReturnValue);

    return 1;
}
