/* RoomCamacAccess.cc */
/* Created by Enomoto Sanshiro on 6 October 1997. */
/* Last updated by Enomoto Sanshiro on 9 August 2002. */


#include <strstream>
#include <unistd.h>
#include "RoomCamacAccess.hh"

using namespace std;


TRoomCamacController::TRoomCamacController(void)
{
}

TRoomCamacController::~TRoomCamacController()
{
}

void TRoomCamacController::Close(void)
{
}

bool TRoomCamacController::IsSignalOnInterruptAvailable(void)
{
    return false;
}



TRoomNullCamacController::TRoomNullCamacController(bool IsErrorChecker)
{
    _IsErrorChecker = IsErrorChecker;
}

TRoomNullCamacController::~TRoomNullCamacController()
{
}

TRoomCamacController* TRoomNullCamacController::CloneController(void)
{
    return new TRoomNullCamacController(_IsErrorChecker);
}

void TRoomNullCamacController::Open(void) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::Open()",
	    "CAMAC controller is not connected"
	);
    }
}

void TRoomNullCamacController::Close(void)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::Close()",
	    "CAMAC controller is not connected"
	);
    }
}

int TRoomNullCamacController::Transact(int StationNumber, int Function, int Address, int &Data, int &Q, int &X) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::Transact()",
	    "CAMAC controller is not connected"
	);
    }

    X = 1;
    Q = 0;
    Data = 0;

    return 0;
}

int TRoomNullCamacController::Initialize(void) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::Initialize()",
	    "CAMAC controller is not connected"
	);
    }

    return 0;
}

int TRoomNullCamacController::Clear(void) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::Clear(",
	    "CAMAC controller is not connected"
	);
    }

    return 0;
}

int TRoomNullCamacController::SetInhibition(void) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::SetInhibition()",
	    "CAMAC controller is not connected"
	);
    }

    return 0;
}

int TRoomNullCamacController::ReleaseInhibition(void) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::ReleaseInhibition()",
	    "CAMAC controller is not connected"
	);
    }

    return 0;
}

int TRoomNullCamacController::EnableInterruptNotification(void) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::EnableInterruptNotification()",
	    "CAMAC controller is not connected"
	);
    }

    return 0;
}

int TRoomNullCamacController::DisableInterruptNotification(void) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::DisableInterruptNotification()",
	    "CAMAC controller is not connected"
	);
    }

    return 0;
}

int TRoomNullCamacController::ReadLam(int LamMask) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::ReadLam()",
	    "CAMAC controller is not connected"
	);
    }

    return 0;
}

int TRoomNullCamacController::WaitLam(int TimeOut_sec) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullCamacController::WaitLam()",
	    "CAMAC controller is not connected"
	);
    }

    sleep(TimeOut_sec);
    return 0;
}



TRoomCamacCrate::TRoomCamacCrate(void)
{
    _CamacController = 0;
}

TRoomCamacCrate::TRoomCamacCrate(TRoomCamacController *Controller)
{
    _CamacController = Controller;
}

TRoomCamacCrate::~TRoomCamacCrate()
{
}

void TRoomCamacCrate::InstallController(TRoomCamacController *Controller)
{
    _CamacController = Controller;    
    _CamacController->Open();
}

void TRoomCamacCrate::Install(TRoomCamacModule *CamacModule, int StationNumber) throw(THardwareException)
{
    if (_CamacController == 0) {
        throw THardwareException(
            "TRoomCamacCrate::Install()", "controller is not installed"
        );
    }

    CamacModule->Attach(this, StationNumber);
    _ModuleList.push_back(CamacModule);
}

void TRoomCamacCrate::Uninstall(TRoomCamacModule *CamacModule)
{
    CamacModule->Detach();
}

TRoomCamacController* TRoomCamacCrate::CamacController(void) const
{
    return _CamacController;
}



TRoomCamacModule::TRoomCamacModule(void)
: TRoomModule("Unknown-Type", "Unknown-Model")
{
    _NullCamacController = new TRoomNullCamacController(true);

    _CamacController = _NullCamacController;
    _CamacCrate = 0;
    _StationNumber = 0;
    _StationNumberBit = 0;

    _IsLamEnabled = 0;
}

TRoomCamacModule::TRoomCamacModule(const string& ModuleType, const string& ModelName)
: TRoomModule(ModuleType, ModelName)
{
    _NullCamacController = new TRoomNullCamacController(true);

    _CamacController = _NullCamacController;
    _CamacCrate = 0;
    _StationNumber = 0;
    _StationNumberBit = 0;

    _IsLamEnabled = 0;
}

TRoomCamacModule::~TRoomCamacModule()
{
    delete _NullCamacController;
}

void TRoomCamacModule::Destroy(void)
{
    if (_CamacCrate != 0) {
	_CamacCrate->Uninstall(this);
    }
}

void TRoomCamacModule::Attach(TRoomCamacCrate* Crate, int StationNumber)
{
    _CamacCrate = Crate;
    _CamacController = Crate->CamacController();

    _StationNumber = StationNumber;
    _StationNumberBit = Bit(_StationNumber - 1);
}

void TRoomCamacModule::Detach()
{
    _CamacCrate = 0;
    _CamacController = _NullCamacController;
}

int TRoomCamacModule::Transact(int Function, int Address, int &Data, int &Q, int &X) 
{
    _CamacController->Transact(
	_StationNumber, Function, Address, Data, Q, X
    );

    return Q;
}

int TRoomCamacModule::Transact(int Function, int Address, int &Data) throw(THardwareException)
{
    int Q, X;

    _CamacController->Transact(
	_StationNumber, Function, Address, Data, Q, X
    );

    if (! X) {
	ostrstream Message;
	Message << "No X-response at station " << _StationNumber << "." << ends;
	throw THardwareException(
	    "TRoomCamacModule::Transact()", Message.str()
	);
    }

    return Q;
}

int TRoomCamacModule::Transact(int Function, int Address) throw(THardwareException)
{
    int Data;
    int Q, X;

    _CamacController->Transact(
	_StationNumber, Function, Address, Data, Q, X
    );

    if (! X) {
	ostrstream Message;
	Message << "No X-response at station " << _StationNumber << "." << ends;
	throw THardwareException(
	    "TRoomCamacModule::Transact()", Message.str()
	);
    }

    return Q;
}

int TRoomCamacModule::EnableLam(void) throw(THardwareException)
{
    if (_IsLamEnabled) {
	return 0;
    }

    int Data;
    int Q, X;
    int Status = _CamacController->Transact(
        _StationNumber, fnEnable, adAny, Data, Q, X
    );

#if 0
    /* Some of CAMAC modules do not support F26. */
    /* In such case, No-X or No-Q will be returned. */
    if (! X) {
	throw THardwareException(
            "TRoomCamacModule::EnableLam()", "LAM enabling fault (NoX)"
        );
    }
    if (! Q) {
	throw THardwareException(
            "TRoomCamacModule::EnableLam()", "LAM enabling fault (NoQ)"
        );
    }
#endif

    _IsLamEnabled = 1;

    return Status;
}

int TRoomCamacModule::DisableLam(void) throw(THardwareException)
{
    if (! _IsLamEnabled) {
	return 0;
    }

    int Data;
    int Q, X;
    int Status = _CamacController->Transact(
        _StationNumber, fnDisable, adAny, Data, Q, X
    );

#if 0
    /* Some of CAMAC modules do not support F24. */
    /* In such case, No-X or No-Q will be returned. */
    if (! X) {
	throw THardwareException(
            "TRoomCamacModule::DisableLam()", "LAM disabling fault (NoX)"
        );
    }
    if (! Q) {
	throw THardwareException(
            "TRoomCamacModule::DisableLam()", "LAM disabling fault (NoQ)"
        );
    }
#endif

    _IsLamEnabled = 0;

    return Status;
}

int TRoomCamacModule::ClearLam(void) throw(THardwareException)
{
    int Data;
    int Q, X;
    _CamacController->Transact(
	_StationNumber, fnClearLam, adAny, Data, Q, X
    );

    return Q;
}

bool TRoomCamacModule::WaitLam(int TimeOut_sec) throw(THardwareException)
{
    if (! _IsLamEnabled) {
	EnableLam();
    }

    int Result = _CamacController->WaitLam(TimeOut_sec);
    if (Result < 0) {
        // timed out
        return false;
    }
    else {
        return ((Result & _StationNumberBit) != 0);
    }
}

bool TRoomCamacModule::IsRequestingLam(void)
{
    return _CamacController->ReadLam(_StationNumberBit);
}

int TRoomCamacModule::Read(int Address, int& Data) throw(THardwareException)
{
    return Transact(fnRead, Address, Data);
}

int TRoomCamacModule::Write(int Address, int Data) throw(THardwareException)
{
    return Transact(fnWrite, Address, Data);
}

int TRoomCamacModule::ReadRegister(int Address, int& Data) throw(THardwareException)
{
    return Transact(fnRead, Address, Data);
}

int TRoomCamacModule::WriteRegister(int Address, int Data) throw(THardwareException)
{
    return Transact(fnWrite, Address, Data);
}

int TRoomCamacModule::Clear(int Address) throw(THardwareException)
{
    if (Address < 0) {
	Address = 0;
    }

    return Transact(fnClear, Address);
}

bool TRoomCamacModule::HasData(int Address) throw(THardwareException)
{
    return IsRequestingLam();
}

bool TRoomCamacModule::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    return WaitLam(TimeOut_sec);
}

int TRoomCamacModule::EnableInterrupt(int SignalId) throw(THardwareException)
{
    return EnableLam();
}

int TRoomCamacModule::DisableInterrupt(void) throw(THardwareException)
{
    return DisableLam();
}

int TRoomCamacModule::ClearInterrupt(void) throw(THardwareException)
{
    return ClearLam();
}

int TRoomCamacModule::AddressBitLength(void)
{
    return 8;
}

int TRoomCamacModule::DataBitLength(void)
{
    return 24;
}

void TRoomCamacModule::EnableServiceRequest(void) throw(THardwareException)
{
    EnableLam();
}

void TRoomCamacModule::DisableServiceRequest(void) throw(THardwareException)
{
    DisableLam();
}

void TRoomCamacModule::ClearServiceRequest(void) throw(THardwareException)
{
    //...
    // Some modules (FADC etc.) need to be Clear()ed bofore Enable()d,
    // and usually Enable()s are called in sequence. In such case
    // Clear() must not be called after the sequence.
    // On the other hand, some modules (Input Register etc.) clears data 
    // when ClearInterrupt() is called, in such case, ClearSRQ cannot be
    // done before the sequence is launched.
    // (Current implimentation is to call ClearSRQ() after sequences
    // are completed.)

    // For now, ClearSRQ() of CAMAC modules does not do anything
    // because most of CAMAC modules clear its LAM when data is cleared.

    // ClearLam();
}

bool TRoomCamacModule::IsRequestingService(void) throw(THardwareException)
{
    return IsRequestingLam();
}

bool TRoomCamacModule::IsSignalOnSerivceRequestAvailable(void)
{
    return _CamacController->IsSignalOnInterruptAvailable();
}
