/* DeviceFactory.cc */
/* Created by Enomoto Sanshiro on 7 May 1999. */
/* Last updated by Enomoto Sanshiro on 7 July 2001. */


#include <string>
#include <map>
#include "RoomModule.hh"
#include "RoomVmeAccess.hh"
#include "RoomCamacAccess.hh"
#include "RoomVmeBridge.hh"
#include "RoomSoftwareDevice.hh"
#include "RoomDeviceFactory.hh"

using namespace std;


TRoomDeviceFactory* TRoomDeviceFactory::_Instance = 0;

TRoomDeviceFactory::TRoomDeviceFactory()
{
    _VmeControllerTable["Null"] = new TRoomNullVmeController(false);
    _CamacControllerTable["Null"] = new TRoomNullCamacController(false);
}

TRoomDeviceFactory::~TRoomDeviceFactory()
{
}

TRoomDeviceFactory* TRoomDeviceFactory::GetInstance(void)
{
    if (_Instance == 0) {
        _Instance = new TRoomDeviceFactory();
    }

    return _Instance;
}

string TRoomDeviceFactory::ReducedNameOf(string Name) const
{
    // replace '-' or ' ' with '_'
    for (unsigned i = 0; i < Name.size(); i++) {
	if ((Name[i] == '-') || (Name[i] == ' ')) {
	    Name[i] = '_';
	}
    }

    //... OBSOLETE: This code will be deleted
#if 0
    // if Name contains more than three words, remove the first word.
    if (Name.find('_') != Name.rfind('_')) {
	return Name.substr(Name.find('_') + 1);
    }
    else {
	return Name;
    }
#endif

    return Name;
}

void TRoomDeviceFactory::RegisterVmeController(const string& Name, TRoomVmeController* Controller)
{
    _VmeControllerTable[Name] = Controller;
}

void TRoomDeviceFactory::RegisterCamacController(const string& Name, TRoomCamacController* Controller)
{
    _CamacControllerTable[Name] = Controller;
}

void TRoomDeviceFactory::RegisterVmeCamacBridge(const string& Name, TRoomVmeCamacBridge* Bridge)
{
    _VmeCamacBridgeTable[Name] = Bridge;
}

void TRoomDeviceFactory::RegisterVmeModule(const string& Name, TRoomVmeModule* Module)
{
    _VmeModuleTable[Name] = Module;
}

void TRoomDeviceFactory::RegisterCamacModule(const string& Name, TRoomCamacModule* Module)
{
    _CamacModuleTable[Name] = Module;
}

void TRoomDeviceFactory::RegisterSoftwareModule(const string& Name, TRoomSoftwareModule* Module)
{
    _SoftwareModuleTable[Name] = Module;
}

TRoomVmeController* TRoomDeviceFactory::CreateVmeController(const string& Name) throw(THardwareException)
{
    TRoomVmeController* ControllerPrototype;

    string ReducedName = ReducedNameOf(Name);
    if (_VmeControllerTable.count(ReducedName) > 0) {
	ControllerPrototype = _VmeControllerTable[ReducedName];
    }
    else {
	throw THardwareException(
	    "TRoomDeviceFactory::CreateVmeController()",
	    "No such VME controller: " + Name
	);
    }

    return ControllerPrototype->CloneController();
}

TRoomCamacController* TRoomDeviceFactory::CreateCamacController(const string& Name) throw(THardwareException)
{
    TRoomCamacController* ControllerPrototype = 0;

    string ReducedName = ReducedNameOf(Name);
    if (_CamacControllerTable.count(ReducedName) > 0) {
	ControllerPrototype = _CamacControllerTable[ReducedName];
    }
    else {
	throw THardwareException(
	    "TRoomDeviceFactory::CreateCamacController()",
	    "No such CAMAC controller: " + Name
	);
    }

    return ControllerPrototype->CloneController();
}

TRoomVmeCamacBridge* TRoomDeviceFactory::CreateVmeCamacBridge(const string& Name) throw(THardwareException)
{
    TRoomVmeCamacBridge* Prototype = 0;

    string ReducedName = ReducedNameOf(Name);
    if (_VmeCamacBridgeTable.count(ReducedName) > 0) {
	Prototype = _VmeCamacBridgeTable[ReducedName];
    }
    else {
	throw THardwareException(
	    "TRoomDeviceFactory::CreateVmeCamacBridge()",
	    "No such VME-CAMAC bridge: " + Name
	);
    }

    return Prototype->CloneBridge();
}

TRoomVmeCrate* TRoomDeviceFactory::CreateVmeCrate(const string& Name) throw(THardwareException)
{
    return new TRoomVmeCrate();
}

TRoomCamacCrate* TRoomDeviceFactory::CreateCamacCrate(const string& Name) throw(THardwareException)
{
    return new TRoomCamacCrate();
}

TRoomVmeModule* TRoomDeviceFactory::CreateVmeModule(const string& Name) throw(THardwareException)
{
    string ReducedName = ReducedNameOf(Name);

    if (_VmeModuleTable.count(ReducedName) == 0) {
        throw THardwareException(
            "TRoomDeviceFactory::CreateVmeModule()",
            "No such VME module: " + Name
        );
    }

    return _VmeModuleTable[ReducedName]->Clone();
}

TRoomCamacModule* TRoomDeviceFactory::CreateCamacModule(const string& Name) throw(THardwareException)
{
    string ReducedName = ReducedNameOf(Name);

    if (_CamacModuleTable.count(ReducedName) == 0) {
        throw THardwareException(
            "TRoomDeviceFactory::CreateCamacModule()",
            "No such CAMAC module: " + Name
        );
    }

    return _CamacModuleTable[ReducedName]->Clone();
}

TRoomSoftwareModule* TRoomDeviceFactory::CreateSoftwareModule(const string& Name) throw(THardwareException)
{
    if (_SoftwareModuleTable.count(Name) == 0) {
        throw THardwareException(
            "TRoomDeviceFactory::CreateSoftwareModule()",
            "No such software module: " + Name
        );
    }

    return _SoftwareModuleTable[Name]->Clone();
}

void TRoomDeviceFactory::ShowDeviceList(std::ostream& os) const
{
    cout << "VME Controllers:" << endl;
    map<string, TRoomVmeController*>::const_iterator VmeController;
    for (
	 VmeController = _VmeControllerTable.begin();
	 VmeController != _VmeControllerTable.end();
	 VmeController++
    ){
        os << "    " << VmeController->first << ": ";
	os << endl;
    }
    os << endl;

    cout << "CAMAC Controllers:" << endl;
    map<string, TRoomCamacController*>::const_iterator CamacController;
    for (
	 CamacController = _CamacControllerTable.begin();
	 CamacController != _CamacControllerTable.end();
	 CamacController++
    ){
        os << "    " << CamacController->first << ": ";
	os << endl;
    }
    os << endl;

    cout << "VME-CAMAC Bridges:" << endl;
    map<string, TRoomVmeCamacBridge*>::const_iterator VmeCamacBridge;
    for (
	 VmeCamacBridge = _VmeCamacBridgeTable.begin();
	 VmeCamacBridge != _VmeCamacBridgeTable.end();
	 VmeCamacBridge++
    ){
        os << "    " << VmeCamacBridge->first << ": ";
	os << endl;
    }
    os << endl;

    cout << "VME Modules:" << endl;
    map<string, TRoomVmeModule*>::const_iterator VmeModule;
    for (
	 VmeModule = _VmeModuleTable.begin();
	 VmeModule != _VmeModuleTable.end();
	 VmeModule++
    ){
        os << "    " << VmeModule->first << ": ";
	os << VmeModule->second->ModelName();
	os << endl;
    }
    os << endl;

    cout << "CAMAC Modules:" << endl;
    map<string, TRoomCamacModule*>::const_iterator CamacModule;
    for (
	 CamacModule = _CamacModuleTable.begin();
	 CamacModule != _CamacModuleTable.end();
	 CamacModule++
    ){
        os << "    " << CamacModule->first << ": ";
	os << CamacModule->second->ModelName();
	os << endl;
    }
    os << endl;

    cout << "Software Modules:" << endl;
    map<string, TRoomSoftwareModule*>::const_iterator SoftwareModule;
    for (
	 SoftwareModule = _SoftwareModuleTable.begin();
	 SoftwareModule != _SoftwareModuleTable.end();
	 SoftwareModule++
    ){
        os << "    " << SoftwareModule->first << ": ";
	os << SoftwareModule->second->ModelName();
	os << endl;
    }
    os << endl;
}



TRoomVmeControllerCreator::TRoomVmeControllerCreator(const string& Name, TRoomVmeController* Controller)
{
    TRoomDeviceFactory::GetInstance()->RegisterVmeController(Name, Controller);
}

TRoomCamacControllerCreator::TRoomCamacControllerCreator(const string& Name, TRoomCamacController* Controller)
{
    TRoomDeviceFactory::GetInstance()->RegisterCamacController(Name, Controller);
}

TRoomVmeCamacBridgeCreator::TRoomVmeCamacBridgeCreator(const string& Name, TRoomVmeCamacBridge* Bridge)
{
    TRoomDeviceFactory::GetInstance()->RegisterVmeCamacBridge(Name, Bridge);
}

TRoomVmeModuleCreator::TRoomVmeModuleCreator(const string& Name, TRoomVmeModule* Module)
{
    TRoomDeviceFactory::GetInstance()->RegisterVmeModule(Name, Module);
}

TRoomCamacModuleCreator::TRoomCamacModuleCreator(const string& Name, TRoomCamacModule* Module)
{
    TRoomDeviceFactory::GetInstance()->RegisterCamacModule(Name, Module);
}

TRoomSoftwareModuleCreator::TRoomSoftwareModuleCreator(const string& Name, TRoomSoftwareModule* Module)
{
    TRoomDeviceFactory::GetInstance()->RegisterSoftwareModule(Name, Module);
}
