/* RoomVmeAccess.cc */
/* Created by Enomoto Sanshiro on 6 October 1997. */
/* Last updated by Enomoto Sanshiro on 9 August 2002. */


#include <cerrno>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include "RoomVmeAccess.hh"

using namespace std;


static const int VmeTransferDataSize[] = {
    /* this must be consistent with TVmeTarnsferMode */
    /* (declared in RoomVmeAccess.hh) */
    sizeof(Word), sizeof(DoubleWord),
    sizeof(Word), sizeof(DoubleWord),
    sizeof(Word), sizeof(DoubleWord)
};


TRoomVmeController::TRoomVmeController(void)
{
}

TRoomVmeController::~TRoomVmeController()
{
}

TRoomVmeAccessProperty* TRoomVmeController::Open(TRoomVmeTransferMode TransferMode) throw(THardwareException)
{
    if (TransferMode >= VmeTransferMode_Unknown) {
	throw THardwareException(
            "TRoomVmeController::Open()", "bad vme transfer mode"
        );	
    }

    int DeviceDescriptor = open(DeviceFileName(TransferMode), O_RDWR);
    if (DeviceDescriptor < 0) {
	throw THardwareException(
            "TRoomVmeController::Open(): open(2)",
            strerror(errno)
        );
    }

    TRoomVmeAccessProperty* Property = new TRoomVmeAccessProperty;
    Property->TransferMode = TransferMode;
    Property->VmeAddress = 0;
    Property->DeviceDescriptor = DeviceDescriptor;
    Property->DmaDeviceDescriptor = 0;
    Property->MappedAddress = 0;
    Property->MapSize = 0;
    Property->MapOffset = 0;
    Property->InterruptNumber = 0;
    Property->InterruptVector = 0;
    Property->DeviceSpecificData = 0;

    Seek(Property, 0);
    
    return Property;
}

void TRoomVmeController::Close(TRoomVmeAccessProperty* Property)
{
    DmaClose(Property);
    close(Property->DeviceDescriptor);
    delete Property;
}

long TRoomVmeController::Read(TRoomVmeAccessProperty* Property, void* Buffer, int Size) throw(THardwareException)
{
    long ReadSize = read(Property->DeviceDescriptor, Buffer, Size);

    if (ReadSize < 0) {
	throw THardwareException(
            "TRoomVmeController::Read(): read(2)",
            strerror(errno)
        );
    }

    return ReadSize;
}

long TRoomVmeController::Write(TRoomVmeAccessProperty* Property, const void* Buffer, int Size) throw(THardwareException)
{
    long WrittenSize = write(Property->DeviceDescriptor, Buffer, Size);

    if (WrittenSize < 0) {
	throw THardwareException(
            "TRoomVmeController::Write(): write(2)",
            strerror(errno)
        );
    }

    return WrittenSize;
}

off_t TRoomVmeController::Seek(TRoomVmeAccessProperty* Property, off_t OffsetAddress) throw(THardwareException)
{
    off_t SeekPosition = Property->VmeAddress + OffsetAddress;

    off_t NewPosition;
    NewPosition = lseek(Property->DeviceDescriptor, SeekPosition, SEEK_SET);

    if (NewPosition < 0) {
	throw THardwareException(
            "TRoomVmeController::Seek(): lseek(2)",
            strerror(errno)
        );	
    }

    return NewPosition;
}

caddr_t TRoomVmeController::Map(TRoomVmeAccessProperty* Property, off_t VmeAddress, size_t MapSize, off_t MapOffset) throw(THardwareException)
{
    caddr_t MappedAddress = 0;

    if (MapSize > 0) {
	caddr_t MapAddress = (caddr_t) 0;  // assigned by kernel
	int MapProtection = PROT_READ | PROT_WRITE;
	int MapFlag = MAP_SHARED;
	MappedAddress = (caddr_t) mmap(
	    MapAddress,
	    MapSize,
	    MapProtection,
	    MapFlag,
	    Property->DeviceDescriptor,
	    VmeAddress + MapOffset
	);
	if (MappedAddress == (caddr_t) MAP_FAILED) {
	    throw THardwareException(
		"TRoomVmeController::Map(): mmap(2)",
		strerror(errno)
	    );
	}
    }

    Property->VmeAddress = VmeAddress;
    Property->MapSize = MapSize;
    Property->MapOffset = MapOffset;
    Property->MappedAddress = MappedAddress;

    return MappedAddress;
}

void TRoomVmeController::Unmap(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    if (Property->MappedAddress > 0) {
	int Result = munmap(
	    (caddr_t) Property->MappedAddress, Property->MapSize
	);
	if (Result < 0) {
	    throw THardwareException(
		"TRoomVmeController::Unmap(): munmap(2)",
		strerror(errno)
	    );	     
	}

	Property->MappedAddress = 0;
    }
}

void TRoomVmeController::DmaOpen(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    TRoomVmeTransferMode TransferMode = Property->TransferMode;
    int DmaDeviceDescriptor = open(DmaDeviceFileName(TransferMode), O_RDWR);
    if (DmaDeviceDescriptor < 0) {
	throw THardwareException(
            "TRoomVmeController::DmaOpen(): open(2)",
            strerror(errno)
        );
    }

    Property->DmaDeviceDescriptor = DmaDeviceDescriptor;
    DmaSeek(Property, 0);
}

void TRoomVmeController::DmaClose(TRoomVmeAccessProperty* Property)
{
    if (Property->DmaDeviceDescriptor > 0) {
	close(Property->DmaDeviceDescriptor);
    }
}

long TRoomVmeController::DmaRead(TRoomVmeAccessProperty* Property, void* Buffer, int Size) throw(THardwareException)
{
    if (Property->DmaDeviceDescriptor <= 0) {
	DmaOpen(Property);
    }  

    long ReadSize = read(Property->DmaDeviceDescriptor, Buffer, Size);

    if (ReadSize < 0) {
	throw THardwareException(
            "TRoomVmeController::DmaRead(): read(2)",
            strerror(errno)
        );
    }

    return ReadSize;
}

long TRoomVmeController::DmaWrite(TRoomVmeAccessProperty* Property, const void* Buffer, int Size) throw(THardwareException)
{
    if (Property->DmaDeviceDescriptor <= 0) {
       DmaOpen(Property);
    }  

    long WrittenSize = write(Property->DmaDeviceDescriptor, Buffer, Size);

    if (WrittenSize < 0) {
	throw THardwareException(
            "TRoomVmeController::DmaWrite(): write(2)",
            strerror(errno)
        );
    }

    return WrittenSize;
}

off_t TRoomVmeController::DmaSeek(TRoomVmeAccessProperty* Property, off_t OffsetAddress) throw(THardwareException)
{
    if (Property->DmaDeviceDescriptor <= 0) {
       DmaOpen(Property);
    }  

    off_t SeekPosition = Property->VmeAddress + OffsetAddress;

    off_t NewPosition;
    NewPosition = lseek(Property->DmaDeviceDescriptor, SeekPosition, SEEK_SET);

    if (NewPosition < 0) {
	throw THardwareException(
            "TRoomVmeController::DmaSeek(): lseek(2)",
            strerror(errno)
        );	
    }

    return NewPosition;
}

bool TRoomVmeController::IsSignalOnInterruptAvailable(void)
{
    return false;
}



TRoomNullVmeController::TRoomNullVmeController(bool IsErrorChecker)
{
    _IsErrorChecker = IsErrorChecker;
}

TRoomNullVmeController::~TRoomNullVmeController()
{
}

TRoomVmeController* TRoomNullVmeController::CloneController(void)
{
    return new TRoomNullVmeController(_IsErrorChecker);
}

caddr_t TRoomNullVmeController::Map(TRoomVmeAccessProperty* Property, off_t VmeAddress, size_t MapSize, off_t MapOffset) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::Map()",
	    "VME controller is not connected"
	);
    }

    caddr_t MappedAddress = (caddr_t) (new U8bit[MapSize]);
    memset(MappedAddress, 0, MapSize);

    Property->VmeAddress = VmeAddress;
    Property->MapSize = MapSize;
    Property->MapOffset = MapOffset;
    Property->MappedAddress = MappedAddress;

    return MappedAddress;
}

void TRoomNullVmeController::Unmap(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::Unmap()",
	    "VME controller is not connected"
	);
    }

    delete[] (U8bit*) Property->MappedAddress;
}

void TRoomNullVmeController::RegisterInterruptNotification(TRoomVmeAccessProperty* Property, int SignalId) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::RegisterInterruptNotification()",
	    "VME controller is not connected"
	);
    }

    Property->InterruptSignalId = SignalId;
}

void TRoomNullVmeController::UnregisterInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::UnregisterInterruptNotification()",
	    "VME controller is not connected"
	);
    }
}

void TRoomNullVmeController::EnableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::EnableInterruptNotification()",
	    "VME controller is not connected"
	);
    }
}

void TRoomNullVmeController::DisableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::DisableInterruptNotification()",
	    "VME controller is not connected"
	);
    }
}

bool TRoomNullVmeController::WaitForInterruptNotification(TRoomVmeAccessProperty* Property, unsigned TimeOut_sec) throw(THardwareException)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::WaitForInterruptNotification()",
	    "VME controller is not connected"
	);
    }

    sleep(TimeOut_sec);
    return false;
}

const char* TRoomNullVmeController::DeviceFileName(TRoomVmeTransferMode TransferMode)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::DeviceFileName()",
	    "VME controller is not connected"
	);
    }

    return "/dev/zero";
}

const char* TRoomNullVmeController::DmaDeviceFileName(TRoomVmeTransferMode TransferMode)
{
    if (_IsErrorChecker) {
	throw THardwareException(
	    "TNullVmeController::DmaDeviceFileName()",
	    "VME controller is not connected"
	);
    }

    return "/dev/zero";
}



TRoomVmeCrate::TRoomVmeCrate(TRoomVmeController *VmeController, int NumberOfSlots) 
{
    _VmeController = VmeController;
    _NumberOfSlots = NumberOfSlots;
}

TRoomVmeCrate::TRoomVmeCrate(int NumberOfSlots) 
{
    _VmeController = 0;
    _NumberOfSlots = NumberOfSlots;
}

TRoomVmeCrate::~TRoomVmeCrate() 
{
}

void TRoomVmeCrate::InstallController(TRoomVmeController *VmeController)
{
    _VmeController = VmeController;
}

void TRoomVmeCrate::Install(TRoomVmeModule *VmeModule, off_t VmeAddress, int InterruptNumber, int InterruptVector) throw(THardwareException)
{
    if (_VmeController == 0) {
        throw THardwareException(
            "TRoomVmeCrate::Install()", "controller is not installed"
        );
    }

    TRoomVmeTransferMode TransferMode = VmeModule->TransferMode();
    size_t MapSize = VmeModule->MapSize();
    off_t MapOffset = VmeModule->MapOffset();

    TRoomVmeAccessProperty* Property = _VmeController->Open(TransferMode);
    caddr_t MappedAddress = _VmeController->Map(Property, VmeAddress, MapSize, MapOffset);
    
    Property->InterruptNumber = InterruptNumber;
    Property->InterruptVector = InterruptVector;
    Property->InterruptSignalId = 0;

    VmeModule->Attach(this, Property, MappedAddress);
    _ModuleList.push_back(VmeModule);
}

void TRoomVmeCrate::Install(TRoomVmeModule *VmeModule, off_t VmeAddress) throw(THardwareException)
{
    int InterruptNumber = 0;
    int Vector = 0;

    Install(VmeModule, VmeAddress, InterruptNumber, Vector);
}

void TRoomVmeCrate::Uninstall(TRoomVmeModule *VmeModule) 
{
    TRoomVmeAccessProperty* Property = VmeModule->VmeAccessProperty();

    VmeModule->Detach();
    _VmeController->Unmap(Property);
    _VmeController->Close(Property);

}

TRoomVmeController* TRoomVmeCrate::VmeController(void) const
{
    return _VmeController;
}



TRoomVmeModule::TRoomVmeModule(TRoomVmeTransferMode TransferMode, size_t MapSize, off_t MapOffset)
: TRoomModule("Unknown-Type", "Unknown-Model")
{
    _TransferMode = TransferMode;
    _MapSize = MapSize;
    _MapOffset = MapOffset;

    _NullVmeController = new TRoomNullVmeController(true);
    _VmeController = _NullVmeController;
    _VmeCrate = 0;
    _VmeAccessProperty = 0;

    _IsInterruptRegistered = false;
    _IsInterruptEnabled = false;
}

TRoomVmeModule::TRoomVmeModule(const string& ModuleType, const string& ModelName, TRoomVmeTransferMode TransferMode, size_t MapSize, off_t MapOffset)
: TRoomModule(ModuleType, ModelName)
{
    _TransferMode = TransferMode;
    _MapSize = MapSize;
    _MapOffset = MapOffset;

    _NullVmeController = new TRoomNullVmeController(true);
    _VmeController = _NullVmeController;
    _VmeCrate = 0;
    _VmeAccessProperty = 0;
}

TRoomVmeModule::~TRoomVmeModule()
{
    delete _NullVmeController;
}

void TRoomVmeModule::Destroy(void)
{
    if (_VmeCrate != 0) {
	_VmeCrate->Uninstall(this);
    }
}

size_t TRoomVmeModule::MapSize(void) const
{
    return _MapSize;
}

off_t TRoomVmeModule::MapOffset(void) const
{
    return _MapOffset;
}

TRoomVmeTransferMode TRoomVmeModule::TransferMode(void) const
{
    return _TransferMode;
}

volatile caddr_t TRoomVmeModule::MappedAddress(void) const
{
    return _MappedAddress;
}

TRoomVmeAccessProperty* TRoomVmeModule::VmeAccessProperty(void) const
{
    return _VmeAccessProperty;
}

long TRoomVmeModule::PioRead(off_t Address, void* Buffer, int Size) throw(THardwareException)
{
    _VmeController->Seek(_VmeAccessProperty, Address);
    return _VmeController->Read(_VmeAccessProperty, Buffer, Size);
}

long TRoomVmeModule::PioWrite(off_t Address, const void* Buffer, int Size) throw(THardwareException)
{
    _VmeController->Seek(_VmeAccessProperty, Address);
    return _VmeController->Write(_VmeAccessProperty, Buffer, Size);
}

long TRoomVmeModule::DmaRead(off_t Address, void* Buffer, int Size) throw(THardwareException)
{
    _VmeController->DmaSeek(_VmeAccessProperty, Address);
    return _VmeController->DmaRead(_VmeAccessProperty, Buffer, Size);
}

long TRoomVmeModule::DmaWrite(off_t Address, const void* Buffer, int Size) throw(THardwareException)
{
    _VmeController->DmaSeek(_VmeAccessProperty, Address);
    return _VmeController->DmaWrite(_VmeAccessProperty, Buffer, Size);
}

int TRoomVmeModule::BlockRead(int Address, void* Data, int MaxSize) throw(THardwareException)
{
    return DmaRead(Address, Data, MaxSize);
}

int TRoomVmeModule::BlockWrite(int Address, const void* Data, int Size) throw(THardwareException)
{
    return DmaWrite(Address, Data, Size);
}

int TRoomVmeModule::ReadRegister(int Address, int& Data) throw(THardwareException)
{
    return PioRead(Address, &Data, VmeTransferDataSize[_TransferMode]);
}

int TRoomVmeModule::WriteRegister(int Address, int Data) throw(THardwareException)
{
    return PioWrite(Address, &Data, VmeTransferDataSize[_TransferMode]);
}

int TRoomVmeModule::EnableInterrupt(int SignalId) throw(THardwareException)
{
    if (SignalId != _VmeAccessProperty->InterruptSignalId) {
	if (_IsInterruptRegistered) {
	    _VmeController->UnregisterInterruptNotification(_VmeAccessProperty);
	    _IsInterruptRegistered = false;
	}
    }

    if (! _IsInterruptRegistered) {
	_VmeController->RegisterInterruptNotification(
	    _VmeAccessProperty, SignalId
	);
	_IsInterruptRegistered = true;
	_IsInterruptEnabled = false;
    }

    if (! _IsInterruptEnabled) {
        _VmeController->EnableInterruptNotification(_VmeAccessProperty);
	_IsInterruptEnabled = true;
    }

    return 0;
}

int TRoomVmeModule::DisableInterrupt(void) throw(THardwareException)
{
    if (_IsInterruptEnabled) {
	_VmeController->DisableInterruptNotification(_VmeAccessProperty);
	_IsInterruptEnabled = false;
    }
	
    return 0;
}

int TRoomVmeModule::ClearInterrupt(void) throw(THardwareException)
{
    return 0;
}

bool TRoomVmeModule::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    if (! _IsInterruptEnabled) {
	EnableInterrupt();
    }

    int WaitResult = _VmeController->WaitForInterruptNotification( 
	_VmeAccessProperty, TimeOut_sec
    );

    return (WaitResult > 0);
}
  
bool TRoomVmeModule::IsSignalOnServiceRequestAvailable(void)
{
    return _VmeController->IsSignalOnInterruptAvailable();
}

void TRoomVmeModule::EnableSignalOnServiceRequest(int SignalId) throw(THardwareException)
{
    EnableInterrupt(SignalId);
}

void TRoomVmeModule::DisableSignalOnServiceRequest(void) throw(THardwareException)
{
    DisableInterrupt();
}

void TRoomVmeModule::Attach(TRoomVmeCrate* Crate, TRoomVmeAccessProperty* Property, caddr_t MappedAddress)
{
    _VmeCrate = Crate;
    _VmeController = Crate->VmeController();
    _VmeAccessProperty = Property;
    _MappedAddress = MappedAddress;
}

void TRoomVmeModule::Detach(void)
{
    if (_IsInterruptRegistered) {
	DisableInterrupt();
	_VmeController->UnregisterInterruptNotification(_VmeAccessProperty);
	_IsInterruptRegistered = false;
    }

    _VmeController = _NullVmeController;

    _VmeCrate = 0;
    _VmeAccessProperty = 0;
    _MappedAddress = 0;
}
