/* bridge-Kinetic_2917.cc */
/* Created by Enomoto Sanshiro on 5 July 2003. */
/* Last updated by Enomoto Sanshiro on 5 July 2003. */


#include <iostream>
#include <unistd.h>
#include "RoomDeviceFactory.hh"
#include "RoomVmeBridge.hh" 
#include "bridge-Kinetic_2917.hh"

using namespace std;


//#define DEBUG(x) (x)
#define DEBUG(x)

//#define DEBUG_INT(x) (x)
#define DEBUG_INT(x)


static const int InitialCrateNumber = 1;
static const int DefaultBufferSize = 0x10000;  // 64kB
static const int WorkingCommandMemoryAddress = 0x7ff0;  // uses last 16 words


#define IsRead(f) ((f & 0x18) == 0)
#define IsWrite(f) ((f & 0x18) == 0x10)
#define IsControl(f) ((f & 0x08) != 0)

#define NAF(n, a, f) (((n) << 9) | ((a) << 5) | (f))


static TRoomVmeCamacBridgeCreator Creator1(
    "Kinetic_2917", new TVmeCamacBridge_Kinetic_2917()
);
static TRoomVmeModuleCreator Creator2(
    "Kinetic_2917", new TVmeCamacBridge_Kinetic_2917()
);


TVmeCamacBridge_Kinetic_2917::TVmeCamacBridge_Kinetic_2917(void)
: TRoomVmeCamacBridge("Kinetic_2917", VmeTransferMode_A16D16, (size_t) 0x100)
{
    _CrateNumber = InitialCrateNumber;
    _BufferSize = DefaultBufferSize;

    _Buffer = 0;
    _NumberOfDataWords = 0;
    _DataWordCounts = 0;

    _IsInterruptInstalled = false;
}

TVmeCamacBridge_Kinetic_2917::~TVmeCamacBridge_Kinetic_2917()
{
    delete[] _Buffer;
}

TRoomVmeCamacBridge* TVmeCamacBridge_Kinetic_2917::CloneBridge(void)
{
    return new TVmeCamacBridge_Kinetic_2917();
}


//// CAMAC Controller Interface ////

void TVmeCamacBridge_Kinetic_2917::Open(void) throw(THardwareException)
{
    DEBUG(cout << "K2917: Open()" << endl);
    WordAt(regCSR) = bitCSR_RST;

    _IsInterruptInstalled = (VmeAccessProperty()->InterruptNumber > 0);
}

void TVmeCamacBridge_Kinetic_2917::Close(void)
{
    DEBUG(cout << "K2917: Close()" << endl);
}

int TVmeCamacBridge_Kinetic_2917::Transact(int StationNumber, int Function, int Address, int &Data, int &Q, int &X) throw(THardwareException)
{
    return ExecuteSingleTransfer(
	_CrateNumber, StationNumber, Function, Address, Data, Q, X
    );
}

int TVmeCamacBridge_Kinetic_2917::Initialize(void) throw(THardwareException)
{
    (cout << "K2917: Initialize()" << endl);

    WordAt(regCSR) = bitCSR_RST;

    // Enable LAM //
    int N, F, A, Data, Q, X;
    ExecuteSingleTransfer(
	_CrateNumber, N = 30, F = 17, A = 13, Data = 0xffffff, Q, X
    );

    // Generate Z, Enable ServiceRequest //
    return ExecuteSingleTransfer(
	_CrateNumber, N = 30, F = 17, A = 0, Data = 0x000101, Q, X
    );
}

int TVmeCamacBridge_Kinetic_2917::Clear(void) throw(THardwareException)
{
    DEBUG(cout << "K2917: Clear()" << endl);

    int N, F, A, Data, Q, X;
    return ExecuteSingleTransfer(
	_CrateNumber, N = 30, F = 17, A = 0, Data = 0x000102, Q, X
    );
}

int TVmeCamacBridge_Kinetic_2917::SetInhibition(void) throw(THardwareException)
{
    DEBUG(cout << "K2917: SetInhibition()" << endl);

    int N, F, A, Data, Q, X;
    return ExecuteSingleTransfer(
	_CrateNumber, N = 30, F = 17, A = 0, Data = 0x000104, Q, X
    );
}

int TVmeCamacBridge_Kinetic_2917::ReleaseInhibition(void) throw(THardwareException)
{
    DEBUG(cout << "K2917: ReleaseInhibition()" << endl);

    int N, F, A, Data, Q, X;
    return ExecuteSingleTransfer(
	_CrateNumber, N = 30, F = 17, A = 0, Data = 0x000100, Q, X
    );
}

int TVmeCamacBridge_Kinetic_2917::EnableInterruptNotification(void) throw(THardwareException)
{
    DEBUG(cout << "K2917: EnableInterruptNotificaton()" << endl);

    return 0;
}

int TVmeCamacBridge_Kinetic_2917::DisableInterruptNotification(void) throw(THardwareException)
{
    DEBUG(cout << "K2917: DisableInterruptNotification()" << endl);

    return 0;
}

int TVmeCamacBridge_Kinetic_2917::ReadLam(int LamMask) throw(THardwareException)
{
    DEBUG(cout << "K2917: ReadLam()" << endl);

    int LamPattern = 0;

    if (WordAt(regSRR) & Bit(_CrateNumber)) {
	int N, F, A, Q, X;
	ExecuteSingleTransfer(
	    _CrateNumber, N = 30, F = 1, A = 12, LamPattern, Q, X
	);
    }

    DEBUG(cout << "K2917: Lam Pattern: " << hex << LamPattern << dec << endl);

    return LamPattern & LamMask;
}

int TVmeCamacBridge_Kinetic_2917::WaitLam(int TimeOut_sec) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: WaitLam()" << endl);

    if (_IsInterruptInstalled) {
        ClearInterrupt();
	TRoomVmeModule::WaitData(TimeOut_sec);
    }
    else {
        int Count = 0;
	while ((WordAt(regSRR) & Bit(_CrateNumber)) == 0) {
	    if (Count++ > 10000) {
	        //usleep(10);
	        if (Count > 100000 * TimeOut_sec) {
		    break;
		}
	    }
	}
    }

    return ReadLam(0xffffff);
}


//// VME Module Interface ////

int TVmeCamacBridge_Kinetic_2917::AddressBitLength(void)
{
    return 8;
}

int TVmeCamacBridge_Kinetic_2917::DataBitLength(void)
{
    return 24;
}

int TVmeCamacBridge_Kinetic_2917::Read(int Address, int &Data) throw(THardwareException)
{
    if (_NumberOfDataWords == 0) {
	if (_Buffer == 0) {
	    _Buffer = new U32bit[_BufferSize / sizeof(U32bit)];
	}

	_NumberOfDataWords = BlockRead(Address, _Buffer, _BufferSize) / sizeof(U32bit);
	_DataWordCounts = 0;
    }

    if (_DataWordCounts < _NumberOfDataWords) {
        Data = _Buffer[_DataWordCounts++];
	return 1;
    }
    else {
        _NumberOfDataWords = 0;
	return 0;
    }
}

int TVmeCamacBridge_Kinetic_2917::SequentialRead(int Address, int Data[], int MaxSize) throw(THardwareException)
{
    if (sizeof(int) == sizeof(U32bit)) {
	return BlockRead(Address, Data, MaxSize*sizeof(int)) / sizeof(int);
    }

    throw THardwareException(
	"TVmeCamacBridge_Kinetic_2917::SequentialRead()",
	"function not implemented yet"
    );

    return 0;
}

int TVmeCamacBridge_Kinetic_2917::NextNumberOfDataElements(int Address) throw(THardwareException)
{
    return _BufferSize / sizeof(int);
}

int TVmeCamacBridge_Kinetic_2917::BlockRead(int Address, void *Data, int MaxSize) throw(THardwareException)
{
    return ExecuteReadList((U32bit*) Data, MaxSize);
}

int TVmeCamacBridge_Kinetic_2917::NextDataBlockSize(int Address) throw(THardwareException)
{
    return _BufferSize;
}

bool TVmeCamacBridge_Kinetic_2917::WaitForServiceRequest(int TimeOut_sec) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: WaitForServiceRequest()..." << flush);

    if (! (WordAt(regSRR) & 0x00ff)) {
        if (_IsInterruptInstalled) {
	    ClearInterrupt();
	    TRoomVmeModule::WaitForServiceRequest(TimeOut_sec);
	}
	else {
	    int Count = 0;
	    while ((WordAt(regSRR) & 0x00ff) == 0) {
	        if (Count++ > 10000) {
		    //usleep(10);
		    if (Count > 100000 * TimeOut_sec) {
		        break;
		    }
		}
	    }
	}
    }

    DEBUG_INT(cout << "done. SRR: " << hex << WordAt(regSRR) << dec << endl);

    return WordAt(regSRR) & 0x00ff;
}

int TVmeCamacBridge_Kinetic_2917::EnableInterrupt(int SignalId) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: EnableInterrupt()" << endl);

    int Irq = VmeAccessProperty()->InterruptNumber;
    int Vector = VmeAccessProperty()->InterruptVector;

    if (Irq <= 0) {
        return 0;
    }
    _IsInterruptInstalled = true;

    TRoomVmeModule::EnableInterrupt(SignalId);

    WordAt(regVECLam) = Vector & 0x00ff;
    WordAt(regINTLam) = bitINT_IntAutoClear | Irq;

    ClearInterrupt();

    DEBUG_INT(cout << "INT: " << hex << WordAt(regINTLam) << dec << endl);
    DEBUG_INT(cout << "VEC: " << hex << WordAt(regVECLam) << dec << endl);

    return 1;
}

int TVmeCamacBridge_Kinetic_2917::DisableInterrupt(void) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: DisableInterrupt()" << endl);

    if (_IsInterruptInstalled) {
        WordAt(regINTLam) = 0;
	TRoomVmeModule::DisableInterrupt();
	_IsInterruptInstalled = false;
    }

    return 1;
}

int TVmeCamacBridge_Kinetic_2917::ClearInterrupt(void) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: ClearInterrupt()" << endl);

    if (_IsInterruptInstalled) {
      WordAt(regINTLam) |= bitINT_Flag | bitINT_IntEnable | bitINT_IntAutoClear;
    }

    return 1;
}

void TVmeCamacBridge_Kinetic_2917::EnableServiceRequest(void) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: EnableServiceRequest()" << endl);
    EnableInterrupt();
}

void TVmeCamacBridge_Kinetic_2917::DisableServiceRequest(void) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: DisableServiceRequest()" << endl);
    DisableInterrupt();
}

void TVmeCamacBridge_Kinetic_2917::ClearServiceRequest(void) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: ClearServiceRequest()" << endl);
    ClearInterrupt();
}

bool TVmeCamacBridge_Kinetic_2917::IsRequestingService(void) throw(THardwareException)
{
    DEBUG_INT(cout << "K2917: IsRequestingService()" << endl);
    return WordAt(regSRR) & 0x00ff;
}

int TVmeCamacBridge_Kinetic_2917::MiscControlIdOf(const std::string& CommandName) throw (THardwareException)
{
    int ControlId = -1;
    if (CommandName == "setBufferSize") {
	ControlId = ControlId_SetBufferSize;
    }
    else if (CommandName == "setCommandMemoryAddress") {
	ControlId = ControlId_SetCommandMemoryAddress;
    }
    else if (CommandName == "loadCommand") {
	ControlId = ControlId_LoadCommand;
    }
    else if (CommandName == "executeSingleTransfer") {
	ControlId = ControlId_ExecuteSingleTransfer;
    }
    else if (CommandName == "loadCommand") {
	ControlId = ControlId_LoadCommand;
    }
    else if (CommandName == "loadSingleTransferCommand") {
	ControlId = ControlId_LoadSingleTransferCommand;
    }
    else if (CommandName == "loadInlineWriteCommand") {
	ControlId = ControlId_LoadInlineWriteCommand;
    }
    else if (CommandName == "loadQStopBlockTransferCommand") {
	ControlId = ControlId_LoadQStopBlockTransferCommand;
    }
    else if (CommandName == "loadQIgnoreBlockTransferCommand") {
	ControlId = ControlId_LoadQIgnoreBlockTransferCommand;
    }
    else if (CommandName == "loadQRepeatBlockTransferCommand") {
	ControlId = ControlId_LoadQRepeatBlockTransferCommand;
    }
    else if (CommandName == "loadQScanBlockTransferCommand") {
	ControlId = ControlId_LoadQScanBlockTransferCommand;
    }
    else if (CommandName == "loadJumpCommand") {
	ControlId = ControlId_LoadJumpCommand;
    }
    else if (CommandName == "loadHaltCommand") {
	ControlId = ControlId_LoadHaltCommand;
    }
    else {
	throw THardwareException(
	    "Kinetic2917",
	    "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TVmeCamacBridge_Kinetic_2917::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (ControlId == ControlId_SetBufferSize) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"Kinetic2917::setBufferSize(int buffer_size)",
		"too few argument[s]"
	    );
	}
	SetBufferSize(ArgumentList[0]);
    }
    else if (ControlId == ControlId_SetCommandMemoryAddress) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"Kinetic2917::setCommandMemoryAddress(int command_address)",
		"too few argument[s]"
	    );
	}
	SetCommandMemoryAddress(ArgumentList[0]);
    }
    else if (ControlId == ControlId_ExecuteSingleTransfer) {
	if (NumberOfArguments < 4) {
	    throw THardwareException(
		"Kinetic2917::executeSingleTransfer(int crate, int station, int function, int address, [int data, int q, int x])",
		"too few argument[s]"
	    );
	}
	int Crate = ArgumentList[0];
	int Station = ArgumentList[1];
	int Function = ArgumentList[2];
	int Address = ArgumentList[3];

	int Data = 0, Q , X;
	if (NumberOfArguments > 4) {
	    Data = ArgumentList[4];
	}

	ExecuteSingleTransfer(Crate, Station, Function, Address, Data, Q, X);

	if (NumberOfArguments > 4) {
	    ArgumentList[4] = Data;
	}
	if (NumberOfArguments > 5) {
	    ArgumentList[5] = Q;
	}
	if (NumberOfArguments > 6) {
	    ArgumentList[6] = X;
	}
    }
    else if (ControlId == ControlId_LoadCommand) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"Kinetic2917::loadCommand(int command)",
		"too few argument[s]"
	    );
	}
	LoadCommand(ArgumentList[0]);
    }
    else if (ControlId == ControlId_LoadSingleTransferCommand) {
	if (NumberOfArguments < 4) {
	    throw THardwareException(
		"Kinetic2917::loadSingleTransferCommand(int crate, int station, int function, int address)",
		"too few argument[s]"
	    );
	}
	int c = ArgumentList[0] & 0x07;
	int n = ArgumentList[1] & 0x1f;
	int f = ArgumentList[2] & 0x1f;
	int a = ArgumentList[3] & 0x0f;
	
	if (IsWrite(f)) {
	    throw THardwareException(
		"Kinetic2917::loadSingleTransferCommand()",
		"WRITE transfer is not available; use InlieWrite instead"
	    );
	}

	LoadCommand((c << 8) | bitCMR_AD);
	LoadCommand(NAF(n, a, f));
    }
    else if (ControlId == ControlId_LoadInlineWriteCommand) {
	if (NumberOfArguments < 5) {
	    throw THardwareException(
		"Kinetic2917::loadInlienWriteCommand(int crate, int station, int function, int address, int data)",
		"too few argument[s]"
	    );
	}
	int c = ArgumentList[0] & 0x07;
	int n = ArgumentList[1] & 0x1f;
	int f = ArgumentList[2] & 0x1f;
	int a = ArgumentList[3] & 0x0f;
	int data = ArgumentList[4];
	
	LoadCommand((c << 8) | bitCMR_InlineWrite | bitCMR_AD);
	LoadCommand(NAF(n, a, f));
	LoadCommand((data >> 0) & 0xffff);
	LoadCommand((data >> 16) & 0x00ff);
    }
    else if (ControlId == ControlId_LoadQStopBlockTransferCommand) {
	if (NumberOfArguments < 4) {
	    throw THardwareException(
		"Kinetic2917::loadQStopBlockTransferCommand(int crate, int station, int function, int address, int word_count = 0)",
		"too few argument[s]"
	    );
	}
	int c = ArgumentList[0] & 0x07;
	int n = ArgumentList[1] & 0x1f;
	int f = ArgumentList[2] & 0x1f;
	int a = ArgumentList[3] & 0x0f;
	int word_count = 0;
	if (NumberOfArguments > 4) {
	    word_count = ArgumentList[4] & 0xffff;
	}
	
	LoadCommand((c << 8) | bitCMR_Block | bitCMR_QStop | bitCMR_AD);
	LoadCommand(NAF(n, a, f));
	LoadCommand(word_count);
	LoadCommand(0xffff);
    }
    else if (ControlId == ControlId_LoadQIgnoreBlockTransferCommand) {
	if (NumberOfArguments < 4) {
	    throw THardwareException(
		"Kinetic2917::loadQIngoreBlockTransferCommand(int crate, int station, int function, int address, int word_count = 0)",
		"too few argument[s]"
	    );
	}
	int c = ArgumentList[0] & 0x07;
	int n = ArgumentList[1] & 0x1f;
	int f = ArgumentList[2] & 0x1f;
	int a = ArgumentList[3] & 0x0f;
	int word_count = 0;
	if (NumberOfArguments > 4) {
	    word_count = ArgumentList[4] & 0xffff;
	}
	
	LoadCommand((c << 8) | bitCMR_Block | bitCMR_QIgnore | bitCMR_AD);
	LoadCommand(NAF(n, a, f));
	LoadCommand(word_count);
	LoadCommand(0xffff);
    }
    else if (ControlId == ControlId_LoadQRepeatBlockTransferCommand) {
	if (NumberOfArguments < 4) {
	    throw THardwareException(
		"Kinetic2917::loadQRepeatBlockTransferCommand(int crate, int station, int function, int address, int word_count = 0)",
		"too few argument[s]"
	    );
	}
	int c = ArgumentList[0] & 0x07;
	int n = ArgumentList[1] & 0x1f;
	int f = ArgumentList[2] & 0x1f;
	int a = ArgumentList[3] & 0x0f;
	int word_count = 0;
	if (NumberOfArguments > 4) {
	    word_count = ArgumentList[4] & 0xffff;
	}
	
	LoadCommand((c << 8) | bitCMR_Block | bitCMR_QRepeat | bitCMR_AD);
	LoadCommand(NAF(n, a, f));
	LoadCommand(word_count);
	LoadCommand(0xffff);
    }
    else if (ControlId == ControlId_LoadQScanBlockTransferCommand) {
	if (NumberOfArguments < 4) {
	    throw THardwareException(
		"Kinetic2917::loadQScanBlockTransferCommand(int crate, int station, int function, int address, int word_count = 0)",
		"too few argument[s]"
	    );
	}
	int c = ArgumentList[0] & 0x07;
	int n = ArgumentList[1] & 0x1f;
	int f = ArgumentList[2] & 0x1f;
	int a = ArgumentList[3] & 0x0f;
	int word_count = 0;
	if (NumberOfArguments > 4) {
	    word_count = ArgumentList[4] & 0xffff;
	}
	
	LoadCommand((c << 8) | bitCMR_Block | bitCMR_QScan | bitCMR_AD);
	LoadCommand(NAF(n, a, f));
	LoadCommand(word_count);
	LoadCommand(0xffff);
    }
    else if (ControlId == ControlId_LoadJumpCommand) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"Kinetic2917::loadJumpCommand(int address)",
		"too few argument[s]"
	    );
	}
	LoadCommand(bitCMR_Jump);
	LoadCommand(ArgumentList[0]);
    }
    else if (ControlId == ControlId_LoadHaltCommand) {
	LoadCommand(bitCMR_Halt);
    }
    else {
	return 0;
    }

    return 1;
}


//// Common Functions ////

void TVmeCamacBridge_Kinetic_2917::SetCrateNumber(int CrateNumber)
{
    _CrateNumber = CrateNumber & 0x07;
}

void TVmeCamacBridge_Kinetic_2917::SetBufferSize(int BufferSize)
{
    _BufferSize = BufferSize;

    if (_Buffer != 0) {
	delete[] _Buffer;
	_Buffer = new U32bit[_BufferSize / sizeof(U32bit)];
    }
}

void TVmeCamacBridge_Kinetic_2917::SetCommandMemoryAddress(int Address)
{
    WordAt(regCMA) = Address & 0x7fff;
}

void TVmeCamacBridge_Kinetic_2917::LoadCommand(int Command)
{
    WordAt(regCMR) = Command & 0xffff;
}

int TVmeCamacBridge_Kinetic_2917::ExecuteSingleTransfer(int CrateNumber, int StationNumber, int Function, int Address, int& Data, int& Q, int&X) throw(THardwareException)
{
    DEBUG(cout << "K2917: ExecuteSingleTransfer()" << endl);

    int IsError = 0;

    WordAt(regCMA) = WorkingCommandMemoryAddress;

    WordAt(regCMR) = (CrateNumber << 8) | bitCMR_AD;
    WordAt(regCMR) = NAF(StationNumber, Address, Function);
    WordAt(regCMR) = bitCMR_Halt;

    WordAt(regCMA) = WorkingCommandMemoryAddress;

    if (IsRead(Function)) {
	WordAt(regCSR) = bitCSR_GO;
    }
    else if (IsWrite(Function)) {
	WordAt(regCSR) = bitCSR_DIR | bitCSR_GO;
    }
    else {
	WordAt(regCSR) = bitCSR_GO;
    }

    int CSR;
    int NumberOfTrys = 0;
    if (! IsControl(Function)) {
	while (! ((CSR = WordAt(regCSR)) & (bitCSR_RDY | bitCSR_ERR))) {
	    if (++NumberOfTrys > 100) {
	        usleep(10);

		if (NumberOfTrys > 1000) {
		    DEBUG(cout << "K2917: timeout on waiting for CSR_RDY: ");
		    DEBUG(cout << "CSR: " << hex << CSR << dec << endl);
		    IsError = 1;
		    break;
		}
	    }
	}
	if (CSR & bitCSR_ERR) {
	    DEBUG(cout << "K2917: error on waiting for CSR_RDY: ");
	    DEBUG(cout << "CSR: " << hex << CSR << dec << endl);
	    IsError = 1;
	}

	if (! IsError) {
	    if (IsRead(Function)) {
                Data = (WordAt(regDHR) & 0x00ff) << 16;
                Data |= WordAt(regDLR);
	    }
	    else if (IsWrite(Function)) {
		WordAt(regDHR) = (Data >> 16) & 0x00ff;
		WordAt(regDLR) = (Data >> 0) & 0xffff;
	    }
	}
    }

    NumberOfTrys = 0;
    while (! ((CSR = WordAt(regCSR)) & (bitCSR_DONE | bitCSR_ERR))) {
        if (++NumberOfTrys > 100) {
            usleep(10);

	    if (NumberOfTrys > 1000) {
		DEBUG(cout << "K2917: timeout on waiting for CSR_DONE");
		DEBUG(cout << "CSR: " << hex << CSR << dec << endl);
	        IsError = 1;
		break;
	    }
	}
    }
    if (CSR & bitCSR_ERR) {
	DEBUG(cout << "K2917: error on waiting for CSR_DONE: ");
        DEBUG(cout << "CSR: " << hex << CSR << dec << endl);
        IsError = 1; 
    }

    Q = (CSR & bitCSR_NOQ) ? 0 : 1;
    X = (CSR & bitCSR_NOX) ? 0 : 1;

    if (IsError) {
        DEBUG(cout << "\tN: " << StationNumber << ", F: " << Function << ", A: " << Address << ", Data: " << hex << Data << dec << endl);
	DEBUG(cout << "\tQ: " << Q << ", X: " << X << ", Data: " << hex << Data << dec << endl);
    }

    return 0;
}

int TVmeCamacBridge_Kinetic_2917::ExecuteReadList(U32bit* Data, int MaxSize) throw (THardwareException)
{
    DEBUG(cout << "K2917: ExecuteReadList()" << endl);
    DEBUG(cout << "\tCMA: " << hex << WordAt(regCMA) << dec << endl);

    WordAt(regCSR) = bitCSR_GO;

    int DataWordCounts = 0;
    int MaxNumberOfDataWords = MaxSize / sizeof(U32bit);

    int CSR;
    int NumberOfTrys = 0;
    while (((CSR = WordAt(regCSR)) & bitCSR_DONE) == 0) {
        // The last readout of Q-stop transfer results in No-Q,
        // which causes an error. Breaking this loop on this error 
	// spoils all remaining commands, so we ignore the error here.

	if (CSR & bitCSR_RDY) {
	    if (DataWordCounts++ > MaxNumberOfDataWords) {
		throw THardwareException(
		    "TVmeCamacBridge_Kinetic_2917::ExecuteReadList()",
		    "data buffer exhausted"
		);
	    }

            DEBUG(cout << "\tRDY: CSR: " << hex << WordAt(regCSR) << dec << endl);
	    Data[DataWordCounts] = (WordAt(regDHR) & 0x00ff) << 16;
	    Data[DataWordCounts] |= WordAt(regDLR);

	    NumberOfTrys = 0;
	}
	else {
	    if (++NumberOfTrys > 100) {
	        usleep(10);
		
		if (NumberOfTrys > 1000) {
		    DEBUG(cout << "K2917: timeout on waiting CSR_RDY: ");
		    DEBUG(cout << "CSR: " << hex << CSR << dec << endl);
		    break;
		}
	    }
	}
    }

    DEBUG(cout << "\tCSR: " << hex << WordAt(regCSR) << dec << endl);

    return DataWordCounts * sizeof(U32bit);
}
