/* module-Rinei_RPV160.cc */
/* Created by Enomoto Sanshiro on 1 October 2002. */
/* Last updated by Enomoto Sanshiro on 24 November 2002. */


#include "RoomDeviceFactory.hh"
#include "RoomVmeAccess.hh"
#include "module-Rinei_RPV160.hh"

using namespace std;


static TRoomVmeModuleCreator Creator("Rinei_RPV160", new TVmeFADC_Rinei_RPV160());


TVmeFADC_Rinei_RPV160::TVmeFADC_Rinei_RPV160(void)
: TRoomVmeModule("VmeFADC", "Rinei_RPV160", (TRoomVmeTransferMode) RPV160_TransferMode, (size_t) RPV160_MapSize, (off_t) RPV160_MapOffset)
{
    _EnabledChannelBits = 0;
    _NumberOfEnabledChannels = 0;

    for (int Channel = 0; Channel < RPV160_NumberOfChannels; Channel++) {
        _Buffer[Channel] = new U16bit[RPV160_NumberOfDataWords];
        _ReadoutCount[Channel] = 0;
    }
}

TVmeFADC_Rinei_RPV160::~TVmeFADC_Rinei_RPV160()
{
    for (int Channel = 0; Channel < RPV160_NumberOfChannels; Channel++) {
        delete[] _Buffer[Channel];
    }
}

TRoomVmeModule* TVmeFADC_Rinei_RPV160::Clone(void)
{
    return new TVmeFADC_Rinei_RPV160();
}

int TVmeFADC_Rinei_RPV160::NumberOfChannels(void) throw(THardwareException)
{
    return RPV160_NumberOfChannels;
}

int TVmeFADC_Rinei_RPV160::AddressBitLength(void)
{
    return RPV160_AddressBitLength;
}

int TVmeFADC_Rinei_RPV160::DataBitLength(void)
{
    return RPV160_DataBitLength;
}

int TVmeFADC_Rinei_RPV160::Initialize(int InitialState) throw(THardwareException)
{
    /* initialize channels */
    for (int Channel = 0; Channel < RPV160_NumberOfChannels; Channel++) {
        long Address = 0x21002 + 0x200 * Channel;
	static const U16bit Data = 0x0080;
	PioWrite(Address, &Data, sizeof(Data));
    }

    Clear();
    
    return 1;
}

int TVmeFADC_Rinei_RPV160::Finalize(int FinalState) throw(THardwareException)
{
    return 1;
}

int TVmeFADC_Rinei_RPV160::Enable(int Address) throw(THardwareException)
{
    if ((Address < 0) || (Address >= RPV160_NumberOfChannels)) {
        return 0;
    }

    if (! (_EnabledChannelBits & Bit(Address))) {
        _EnabledChannelBits |= Bit(Address);
	_NumberOfEnabledChannels++;
    }

    return 1;
}

int TVmeFADC_Rinei_RPV160::Disable(int Address) throw(THardwareException)
{
    if ((Address < 0) || (Address >= RPV160_NumberOfChannels)) {
        return 0;
    }

    if (_EnabledChannelBits & Bit(Address)) {
        _EnabledChannelBits &= ~Bit(Address);
	_NumberOfEnabledChannels--;
    }

    return 1;
}

int TVmeFADC_Rinei_RPV160::Clear(int Address) throw(THardwareException)
{
    for (int Channel = 0; Channel < RPV160_NumberOfChannels; Channel++) {
        _ReadoutCount[Channel] = 0;
    }

    WordAt(regControl) = bitClear;

    return 1;
}

bool TVmeFADC_Rinei_RPV160::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    return TRoomModule::WaitData(TimeOut_sec);
}

bool TVmeFADC_Rinei_RPV160::HasData(int Address) throw(THardwareException)
{
    return WordAt(regStatus) & bitDone;
}

int TVmeFADC_Rinei_RPV160::Read(int Address, int &Data) throw(THardwareException)
{
    if (_ReadoutCount[Address] >= RPV160_NumberOfSamples) {
	return 0;
    }

    if (_ReadoutCount[Address] == 0) {
        int DataOffset = Address * RPV160_DataBlockSize;
	PioRead(DataOffset, _Buffer[Address], RPV160_DataBlockSize);
	_StartWordIndex = WordAt(regAddressCounter);
    }

    int SampleIndex = _ReadoutCount[Address];
    _ReadoutCount[Address]++;

    int WordIndex = (_StartWordIndex + SampleIndex / 2) % RPV160_NumberOfDataWords;
    if (SampleIndex & 0x0001) {
        Data = (_Buffer[Address][WordIndex] >> 0) & 0x00ff;
    }
    else {
        Data = (_Buffer[Address][WordIndex] >> 8) & 0x00ff;
    }

    return 1;
}

int TVmeFADC_Rinei_RPV160::NextNumberOfDataElements(int Address) throw(THardwareException)
{
    return RPV160_NumberOfSamples;
}

int TVmeFADC_Rinei_RPV160::SequentialRead(int Address, int Data[], int MaxSize) throw(THardwareException)
{
    int DataOffset = Address * RPV160_DataBlockSize;
    PioRead(DataOffset, _Buffer[Address], RPV160_DataBlockSize);
    
    _StartWordIndex = WordAt(regAddressCounter);

    for (int Index = 0; Index < RPV160_NumberOfDataWords; Index++) {
        int Word = (_StartWordIndex + Index) % RPV160_NumberOfDataWords;
        Data[2 * Word] = (_Buffer[Address][Word] >> 8) & 0x00ff;
        Data[2 * Word + 1] = (_Buffer[Address][Word] >> 0) & 0x00ff;
    }

    return RPV160_NumberOfSamples;
}

int TVmeFADC_Rinei_RPV160::NextDataBlockSize(int Address) throw(THardwareException)
{
    return _NumberOfEnabledChannels * RPV160_DataBlockSize;
}

int TVmeFADC_Rinei_RPV160::BlockRead(int Address, void* Data, int MaxSize) throw(THardwareException)
{
    size_t ReadSize = 0;

    _StartWordIndex = WordAt(regAddressCounter);
    size_t SecondFlagmentSize = RPV160_DataWordSize * _StartWordIndex;
    size_t FirstFlagmentSize = RPV160_DataBlockSize - SecondFlagmentSize;

    for (int Channel = 0; Channel < RPV160_NumberOfChannels; Channel++) {
        if (Bit(Channel) & _EnabledChannelBits == 0) {
	    continue;
	}

	if (FirstFlagmentSize > 0) {
	    ReadSize += PioRead(
	        RPV160_DataBlockSize * Channel + SecondFlagmentSize, 
                (U8bit*) Data + ReadSize, FirstFlagmentSize
	    );
	}
        if (SecondFlagmentSize > 0) {
            ReadSize += PioRead(
                RPV160_DataBlockSize * Channel,
                (U8bit*) Data + ReadSize, SecondFlagmentSize
	    );
	}
    }

    return ReadSize;
}

int TVmeFADC_Rinei_RPV160::MiscControlIdOf(const string& Command) throw(THardwareException)
{
    int ControlId = -1;
    if (Command == "start") {
	ControlId = ControlId_Start;
    }
    else if (Command == "stop") {
	ControlId = ControlId_Stop;
    }
    else {
	throw THardwareException(
	    "TVmeFADC_Rinei_RPV160::MiscControlIdOf()",
	    "unknown command: " + Command
	);
    }

    return ControlId;
}

int TVmeFADC_Rinei_RPV160::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw(THardwareException)
{
    if (ControlId == ControlId_Start) {
        WordAt(regControl) = bitStart;
    }
    else if (ControlId == ControlId_Stop) {
        WordAt(regControl) = bitStop;
    }
    else {
	return 0;
    }

    return 1;
}

bool TVmeFADC_Rinei_RPV160::IsSignalOnServiceRequestAvailable(void)
{
    return false;
}

void TVmeFADC_Rinei_RPV160::ClearServiceRequest(void) throw(THardwareException)
{
    WordAt(regControl) = bitClear;
    WordAt(regControl) = bitStart;

    return;
}



#if 0

// sample KTS script for RPV160 //

datasource RPV160_test_1
{
    int base_address = 0x01000000;
    int readout_channels = #0..#3;
    
    VmeCrate crate;
    VmeController controller("SBS-620");
    VmeModule fadc("Rinei-RPV160");

    crate.installController(controller);
    crate.installModule(fadc, base_address);
    
    on trigger(fadc) {
        fadc.sequentialRead(readout_channels);
	fadc.clear();
    }
}


// another sample KTS script for RPV160 //

datasource RPV160_test_2
{
    int base_address = 0x01000000;
    int readout_channels = #0..#3;
    
    VmeCrate crate;
    VmeController controller("SBS-620");
    VmeModule fadc("Rinei-RPV160");

    crate.installController(controller);
    crate.installModule(fadc, base_address);
    
    on run_begin {
	// RPV160's blockRead() reads only enabled channels.
	fadc.enable(readout_channels);
    }

    on trigger(fadc) {
        fadc.blockRead();
	fadc.clear();
    }
}

#endif
