/* module-SoftwareModules.cc */
/* Created by Enomoto Sanshiro on 23 April 1998. */
/* Last updated by Enomoto Sanshiro on 11 January 2002. */


#include <iostream>
#include <cmath>
#include <cstdlib>
#include <climits>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include "RoomDeviceFactory.hh"
#include "RoomSoftwareDevice.hh"
#include "module-SoftwareModules.hh"

using namespace std;


static TRoomSoftwareModuleCreator Creator1(
    "IntervalTimer", new TSoftwareModule_IntervalTimer(1, 0)
);
static TRoomSoftwareModuleCreator Creator2(
    "OneShotTimer", new TSoftwareModule_OneShotTimer(1, 0)
);
static TRoomSoftwareModuleCreator Creator3(
    "AlarmClock", new TSoftwareModule_AlarmClock(1)
);
static TRoomSoftwareModuleCreator Creator4(
    "SoftwareCounter", new TSoftwareModule_Counter()
);

//... for backward compatibiliy ...//
static TRoomSoftwareModuleCreator CreatorX(
    "TimerInterrupter", new TSoftwareModule_IntervalTimer(1, 0)
);


TSoftwareModule_IntervalTimer::TSoftwareModule_IntervalTimer(long Interval_sec, long Interval_usec)
: TRoomSoftwareModule("SoftwareIntervalTimer", "Software_IntervalTimer")
{
    _Interval_sec = Interval_sec + Interval_usec / 1000000;
    _Interval_usec = Interval_usec % 1000000;
    _Interval = _Interval_sec * 1000000 + Interval_usec;

    _IsRunning = true;
}

TSoftwareModule_IntervalTimer::~TSoftwareModule_IntervalTimer()
{
}

TRoomSoftwareModule* TSoftwareModule_IntervalTimer::Clone(void)
{
    return new TSoftwareModule_IntervalTimer(_Interval_sec, _Interval_usec);
}

void TSoftwareModule_IntervalTimer::SetInterval(int Interval_sec, int Interval_usec) throw(THardwareException)
{
    _Interval_sec = Interval_sec + Interval_usec / 1000000;
    _Interval_usec = Interval_usec % 1000000;
    
    if (_Interval_sec * 2 + 1 > LONG_MAX / 1000000) {
	throw THardwareException(
	    "TSoftwareModule_IntervalTimer::SetInterval()",
	    "too large interval value"
	);
    }

    _Interval = _Interval_sec * 1000000 + _Interval_usec;
}

long TSoftwareModule_IntervalTimer::PassedTime(void)
{
    static struct timeval CurrentTimeValue;
    gettimeofday(&CurrentTimeValue, 0);

    long PassedTime_sec = CurrentTimeValue.tv_sec - _StartTime_sec;
    long PassedTime_usec = CurrentTimeValue.tv_usec - _StartTime_usec;

    return PassedTime_sec * 1000000 + PassedTime_usec;
}

long TSoftwareModule_IntervalTimer::NumberOfTicks(void)
{
    if (_Interval == 0) {
	return 0;
    }

    return PassedTime() / _Interval;
}

void TSoftwareModule_IntervalTimer::Reset(void)
{
    struct timeval TimeValue;
    gettimeofday(&TimeValue, 0);

    _StartTime_sec = TimeValue.tv_sec;
    _StartTime_usec = TimeValue.tv_usec;
    _LastHandledTick = 0;
    _IsRunning = true;
}

int TSoftwareModule_IntervalTimer::Initialize(int InitialState) throw(THardwareException)
{
    TRoomSoftwareModule::Initialize(InitialState);
    Reset();

    return 1;
}

int TSoftwareModule_IntervalTimer::Clear(int Address) throw(THardwareException)
{
    _LastHandledTick = NumberOfTicks();
    if (_LastHandledTick * _Interval > LONG_MAX / 2) {
	Reset();
    }

    return 1;
}

bool TSoftwareModule_IntervalTimer::HasData(int Address) throw(THardwareException)
{
    if (! _IsRunning) {
	return false;
    }

    if (_Interval < _MinimumSleepInterval_usec) {
	return true;
    }

    return (NumberOfTicks() > _LastHandledTick);
}

bool TSoftwareModule_IntervalTimer::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    if (! _IsRunning) {
	Suspend(TimeOut_sec, 0);
	return false;
    }

    if (_Interval < _MinimumSleepInterval_usec) {
	return true;
    }

    // Note that _LastHandledTick can be greater than StartTick
    // because of inaccuracy of the interval timer.
    // (Suspend() may return a little bit earlier than it is set)

    long StartTime = PassedTime();
    long StartTick = StartTime / _Interval;
    long TimeToNextTick = (_LastHandledTick + 1) * _Interval - StartTime;

#if 0
    cout << "time: " << StartTime << endl;
    cout << "tick: " << StartTick << endl;
    cout << "last tick: " << _LastHandledTick << endl;
    cout << "time to next: " << TimeToNextTick << endl;
#endif

    if (TimeToNextTick > (long) TimeOut_sec * 1000000) {
	Suspend(TimeOut_sec, 0);
	return false;
    }
    else {
	if (TimeToNextTick > 0) {
	    Suspend(TimeToNextTick / 1000000, TimeToNextTick % 1000000);
	    _LastHandledTick += 1;
	}
	else {
	    _LastHandledTick = StartTick + 1;
	}
    }

    return true;
}

int TSoftwareModule_IntervalTimer::Read(int Address, int &Data) throw(THardwareException)
{
    Data = (int) time(NULL);
    return 1;
}

int TSoftwareModule_IntervalTimer::WriteRegister(int Address, int Data) throw(THardwareException)
{
    switch (Address) {
      case 0:
	SetInterval(Data, _Interval_usec);
        break;
      case 1:
	SetInterval(_Interval_sec, Data);
        break;
      default:
        return -1;
    }

    return 0;
}

int TSoftwareModule_IntervalTimer::ReadRegister(int Address, int& Data) throw(THardwareException)
{
    switch (Address) {
      case 0:
        Data = _Interval_sec;
        break;
      case 1:
        Data = _Interval_usec;
        break;
      default:
        return -1;
    }

    return 0;
}

int TSoftwareModule_IntervalTimer::MiscControlIdOf(const string& CommandName) throw (THardwareException)
{
    int ControlId = -1;
    if (CommandName == "setInterval") {
	return ControlId_SetInterval;
    }
    else if (CommandName == "start") {
	ControlId = ControlId_Start;
    }
    else if (CommandName == "stop") {
	ControlId = ControlId_Stop;
    }
    else {
	throw THardwareException(
	    _ModelName, "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TSoftwareModule_IntervalTimer::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw(THardwareException)
{
    if (ControlId == ControlId_SetInterval) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		_ModelName + "::setInterval()",
		"too few argument[s]"
	    );
	}

	int Interval_sec = ArgumentList[0];
	int Interval_usec = 0;
	if (NumberOfArguments > 1) {
	    Interval_usec = ArgumentList[1];
	}
	
	SetInterval(Interval_sec, Interval_usec);
    }
    else if (ControlId == ControlId_Start) {
	_IsRunning = true;
    }
    else if (ControlId == ControlId_Stop) {
	_IsRunning = false;
    }
    else {
	return 0;
    }

    return 1;
}

void TSoftwareModule_IntervalTimer::ClearServiceRequest(void) throw(THardwareException)
{
    Clear();
}



TSoftwareModule_OneShotTimer::TSoftwareModule_OneShotTimer(long Interval_sec, long Interval_usec)
: TRoomSoftwareModule("SoftwareOneShotTimer", "Software_OneShotTimer")
{
    _Interval_sec = Interval_sec + Interval_usec / 1000000;
    _Interval_usec = Interval_usec % 1000000;
    _Interval = _Interval_sec * 1000000 + Interval_usec;

    _IsRunning = false;
}

TSoftwareModule_OneShotTimer::~TSoftwareModule_OneShotTimer()
{
}

TRoomSoftwareModule* TSoftwareModule_OneShotTimer::Clone(void)
{
    return new TSoftwareModule_OneShotTimer(_Interval_sec, _Interval_usec);
}

void TSoftwareModule_OneShotTimer::SetInterval(int Interval_sec, int Interval_usec) throw(THardwareException)
{
    _Interval_sec = Interval_sec + Interval_usec / 1000000;
    _Interval_usec = Interval_usec % 1000000;
    
    if (_Interval_sec + 1 > LONG_MAX / 1000000) {
	throw THardwareException(
	    "TSoftwareModule_OneShotTimer::SetInterval()",
	    "too large interval value"
	);
    }

    _Interval = _Interval_sec * 1000000 + _Interval_usec;
}

long TSoftwareModule_OneShotTimer::PassedTime(void)
{
    static struct timeval CurrentTimeValue;
    gettimeofday(&CurrentTimeValue, 0);

    long PassedTime_sec = CurrentTimeValue.tv_sec - _StartTime_sec;
    long PassedTime_usec = CurrentTimeValue.tv_usec - _StartTime_usec;

    return PassedTime_sec * 1000000 + PassedTime_usec;
}

int TSoftwareModule_OneShotTimer::Clear(int Address) throw(THardwareException)
{
    _IsRunning = false;

    return 1;
}

bool TSoftwareModule_OneShotTimer::HasData(int Address) throw(THardwareException)
{
    return _IsRunning && (PassedTime() > _Interval);
}

bool TSoftwareModule_OneShotTimer::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    if (! _IsRunning) {
	Suspend(TimeOut_sec, 0);
	return false;
    }

    long RemainingTime = _Interval - PassedTime();

    if (RemainingTime <= 0) {
	_IsRunning = false;
	return true;
    }
    else if (RemainingTime <= (long) TimeOut_sec * 1000000) {
	Suspend(RemainingTime / 1000000, RemainingTime % 1000000);
	_IsRunning = false;
	return true;
    }
    else {
	Suspend(TimeOut_sec, 0);
	return false;
    }
}

int TSoftwareModule_OneShotTimer::Read(int Address, int &Data) throw(THardwareException)
{
    Data = (int) time(NULL);
    return 1;
}

int TSoftwareModule_OneShotTimer::MiscControlIdOf(const string& CommandName) throw(THardwareException)
{
    int ControlId = -1;
    if (CommandName == "setInterval") {
	ControlId = ControlId_SetInterval;
    }
    else if (CommandName == "start") {
	ControlId = ControlId_Start;
    }
    else if (CommandName == "stop") {
	ControlId = ControlId_Stop;
    }
    else {
	throw THardwareException(
	    _ModelName, "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TSoftwareModule_OneShotTimer::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw(THardwareException)
{
    if (ControlId == ControlId_SetInterval) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		_ModelName + "::setInterval()", "too few argument[s]"
	    );
	}

	int Interval_sec = ArgumentList[0];
	int Interval_usec = 0;
	if (NumberOfArguments > 1) {
	    Interval_usec = ArgumentList[1];
	}
	
	SetInterval(Interval_sec, Interval_usec);
    }
    else if (ControlId == ControlId_Start) {
	static struct timeval CurrentTimeValue;
	gettimeofday(&CurrentTimeValue, 0);

	_StartTime_sec = CurrentTimeValue.tv_sec;
	_StartTime_usec = CurrentTimeValue.tv_usec;
	
	_IsRunning = true;
    }
    else if (ControlId == ControlId_Stop) {
	_IsRunning = false;
    }
    else {
	return 0;
    }

    return 1;
}

void TSoftwareModule_OneShotTimer::ClearServiceRequest(void) throw(THardwareException)
{
    Clear();
}



TSoftwareModule_AlarmClock::TSoftwareModule_AlarmClock(long Interval_sec)
: TRoomSoftwareModule("SoftwareAlarmClock", "Software_AlarmClock")
{
    _Interval = Interval_sec;
    _IsRunning = false;
}

TSoftwareModule_AlarmClock::~TSoftwareModule_AlarmClock()
{
}

TRoomSoftwareModule* TSoftwareModule_AlarmClock::Clone(void)
{
    return new TSoftwareModule_AlarmClock(_Interval);
}

void TSoftwareModule_AlarmClock::SetInterval(int Interval_sec) throw(THardwareException)
{
    _Interval = Interval_sec;
}

long TSoftwareModule_AlarmClock::PassedTime(void)
{
    return (long) time(NULL) - _StartTime;
}

int TSoftwareModule_AlarmClock::Clear(int Address) throw(THardwareException)
{
    _IsRunning = false;

    return 1;
}

bool TSoftwareModule_AlarmClock::HasData(int Address) throw(THardwareException)
{
    return _IsRunning && (PassedTime() > _Interval);
}

bool TSoftwareModule_AlarmClock::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    if (! _IsRunning){
	Suspend(TimeOut_sec, 0);
	return false;
    }

    long RemainingTime = _Interval - PassedTime();

    if (RemainingTime <= 0) {
	_IsRunning = false;
	return true;
    }
    else if (RemainingTime <= (long) TimeOut_sec) {
	Suspend(RemainingTime, 0);
	_IsRunning = false;
	return true;
    }
    else {
	Suspend(TimeOut_sec, 0);
	return false;
    }
}

int TSoftwareModule_AlarmClock::Read(int Address, int &Data) throw(THardwareException)
{
    Data = (int) time(NULL);
    return 1;
}

int TSoftwareModule_AlarmClock::MiscControlIdOf(const string& CommandName) throw(THardwareException)
{
    int ControlId = -1;
    if (CommandName == "setInterval") {
	ControlId = ControlId_SetInterval;
    }
    else if (CommandName == "start") {
	ControlId = ControlId_Start;
    }
    else if (CommandName == "stop") {
	ControlId = ControlId_Stop;
    }
    else {
	throw THardwareException(
	    _ModelName, "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TSoftwareModule_AlarmClock::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw(THardwareException)
{
    if (ControlId == ControlId_SetInterval) {
	if ((NumberOfArguments <= 0) || (NumberOfArguments > 4)) {
	    throw THardwareException(
		_ModelName + "::setInterval()", "too few argument[s]"
	    );
	}

	const int CoeffList[] = {1, 60, 60, 24};
	long Interval = 0;
	for (int i = 0; i < NumberOfArguments; i++) {
	    Interval += CoeffList[NumberOfArguments - i - 1] * ArgumentList[i];
	}
	
	SetInterval(Interval);
    }
    else if (ControlId == ControlId_Start) {
	_StartTime = (long) time(NULL);
	_IsRunning = true;
    }
    else if (ControlId == ControlId_Stop) {
	_IsRunning = false;
    }
    else {
	return 0;
    }

    return 1;
}

void TSoftwareModule_AlarmClock::ClearServiceRequest(void) throw(THardwareException)
{
}



TSoftwareModule_Counter::TSoftwareModule_Counter(int NumberOfChannels, long FullScale)
: TRoomSoftwareModule("SoftwareCounter", "Software_Counter")
{
    _NumberOfChannels = NumberOfChannels;
    _FullScale = FullScale;

    _ReadoutCount = new int[NumberOfChannels];
    Initialize();
}

TSoftwareModule_Counter::~TSoftwareModule_Counter()
{
    delete[] _ReadoutCount;
}
    
TRoomSoftwareModule* TSoftwareModule_Counter::Clone(void)
{
    return new TSoftwareModule_Counter(_NumberOfChannels, _FullScale);
}

int TSoftwareModule_Counter::Initialize(int InitialState) throw(THardwareException)
{
    for (int Address = 0; Address < _NumberOfChannels; Address++) {
	_ReadoutCount[Address] = 0;
    }

    return 0;
}

int TSoftwareModule_Counter::Read(int Address, int &Data) throw(THardwareException)
{
    Data = _ReadoutCount[Address];
    
    if (Data >= _FullScale) {
	Data = _FullScale - 1;
    }
    else {
	_ReadoutCount[Address]++;
    }

    return 1;
}

int TSoftwareModule_Counter::Clear(int Address) throw(THardwareException)
{
    if (Address == -1) {
	for (int Address = 0; Address < _NumberOfChannels; Address++) {
	    _ReadoutCount[Address] = 0;
	}
    }
    else {
	_ReadoutCount[Address] = 0;
    }

    return 0;
}

int TSoftwareModule_Counter::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}
