/* KinokoAnalysisMessenger.cc */
/* Created by Enomoto Sanshiro on 31 December 2000. */
/* Last updated by Enomoto Sanshiro on 4 May 2002. */


#include <string>
#include "ParaPackage.hh"
#include "KinokoSectionDataAnalyzer.hh"
#include "KinokoDataChecker.hh"
#include "KinokoAnalysisMessenger.hh"

using namespace std;


TKinokoDataElementMessenger::TKinokoDataElementMessenger(void)
: TParaObjectPrototype("DataElement")
{
}

TKinokoDataElementMessenger::~TKinokoDataElementMessenger()
{
}

TParaObjectPrototype* TKinokoDataElementMessenger::Clone(void)
{
    return new TKinokoDataElementMessenger();
}

int TKinokoDataElementMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "setOffset") || (Message == "SetOffset")) {
	return SetOffset(ArgumentList, ReturnValue);
    }
    else if ((Message == "setFactor") || (Message == "SetFactor")) {
	return SetFactor(ArgumentList, ReturnValue);
    }

    return 0;
}

void TKinokoDataElementMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "DataElement::DataElement(): too few argument[s]"
	);
    }

    _SectionPath = ArgumentList[0]->AsString();

    if (ArgumentList.size() >= 2) {
	if (ArgumentList[1]->IsLong()) {
	    int Address = ArgumentList[1]->AsLong();
	    _ElementSpecifier = TKinokoDataElementSpecifier(Address);
	}
	else {
	    string FieldName = ArgumentList[1]->AsString();
	    _ElementSpecifier = TKinokoDataElementSpecifier(FieldName);
	}
    }
}

TParaValue& TKinokoDataElementMessenger::EvaluateOperator(TParaOperator* Operator, TParaValue& LeftValue, TParaValue& RightValue, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    string OperatorSymbol = Operator->Symbol();
    long Parameter = RightValue.AsLong();

    //... BUG: This ConditionMessenger will not be deleted
    TKinokoAnalysisConditionMessenger* ConditionMessenger;

    ConditionMessenger = new TKinokoDataConditionMessenger(
	OperatorSymbol, this, Parameter
    );

    return Result = TParaValue(ConditionMessenger);
}

int TKinokoDataElementMessenger::SetOffset(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsNumeric())) {
	throw TScriptException(
	    "DataElement::setOffset(double offset)", "invalid argument[s]"
	);
    }

    double Offset = ArgumentList[0]->AsDouble();
    _ElementSpecifier.SetOffset(Offset);

    ReturnValue = TParaValue(this);
    
    return 1;
}

int TKinokoDataElementMessenger::SetFactor(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsNumeric())) {
	throw TScriptException(
	    "DataElement::setFactor(double factor)", "invalid argument[s]"
	);
    }

    double Factor = ArgumentList[0]->AsDouble();
    _ElementSpecifier.SetFactor(Factor);

    ReturnValue = TParaValue(this);

    return 1;
}

const string& TKinokoDataElementMessenger::SectionPath(void)
{
    return _SectionPath;
}

const TKinokoDataElementSpecifier& TKinokoDataElementMessenger::DataElementSpecifier(void)
{
    return _ElementSpecifier;
}



TKinokoAnalysisConditionMessenger::TKinokoAnalysisConditionMessenger(void)
: TParaObjectPrototype("AnalysisCondition")
{
}

TKinokoAnalysisConditionMessenger::~TKinokoAnalysisConditionMessenger()
{
}

TParaObjectPrototype* TKinokoAnalysisConditionMessenger::Clone(void)
{
    return 0;
}

TParaValue& TKinokoAnalysisConditionMessenger::EvaluateOperator(TParaOperator* Operator, TParaValue& LeftValue, TParaValue& RightValue, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    //... BUG: This ConditionMessenger will not be deleted
    TKinokoAnalysisConditionMessenger* ConditionMessenger;

    if (Operator->Name() == "Not") {
	ConditionMessenger = new TKinokoComplexAnalysisConditionMessenger(
	    Operator->Symbol(), 0, this
	);
    }
    else if (RightValue.IsObject("AnalysisCondition")) {
	ConditionMessenger = new TKinokoComplexAnalysisConditionMessenger(
	    Operator->Symbol(), 
	    this, 
	    (TKinokoAnalysisConditionMessenger*) RightValue.AsObject()
	);
    }
    else {
	throw TScriptException("invalid operation: " + Operator->Symbol());
    }

    return Result = TParaValue(ConditionMessenger);
}



TKinokoDataConditionMessenger::TKinokoDataConditionMessenger(const std::string& OperatorSymbol, TKinokoDataElementMessenger* DataElementMessenger, long Parameter)
{
    _OperatorSymbol = OperatorSymbol;
    _DataElementMessenger = DataElementMessenger;
    _Parameter = Parameter;
}

TKinokoDataConditionMessenger::~TKinokoDataConditionMessenger()
{
}

TKinokoDataChecker* TKinokoDataConditionMessenger::GetDataChecker(void) throw(TScriptException)
{
    int OperatorId;
    if (_OperatorSymbol == "==") { 
	OperatorId = TKinokoElementDataChecker::Operator_Equal;
    }
    else if (_OperatorSymbol == "!=") { 
	OperatorId = TKinokoElementDataChecker::Operator_NotEqual;
    }
    else if (_OperatorSymbol == ">") { 
	OperatorId = TKinokoElementDataChecker::Operator_GreaterThan;
    }
    else if (_OperatorSymbol == ">=") { 
	OperatorId = TKinokoElementDataChecker::Operator_GreaterEqual;
    }
    else if (_OperatorSymbol == "<") { 
	OperatorId = TKinokoElementDataChecker::Operator_LessThan;
    }
    else if (_OperatorSymbol == "<=") { 
	OperatorId = TKinokoElementDataChecker::Operator_LessEqual;
    }
    else if (_OperatorSymbol == "&") { 
	OperatorId = TKinokoElementDataChecker::Operator_BitAnd;
    }
    else if (_OperatorSymbol == "^") { 
	OperatorId = TKinokoElementDataChecker::Operator_BitXor;
    }
    else {
	throw TScriptException("invalid register operator: " + _OperatorSymbol);
    }

    TKinokoDataChecker* DataChecker = new TKinokoElementDataChecker(
	_DataElementMessenger->SectionPath(), 
	_DataElementMessenger->DataElementSpecifier(),
	_Parameter, OperatorId
    );

    return DataChecker;
}



TKinokoComplexAnalysisConditionMessenger::TKinokoComplexAnalysisConditionMessenger(const std::string& OperatorSymbol, TKinokoAnalysisConditionMessenger* LeftCondition, TKinokoAnalysisConditionMessenger* RightCondition)
{
    _OperatorSymbol = OperatorSymbol;
    _LeftCondition = LeftCondition;
    _RightCondition = RightCondition;
}

TKinokoComplexAnalysisConditionMessenger::~TKinokoComplexAnalysisConditionMessenger()
{
    delete _RightCondition;
    delete _LeftCondition;
}

TKinokoDataChecker* TKinokoComplexAnalysisConditionMessenger::GetDataChecker(void) throw(TScriptException)
{
    int OperatorId;
    if (_OperatorSymbol == "!") { 
	OperatorId = TKinokoComplexDataChecker::Operator_Not;
    }
    else if (_OperatorSymbol == "&&") { 
	OperatorId = TKinokoComplexDataChecker::Operator_And;
    }
    else if (_OperatorSymbol == "||") { 
	OperatorId = TKinokoComplexDataChecker::Operator_Or;
    }
    else {
	throw TScriptException("invalid operator: " + _OperatorSymbol);
    }

    TKinokoDataChecker* LeftChecker = 0;
    TKinokoDataChecker* RightChecker = 0;

    if (_LeftCondition) {
	LeftChecker = _LeftCondition->GetDataChecker();
    }
    if (_RightCondition) {
	RightChecker = _RightCondition->GetDataChecker();
    }

    TKinokoDataChecker* DataChecker = new TKinokoComplexDataChecker(
	LeftChecker, RightChecker, OperatorId
    );

    return DataChecker;
}
