/* KinokoAnalysisSequence.cc */
/* Created by Enomoto Sanshiro on 17 September 2001 */
/* Last updated by Enomoto Sanshiro on 17 September 2001. */


#include <iostream>
#include <string>
#include "KinokoDataAnalyzer.hh"
#include "KinokoDataChecker.hh"
#include "KinokoAnalysisSequence.hh"

using namespace std;


TKinokoAnalysisSequence::TKinokoAnalysisSequence(void)
{
    _SequenceListCapacity = 16;
    _NumberOfSequences = 0;

    _SequenceList = new TKinokoAnalysisSequence* [_SequenceListCapacity];
}

TKinokoAnalysisSequence::~TKinokoAnalysisSequence()
{
    delete[] _SequenceList;
}

int TKinokoAnalysisSequence::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    return TKinokoDataAnalyzerList::ProcessTrailerPacket(Packet, DataSource);
}

void TKinokoAnalysisSequence::AddSequence(TKinokoAnalysisSequence* Sequence)
{
    if (_NumberOfSequences >= _SequenceListCapacity) {
	TKinokoAnalysisSequence** OldSequenceList = _SequenceList;
	_SequenceListCapacity *= 2;
	_SequenceList = new TKinokoAnalysisSequence* [_SequenceListCapacity];
	for (int i = 0; i < _NumberOfSequences; i++) {
	    _SequenceList[i] = OldSequenceList[i];
	}
	delete[] OldSequenceList;
    }

    TKinokoDataAnalyzerList::AddAnalyzer(Sequence);
    _SequenceList[_NumberOfSequences++] = Sequence;
}

void TKinokoAnalysisSequence::Flush(void)
{
    for (int i = 0; i < _NumberOfSequences; i++) {
	_SequenceList[i]->Flush();
    }
}

void TKinokoAnalysisSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence" << endl;
    TKinokoDataAnalyzerList::Dump(os, Indent + "->\t");
    os << Indent << ".end" << endl;
}



TKinokoSingleEventAnalysisSequence::TKinokoSingleEventAnalysisSequence(void)
{
    _ProcessorState = ProcessorState_WaitingNewEvent;
    _IsLastEventCompleted = true;
}

TKinokoSingleEventAnalysisSequence::~TKinokoSingleEventAnalysisSequence()
{
}

int TKinokoSingleEventAnalysisSequence::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    _IsLastEventCompleted = false;

    if (_ProcessorState == ProcessorState_ReadingData) {
	return TKinokoAnalysisSequence::ProcessDataPacket(
	    DataPacket, DataSource, DataSection
	);
    }
    else {
	return 0;
    }
}

int TKinokoSingleEventAnalysisSequence::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    _IsLastEventCompleted = true;

    if (_ProcessorState == ProcessorState_WaitingNewEvent) {
        _ProcessorState = ProcessorState_ReadingData;
    }
    else if (_ProcessorState == ProcessorState_ReadingData) {
        _ProcessorState = ProcessorState_HoldingData;
	return TKinokoAnalysisSequence::ProcessTrailerPacket(Packet, DataSource);
    }

    return 0;
}

void TKinokoSingleEventAnalysisSequence::Flush(void)
{
    TKinokoAnalysisSequence::Flush();
    TKinokoAnalysisSequence::Reset();

    if (_IsLastEventCompleted) {
        _ProcessorState = ProcessorState_ReadingData;
    }
    else {
        _ProcessorState = ProcessorState_WaitingNewEvent;
    }
}

void TKinokoSingleEventAnalysisSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence single_event" << endl;
    TKinokoDataAnalyzerList::Dump(os, Indent + "->\t");
    os << Indent << ".end" << endl;
}



TKinokoConditionalAnalysisSequence::TKinokoConditionalAnalysisSequence(TKinokoDataChecker* DataChecker)
{
    _DataChecker = DataChecker;
}

TKinokoConditionalAnalysisSequence::~TKinokoConditionalAnalysisSequence()
{
    delete _DataChecker;
}

void TKinokoConditionalAnalysisSequence::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    TKinokoAnalysisSequence::ReadDataSource(DataSource);
    _DataChecker->ReadDataSource(DataSource);
}

int TKinokoConditionalAnalysisSequence::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    if (
	(_DataChecker->ProcessDataPacket(DataPacket) > 0) &&
	_DataChecker->IsConditionSatisfied()
    ){
	return TKinokoAnalysisSequence::ProcessDataPacket(
	    DataPacket, DataSource, DataSection
	);
    }
    else {
	return 0;
    }
}

void TKinokoConditionalAnalysisSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence conditional (";
    _DataChecker->Dump(os);
    os << ")" << endl;

    TKinokoDataAnalyzerList::Dump(os, Indent + "->\t");
    os << Indent << ".end" << endl;
}
