/* KinokoDataDumper.cc */
/* Created by Enomoto Sanshiro on 6 May 2001. */
/* Last updated by Enomoto Sanshiro on 6 May 2001. */


#include <iostream>
#include <iomanip>
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoDataAnalyzer.hh"
#include "KinokoIndexedDataSection.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoTaggedDataSection.hh"
#include "KinokoNestedDataSection.hh"
#include "KinokoDataDumper.hh"

using namespace std;


TKinokoDataDumper::TKinokoDataDumper(ostream& OutputStream)
: _OutputStream(OutputStream)
{
    _NestCount = 0;
}

TKinokoDataDumper::~TKinokoDataDumper()
{
}

bool TKinokoDataDumper::IsSharable(void)
{
    return true;
}

int TKinokoDataDumper::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    if ((DataSource != 0)  && (_NestCount == 0)) {
	_OutputStream << "### [" << DataSource->DataSourceName() << "] ###" << endl;
    }

    ++_NestCount;
    int Result = TKinokoDataAnalyzer::ProcessDataPacket(DataPacket, DataSource, DataSection);
    --_NestCount;

    return Result;
}

int TKinokoDataDumper::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    TKinokoIndexedDataSectionScanner* Scanner = DataSection->Scanner();

    const string& SectionName = DataSection->FullSectionName();
    int NumberOfElements = Scanner->NumberOfElements(DataPacket);

    int Address, Data;
    for (int Index = 0; Index < NumberOfElements; Index++) {
	Scanner->ReadFrom(DataPacket, Index, Address, Data);
	_OutputStream << SectionName << " " << Address << " " << Data << endl;
    }

    return 1;
}

int TKinokoDataDumper::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* DataSection) throw(TKinokoException)
{
    TKinokoTaggedDataSectionScanner* Scanner = DataSection->Scanner();

    const string& SectionName = DataSection->FullSectionName();
    int NumberOfFields = DataSection->NumberOfFields();

    string FieldName;
    int Data;
    for (int Index = 0; Index < NumberOfFields; Index++) {
	FieldName = DataSection->FieldNameOf(Index);
	Scanner->ReadFrom(DataPacket, Index, Data);
	_OutputStream << SectionName << " " << FieldName << " ";
	_OutputStream << Data << endl;
    }

    return 1;
}

int TKinokoDataDumper::ProcessBlockData(void* DataPacket, TKinokoBlockDataSection* DataSection) throw(TKinokoException)
{
    TKinokoBlockDataSectionScanner* Scanner = DataSection->Scanner();

    const string& SectionName = DataSection->FullSectionName();
    int DataSize = Scanner->DataSizeOf(DataPacket);
    void* DataArea = Scanner->DataAreaOf(DataPacket);

    _OutputStream << hex << setfill('0');
    _OutputStream << "[" << SectionName << "]" << flush;
    for (int Offset = 0; Offset < DataSize; Offset++) {
	if (Offset % 16 == 0) {
	    _OutputStream << endl << "  ";
	    _OutputStream << setw(4) << (Offset >> 16) << " ";
	    _OutputStream << setw(4) << (Offset & 0xffff) << ":  ";
	}
	else if (Offset % 8 == 0) {
	    _OutputStream << "  ";
	}
	
	_OutputStream << setw(2) << (int) (((U8bit*) DataArea)[Offset]) << " ";
    }

    _OutputStream << dec << endl;

    return 1;
}

int TKinokoDataDumper::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    if (DataSource != 0) {
	_OutputStream << "### [" << DataSource->DataSourceName() << "] ###" << endl;
    }

    _OutputStream << endl;

    return 1;
}



TKinokoDataTabler::TKinokoDataTabler(ostream& OutputStream)
: _OutputStream(OutputStream)
{
    _EventCount = 0;
    _IsVeryFirst = true;
    _IsValidRecord = false;
}

TKinokoDataTabler::~TKinokoDataTabler()
{
}

int TKinokoDataTabler::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    TKinokoIndexedDataSectionScanner* Scanner = DataSection->Scanner();

    const string& SectionName = DataSection->FullSectionName();
    int NumberOfElements = Scanner->NumberOfElements(DataPacket);
    
    if (NumberOfElements == 0) {
	return 0;
    }
    _IsValidRecord = true;

    int Address, Data;
    for (int Index = 0; Index < NumberOfElements; Index++) {
	Scanner->ReadFrom(DataPacket, Index, Address, Data);
	_DataList.push_back(Data);

	if (_EventCount == 0) {
	    if (_IsVeryFirst) {
		_OutputStream << "# Fields: index ";
		_IsVeryFirst = false;
	    }
	    _OutputStream << SectionName << "." << Address << " ";
	}
    }

    return 1;
}

int TKinokoDataTabler::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* DataSection) throw(TKinokoException)
{
    TKinokoTaggedDataSectionScanner* Scanner = DataSection->Scanner();

    const string& SectionName = DataSection->FullSectionName();
    int NumberOfFields = DataSection->NumberOfFields();
    
    _IsValidRecord = true;

    string FieldName;
    int Data;
    for (int Index = 0; Index < NumberOfFields; Index++) {
	FieldName = DataSection->FieldNameOf(Index);
	Scanner->ReadFrom(DataPacket, Index, Data);
	_DataList.push_back(Data);

	if (_EventCount == 0) {
	    if (_IsVeryFirst) {
		_OutputStream << "# Fields: index ";
		_IsVeryFirst = false;
	    }
	    _OutputStream << SectionName << "." << FieldName << " ";
	}
    }

    return 1;
}

int TKinokoDataTabler::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    if (! _IsValidRecord) {
	// Note that "on run_begin" etc. may make an empty event packet.
	return 0;
    }

    if (_EventCount == 0) {
	_OutputStream << endl;
    }

    _OutputStream << _EventCount << " ";
    for (unsigned i = 0; i < _DataList.size(); i++) {
	_OutputStream << _DataList[i] << " ";
    }
    _OutputStream << endl;

    _DataList.erase(_DataList.begin(), _DataList.end());
    _EventCount++;
    _IsValidRecord = false;

    return 1;
}
