/* KinokoDataProcessor.cc */
/* Created by Enomoto Sanshiro on 20 April 2001. */
/* Last updated by Enomoto Sanshiro on 23 April 2001. */


#include <iostream>
#include <strstream>
#include <string>
#include "MushDataEncoder.hh"
#include "KinokoStream.hh"
#include "KinokoDataStreamFormatter.hh"
#include "KinokoDataDescriptor.hh"
#include "KinokoDataSource.hh"
#include "KinokoDataProcessor.hh"

using namespace std;



TKinokoDataSender::TKinokoDataSender(void)
{
    _OutputStream = 0;
    _OutputStreamFormatter = 0;
    _OutputDataDescriptor = 0;
    _OutputDataSource = 0;
}

TKinokoDataSender::~TKinokoDataSender()
{
    // _OutputDataSource will be deleted by _OutputDataDescriptor
    delete _OutputDataDescriptor;
    delete _OutputStreamFormatter;
}

void TKinokoDataSender::ConstructOutlet(TKinokoOutputStream* OutputStream, TKinokoDataSource* DataSource)
{
    _OutputStream = OutputStream;
    _OutputDataSource = DataSource;

    _OutputStreamFormatter = new TKinokoDataStreamFormatter(_OutputDataSource);
    _OutputDataDescriptor = new TKinokoDataDescriptor();
    _OutputDataDescriptor->AddDataSource(_OutputDataSource);
}

void TKinokoDataSender::SendDataDescriptorPacket(void) throw(TKinokoException)
{
    if (_OutputDataSource == 0) {
	throw TKinokoException(
	    "TKinokoDataSender::SendDataDescriptorPacket()",
	    "datasource object is not initialized"
	);
    }

    void* Packet;
    unsigned PacketSize = _OutputStreamFormatter->DescriptorPacketSize();
    do {
	_OutputStream->NextEntry(Packet, PacketSize);
    } while (Packet == 0);

    _OutputStreamFormatter->WriteDescriptor(Packet);
    _OutputStream->Flush(Packet, PacketSize);
}

void TKinokoDataSender::SendRunBeginPacket(void) throw(TKinokoException)
{
    void* Packet;
    unsigned PacketSize = _OutputStreamFormatter->CommandPacketSize();
    do {
	_OutputStream->NextEntry(Packet, PacketSize);
    } while (Packet == 0);

    static const int Command = TKinokoDataStreamFormatter::Command_RunBegin;
    _OutputStreamFormatter->WriteCommand(Packet, Command);
    _OutputStream->Flush(Packet, PacketSize);
}

void TKinokoDataSender::SendRunEndPacket(void) throw(TKinokoException)
{
    void* Packet;
    unsigned PacketSize = _OutputStreamFormatter->CommandPacketSize();
    do {
	_OutputStream->NextEntry(Packet, PacketSize);
    } while (Packet == 0);

    static const int Command = TKinokoDataStreamFormatter::Command_RunEnd;
    _OutputStreamFormatter->WriteCommand(Packet, Command);
    _OutputStream->Flush(Packet, PacketSize);
}

void TKinokoDataSender::SendEventTrailerPacket(void) throw(TKinokoException)
{
    void* Packet;
    unsigned PacketSize = _OutputStreamFormatter->CommandPacketSize();
    do {
	_OutputStream->NextEntry(Packet, PacketSize);
    } while (Packet == 0);

    static const int Trailer = TKinokoDataStreamFormatter::Trailer_Event;
    _OutputStreamFormatter->WriteTrailer(Packet, Trailer);
    _OutputStream->Flush(Packet, PacketSize);
}

void TKinokoDataSender::SendPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    int WrittenLength;
    do {
	WrittenLength = _OutputStream->Write(Packet, PacketSize);
    } while (WrittenLength == 0);

    if (WrittenLength != PacketSize) {
	throw TKinokoException(
	    "TKinokoDataSender::ProcessData()", "stream I/O error (internal)"
	);
    }
}



TKinokoDataReceiver::TKinokoDataReceiver(void)
{
    _InputStream = 0;
    _InputDataDescriptor = 0;
    _InputStreamScanner = 0;

    _DataStreamMonitor = new TKinokoDataStreamMonitor();

    _DefaultStreamCommandProcessor = new TKinokoStreamCommandProcessor();
    _StreamCommandProcessor = _DefaultStreamCommandProcessor;

    _ChainedOutputStream = 0;
}

TKinokoDataReceiver::~TKinokoDataReceiver()
{
    delete _DefaultStreamCommandProcessor;

    delete _DataStreamMonitor;

    delete _InputStreamScanner;
    delete _InputDataDescriptor;
}

void TKinokoDataReceiver::ConstructInlet(TKinokoInputStream* InputStream)
{
    _InputStream = InputStream;
    _InputDataDescriptor = new TKinokoDataDescriptor(); 
    _InputStreamScanner = new TKinokoDataStreamScanner();

    _DataStreamMonitor->Clear();
}

void TKinokoDataReceiver::SetChainedOutput(TKinokoOutputStream* OutputStream)
{
    _ChainedOutputStream = OutputStream;
}

void TKinokoDataReceiver::SetStreamCommandProcessor(TKinokoStreamCommandProcessor* Processor)
{
    _StreamCommandProcessor = Processor;
}

TKinokoDataDescriptor* TKinokoDataReceiver::InputDataDescriptor(void)
{
    return _InputDataDescriptor;
}

TKinokoDataStreamMonitor* TKinokoDataReceiver::DataStreamMonitor(void)
{
    return _DataStreamMonitor;
}

int TKinokoDataReceiver::ProcessInputStream(void) throw(TKinokoException)
{
    void* Packet;
    int PacketSize;
    if ((PacketSize = _InputStream->NextEntry(Packet)) == 0) {
	return 0;
    }

    if (_ChainedOutputStream != 0) {
	int WrittenLength;
	do {
	    WrittenLength = _ChainedOutputStream->Write(Packet, PacketSize);
	} while (WrittenLength == 0);

	if (WrittenLength != PacketSize) {
	    throw TKinokoException(
		"TKinokoDataReceiver::ProcessInputStream()", 
		"stream I/O error (internal)"
	    );
	}
    }

    _InputStreamScanner->CorrectByteOrder(Packet, PacketSize);

    ProcessPacket(Packet, PacketSize);    
    _DataStreamMonitor->ProcessPacket(Packet, PacketSize);

    _InputStream->Flush(Packet);

    return 1;
}

void TKinokoDataReceiver::ProcessPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    OnReceivePacket(Packet, PacketSize);

    if (ProcessDataPacket(Packet, PacketSize)) {
	;
    }
    else if (ProcessTrailerPacket(Packet, PacketSize)) {
	;
    }
    else if (ProcessCommandPacket(Packet, PacketSize)) {
	;
    }
    else if (ProcessDataDescriptorPacket(Packet, PacketSize)) {
	;
    }
}

bool TKinokoDataReceiver::ProcessDataDescriptorPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    if (! _InputStreamScanner->IsDataDescriptorPacket(Packet)) {
	return false;
    }

    char* DescriptorText = _InputStreamScanner->DataDescriptorOf(Packet);
    istrstream DescriptorTextStream(DescriptorText);
    try {
	_InputDataDescriptor->ReadFrom(DescriptorTextStream);
    }
    catch (TKinokoException &e) {
	throw TKinokoException(
	    "TKinokoDataReceiver::ProcessDataDescriptorPacket()",
	    "badly formatted data descriptor: " + e.Message()
	);
    }

    OnReceiveDataDescriptorPacket(Packet, PacketSize);

    return true;
}

bool TKinokoDataReceiver::ProcessCommandPacket(void* Packet, long PacketSize)
{
    bool IsCommandPacket = _InputStreamScanner->IsCommandPacket(Packet);
    if (IsCommandPacket) {
	int CommandValue = _InputStreamScanner->CommandValueOf(Packet);
	_StreamCommandProcessor->ProcessCommand(CommandValue);

	OnReceiveCommandPacket(Packet, PacketSize);

	if (CommandValue == TKinokoDataStreamScanner::Command_RunBegin) {
	    if (_StreamCommandProcessor->NumberOfRunningDataSources() == 1) {
		OnRunBegin();
	    }
	}
	if (CommandValue == TKinokoDataStreamScanner::Command_RunEnd) {
	    if (_StreamCommandProcessor->NumberOfRunningDataSources() == 0) {
		OnRunEnd();
	    }
	}
    }

    return IsCommandPacket;
}

bool TKinokoDataReceiver::ProcessDataPacket(void* Packet, long PacketSize)
{
    bool IsDataPacket = _InputStreamScanner->IsDataPacket(Packet);
    if (IsDataPacket) {
	OnReceiveDataPacket(Packet, PacketSize);
    }

    return IsDataPacket;
}

bool TKinokoDataReceiver::ProcessTrailerPacket(void* Packet, long PacketSize)
{
    bool IsTrailerPacket = _InputStreamScanner->IsTrailerPacket(Packet);
    if (IsTrailerPacket) {
	OnReceiveTrailerPacket(Packet, PacketSize);

	switch (_InputStreamScanner->TrailerValueOf(Packet)) {
          case TKinokoDataStreamScanner::Trailer_Event:
	    OnReceiveEventTrailerPacket(Packet, PacketSize);
	    break;
          case TKinokoDataStreamScanner::Trailer_Command:
	    ;
	    break;
          case TKinokoDataStreamScanner::Trailer_Trap:
	    ;
	    break;
          default:
	    ;
	}
    }

    return IsTrailerPacket;
}



TKinokoDataProducer::TKinokoDataProducer(void)
{
    _InputIoStream = 0;
    _OutputIoStream = 0;
    _Logger = 0;
}

TKinokoDataProducer::~TKinokoDataProducer()
{
}

void TKinokoDataProducer::AttachIO(istream& InputIoStream, ostream& OutputIoStream, TKinokoLogger* Logger)
{
    _InputIoStream = &InputIoStream;
    _OutputIoStream = &OutputIoStream;
    _Logger = Logger;
}

void TKinokoDataProducer::Construct(const string& Name, TKinokoOutputStream* OutputStream, int DataSourceId) throw(TKinokoException)
{
    TKinokoDataSource* DataSource = new TKinokoDataSource(Name, DataSourceId);
    BuildDataSource(DataSource);

    ConstructOutlet(OutputStream, DataSource);
    SendDataDescriptorPacket();

    OnConstruct();
}

void TKinokoDataProducer::Destruct(void) throw(TKinokoException)
{
    OnDestruct();
}

int TKinokoDataProducer::ProcessData(void) throw(TKinokoException)
{
    return 0;
}



TKinokoDataProcessor::TKinokoDataProcessor(void)
{
    _InputIoStream = 0;
    _OutputIoStream = 0;
    _Logger = 0;
}

TKinokoDataProcessor::~TKinokoDataProcessor()
{
}

void TKinokoDataProcessor::AttachIO(istream& InputIoStream, ostream& OutputIoStream, TKinokoLogger* Logger)
{
    _InputIoStream = &InputIoStream;
    _OutputIoStream = &OutputIoStream;
    _Logger = Logger;
}

void TKinokoDataProcessor::Construct(const string& Name, TKinokoInputStream* InputStream, TKinokoOutputStream* OutputStream, int DataSourceId) throw(TKinokoException)
{
    ConstructInlet(InputStream);

    TKinokoDataSource* DataSource = new TKinokoDataSource(Name, DataSourceId);
    BuildDataSource(DataSource);
    ConstructOutlet(OutputStream, DataSource);

    OnConstruct();

    SendDataDescriptorPacket();
}

void TKinokoDataProcessor::Destruct(void) throw(TKinokoException)
{
    OnDestruct();
}

int TKinokoDataProcessor::ProcessData(void) throw(TKinokoException)
{
    if (_InputStream->HasData()) {
        return ProcessInputStream();
    }
    else {
        return 0;
    }
}



TKinokoDataConsumer::TKinokoDataConsumer(void)
{
    _InputIoStream = 0;
    _OutputIoStream = 0;
    _Logger = 0;
}

TKinokoDataConsumer::~TKinokoDataConsumer()
{
}

void TKinokoDataConsumer::AttachIO(istream& InputIoStream, ostream& OutputIoStream, TKinokoLogger* Logger)
{
    _InputIoStream = &InputIoStream;
    _OutputIoStream = &OutputIoStream;
    _Logger = Logger;
}

void TKinokoDataConsumer::Construct(const string& Name, TKinokoInputStream* InputStream) throw(TKinokoException)
{
    ConstructInlet(InputStream);
    OnConstruct();
}

void TKinokoDataConsumer::Destruct(void) throw(TKinokoException)
{
    OnDestruct();
}

int TKinokoDataConsumer::ProcessData(void) throw(TKinokoException)
{
    if (_InputStream->HasData()) {
        return ProcessInputStream();
    }
    else {
        return 0;
    }
}
